<?php
declare(strict_types=1);

namespace PrinsFrank\Standards\Country;

use NumberFormatter;
use PrinsFrank\Enums\BackedEnum;
use PrinsFrank\Standards\Country\Groups\GroupInterface;
use PrinsFrank\Standards\Country\Subdivision\CountrySubdivision;
use PrinsFrank\Standards\CountryCallingCode\CountryCallingCode;
use PrinsFrank\Standards\Currency\CurrencyAlpha3;
use PrinsFrank\Standards\InternationalCallPrefix\InternationalCallPrefix;
use PrinsFrank\Standards\InvalidArgumentException;
use PrinsFrank\Standards\Language\LanguageAlpha2;
use PrinsFrank\Standards\Language\LanguageAlpha3Bibliographic;
use PrinsFrank\Standards\Language\LanguageAlpha3Extensive;
use PrinsFrank\Standards\Language\LanguageAlpha3Terminology;
use PrinsFrank\Standards\Name\NameOrder;
use PrinsFrank\Standards\NationalCallPrefix\NationalCallPrefix;
use PrinsFrank\Standards\TopLevelDomain\CountryCodeTLD;

/**
 * @standard ISO3166_1
 * @source https://www.iso.org/obp/ui/#search/code/
 *
 * @updated-by \PrinsFrank\Standards\Dev\DataSource\Mapping\CountryMapping
 */
enum CountryAlpha2: string {
    case Afghanistan = 'AF';
    case Aland_Islands = 'AX';
    case Albania = 'AL';
    case Algeria = 'DZ';
    case American_Samoa = 'AS';
    case Andorra = 'AD';
    case Angola = 'AO';
    case Anguilla = 'AI';
    case Antarctica = 'AQ';
    case Antigua_and_Barbuda = 'AG';
    case Argentina = 'AR';
    case Armenia = 'AM';
    case Aruba = 'AW';
    case Australia = 'AU';
    case Austria = 'AT';
    case Azerbaijan = 'AZ';
    case Bahamas = 'BS';
    case Bahrain = 'BH';
    case Bangladesh = 'BD';
    case Barbados = 'BB';
    case Belarus = 'BY';
    case Belgium = 'BE';
    case Belize = 'BZ';
    case Benin = 'BJ';
    case Bermuda = 'BM';
    case Bhutan = 'BT';
    case Bolivia = 'BO';
    case Bonaire_Sint_Eustatius_and_Saba = 'BQ';
    case Bosnia_and_Herzegovina = 'BA';
    case Botswana = 'BW';
    case Bouvet_Island = 'BV';
    case Brazil = 'BR';
    case British_Indian_Ocean_Territory = 'IO';
    case Brunei_Darussalam = 'BN';
    case Bulgaria = 'BG';
    case Burkina_Faso = 'BF';
    case Burundi = 'BI';
    case Cabo_Verde = 'CV';
    case Cambodia = 'KH';
    case Cameroon = 'CM';
    case Canada = 'CA';
    case Cayman_Islands = 'KY';
    case Central_African_Republic = 'CF';
    case Chad = 'TD';
    case Chile = 'CL';
    case China = 'CN';
    case Christmas_Island = 'CX';
    case Cocos_Islands = 'CC';
    case Colombia = 'CO';
    case Comoros = 'KM';
    case Congo = 'CG';
    case Congo_Democratic_Republic = 'CD';
    case Cook_Islands = 'CK';
    case Costa_Rica = 'CR';
    case Cote_d_Ivoire = 'CI';
    case Croatia = 'HR';
    case Cuba = 'CU';
    case Curacao = 'CW';
    case Cyprus = 'CY';
    case Czechia = 'CZ';
    case Denmark = 'DK';
    case Djibouti = 'DJ';
    case Dominica = 'DM';
    case Dominican_Republic = 'DO';
    case Ecuador = 'EC';
    case Egypt = 'EG';
    case El_Salvador = 'SV';
    case Equatorial_Guinea = 'GQ';
    case Eritrea = 'ER';
    case Estonia = 'EE';
    case Eswatini = 'SZ';
    case Ethiopia = 'ET';
    case Falkland_Islands = 'FK';
    case Faroe_Islands = 'FO';
    case Fiji = 'FJ';
    case Finland = 'FI';
    case France = 'FR';
    case French_Guiana = 'GF';
    case French_Polynesia = 'PF';
    case French_Southern_Territories = 'TF';
    case Gabon = 'GA';
    case Gambia = 'GM';
    case Georgia = 'GE';
    case Germany = 'DE';
    case Ghana = 'GH';
    case Gibraltar = 'GI';
    case Greece = 'GR';
    case Greenland = 'GL';
    case Grenada = 'GD';
    case Guadeloupe = 'GP';
    case Guam = 'GU';
    case Guatemala = 'GT';
    case Guernsey = 'GG';
    case Guinea = 'GN';
    case Guinea_Bissau = 'GW';
    case Guyana = 'GY';
    case Haiti = 'HT';
    case Heard_Island_and_McDonald_Islands = 'HM';
    case Holy_See = 'VA';
    case Honduras = 'HN';
    case Hong_Kong = 'HK';
    case Hungary = 'HU';
    case Iceland = 'IS';
    case India = 'IN';
    case Indonesia = 'ID';
    case Iran = 'IR';
    case Iraq = 'IQ';
    case Ireland = 'IE';
    case Isle_of_Man = 'IM';
    case Israel = 'IL';
    case Italy = 'IT';
    case Jamaica = 'JM';
    case Japan = 'JP';
    case Jersey = 'JE';
    case Jordan = 'JO';
    case Kazakhstan = 'KZ';
    case Kenya = 'KE';
    case Kiribati = 'KI';
    case Korea_Democratic_Peoples_Republic = 'KP';
    case Korea_Republic = 'KR';
    case Kuwait = 'KW';
    case Kyrgyzstan = 'KG';
    case Lao_Peoples_Democratic_Republic = 'LA';
    case Latvia = 'LV';
    case Lebanon = 'LB';
    case Lesotho = 'LS';
    case Liberia = 'LR';
    case Libya = 'LY';
    case Liechtenstein = 'LI';
    case Lithuania = 'LT';
    case Luxembourg = 'LU';
    case Macao = 'MO';
    case Madagascar = 'MG';
    case Malawi = 'MW';
    case Malaysia = 'MY';
    case Maldives = 'MV';
    case Mali = 'ML';
    case Malta = 'MT';
    case Marshall_Islands = 'MH';
    case Martinique = 'MQ';
    case Mauritania = 'MR';
    case Mauritius = 'MU';
    case Mayotte = 'YT';
    case Mexico = 'MX';
    case Micronesia = 'FM';
    case Moldova = 'MD';
    case Monaco = 'MC';
    case Mongolia = 'MN';
    case Montenegro = 'ME';
    case Montserrat = 'MS';
    case Morocco = 'MA';
    case Mozambique = 'MZ';
    case Myanmar = 'MM';
    case Namibia = 'NA';
    case Nauru = 'NR';
    case Nepal = 'NP';
    case Netherlands = 'NL';
    case New_Caledonia = 'NC';
    case New_Zealand = 'NZ';
    case Nicaragua = 'NI';
    case Niger = 'NE';
    case Nigeria = 'NG';
    case Niue = 'NU';
    case Norfolk_Island = 'NF';
    case North_Macedonia = 'MK';
    case Northern_Mariana_Islands = 'MP';
    case Norway = 'NO';
    case Oman = 'OM';
    case Pakistan = 'PK';
    case Palau = 'PW';
    case Palestine = 'PS';
    case Panama = 'PA';
    case Papua_New_Guinea = 'PG';
    case Paraguay = 'PY';
    case Peru = 'PE';
    case Philippines = 'PH';
    case Pitcairn = 'PN';
    case Poland = 'PL';
    case Portugal = 'PT';
    case Puerto_Rico = 'PR';
    case Qatar = 'QA';
    case Reunion = 'RE';
    case Romania = 'RO';
    case Russian_Federation = 'RU';
    case Rwanda = 'RW';
    case Saint_Barthelemy = 'BL';
    case Saint_Helena_Ascension_Tristan_da_Cunha = 'SH';
    case Saint_Kitts_and_Nevis = 'KN';
    case Saint_Lucia = 'LC';
    case Saint_Martin_French_part = 'MF';
    case Saint_Pierre_and_Miquelon = 'PM';
    case Saint_Vincent_and_the_Grenadines = 'VC';
    case Samoa = 'WS';
    case San_Marino = 'SM';
    case Sao_Tome_and_Principe = 'ST';
    case Saudi_Arabia = 'SA';
    case Senegal = 'SN';
    case Serbia = 'RS';
    case Seychelles = 'SC';
    case Sierra_Leone = 'SL';
    case Singapore = 'SG';
    case Sint_Maarten_Dutch_part = 'SX';
    case Slovakia = 'SK';
    case Slovenia = 'SI';
    case Solomon_Islands = 'SB';
    case Somalia = 'SO';
    case South_Africa = 'ZA';
    case South_Georgia_South_Sandwich_Islands = 'GS';
    case South_Sudan = 'SS';
    case Spain = 'ES';
    case Sri_Lanka = 'LK';
    case Sudan = 'SD';
    case Suriname = 'SR';
    case Svalbard_Jan_Mayen = 'SJ';
    case Sweden = 'SE';
    case Switzerland = 'CH';
    case Syrian_Arab_Republic = 'SY';
    case Taiwan_Province_of_China = 'TW';
    case Tajikistan = 'TJ';
    case Tanzania = 'TZ';
    case Thailand = 'TH';
    case Timor_Leste = 'TL';
    case Togo = 'TG';
    case Tokelau = 'TK';
    case Tonga = 'TO';
    case Trinidad_and_Tobago = 'TT';
    case Tunisia = 'TN';
    case Turkey = 'TR';
    case Turkmenistan = 'TM';
    case Turks_and_Caicos_Islands = 'TC';
    case Tuvalu = 'TV';
    case Uganda = 'UG';
    case Ukraine = 'UA';
    case United_Arab_Emirates = 'AE';
    case United_Kingdom = 'GB';
    case United_States_Outlying_Islands = 'UM';
    case United_States_of_America = 'US';
    case Uruguay = 'UY';
    case Uzbekistan = 'UZ';
    case Vanuatu = 'VU';
    case Venezuela = 'VE';
    case Viet_Nam = 'VN';
    case Virgin_Islands_British = 'VG';
    case Virgin_Islands_U_S = 'VI';
    case Wallis_and_Futuna = 'WF';
    case Western_Sahara = 'EH';
    case Yemen = 'YE';
    case Zambia = 'ZM';
    case Zimbabwe = 'ZW';

    public function toCountryAlpha3(): CountryAlpha3 {
        return BackedEnum::fromName(CountryAlpha3::class, $this->name);
    }

    public function toCountryNumeric(): CountryNumeric {
        return BackedEnum::fromName(CountryNumeric::class, $this->name);
    }

    /** @deprecated Will be removed in v4. Please use ::getNameInLanguage(LanguageAlpha2::English) instead */
    public function toCountryName(): CountryName {
        return BackedEnum::fromName(CountryName::class, $this->name);
    }

    public function lowerCaseValue(): string {
        return strtolower($this->value);
    }

    public function getNameInLanguage(LanguageAlpha2|LanguageAlpha3Terminology|LanguageAlpha3Bibliographic|LanguageAlpha3Extensive $language): ?string {
        if ($language instanceof LanguageAlpha3Bibliographic) {
            $language = $language->toLanguageAlpha3Terminology();
        }

        $countryInLanguage = locale_get_display_region('-' . $this->value, $language->value);
        if ($countryInLanguage === false) {
            // @codeCoverageIgnoreStart
            return null;
            // @codeCoverageIgnoreEnd
        }

        return $countryInLanguage;
    }

    public function formatNumber(float $amount, LanguageAlpha2|LanguageAlpha3Terminology|LanguageAlpha3Bibliographic|LanguageAlpha3Extensive $language): ?string {
        if ($language instanceof LanguageAlpha3Bibliographic) {
            $language = $language->toLanguageAlpha3Terminology();
        }

        $formattedNumber = (new NumberFormatter($language->value . '-' . $this->value, NumberFormatter::DECIMAL))
            ->format($amount);

        return $formattedNumber === false ? null : $formattedNumber;
    }

    /**
     * @param class-string<GroupInterface> $groupFQN
     * @throws InvalidArgumentException
     */
    public function isMemberOf(string $groupFQN): bool {
        if (is_a($groupFQN, GroupInterface::class, true) === false || $groupFQN === GroupInterface::class) {
            throw new InvalidArgumentException('Argument $groupFQN should be a FQN of a class that implements the groupInterface, "' . $groupFQN . '" given');
        }

        return in_array($this, $groupFQN::allAlpha2(), true);
    }

    /** @return list<CountryCallingCode> */
    public function getCountryCallingCodes(): array {
        return CountryCallingCode::forCountry($this);
    }

    public function getNationalCallPrefix(): NationalCallPrefix {
        return NationalCallPrefix::forCountry($this);
    }

    public function getMostCommonNameOrder(): NameOrder {
        return match ($this) {
            self::Cambodia,
            self::China,
            self::Hungary,
            self::Japan,
            self::Korea_Democratic_Peoples_Republic,
            self::Korea_Republic,
            self::Mongolia,
            self::Singapore,
            self::Taiwan_Province_of_China,
            self::Viet_Nam => NameOrder::Eastern,
            default => NameOrder::Western,
        };
    }

    /**
     * When displaying this on web pages, keep in mind the default windows fonts don't have a representation for these.
     *
     * @see https://prinsfrank.nl/2021/01/25/Non-existing-flag-emojis-on-windows to make these flag emojis visible for Windows users.
     */
    public function getFlagEmoji(): string {
        return implode(
            '',
            array_map(
                static function (string $char) {
                    return mb_chr(mb_ord($char) + (127462 - 65)); // The emoji for region letter 'A' in flags is at code point 127462, the capital letter 'A' itself is at 65
                },
                str_split($this->value)
            )
        );
    }

    /** @return list<LanguageAlpha2|LanguageAlpha3Terminology|LanguageAlpha3Extensive> */
    public function getOfficialAndDeFactoLanguages(): array {
        return match($this) {
            self::Afghanistan => [LanguageAlpha2::Persian, LanguageAlpha2::Pushto_Pashto],
            self::Aland_Islands => [LanguageAlpha2::Swedish],
            self::Albania => [LanguageAlpha2::Albanian],
            self::Algeria => [LanguageAlpha2::Arabic],
            self::American_Samoa => [LanguageAlpha2::English, LanguageAlpha2::Samoan],
            self::Andorra => [LanguageAlpha2::Catalan_Valencian],
            self::Angola => [LanguageAlpha2::Portuguese],
            self::Anguilla => [LanguageAlpha2::English],
            self::Antarctica => [LanguageAlpha2::English],
            self::Antigua_and_Barbuda => [LanguageAlpha2::English],
            self::Argentina => [LanguageAlpha2::Spanish_Castilian],
            self::Armenia => [LanguageAlpha2::Armenian],
            self::Aruba => [LanguageAlpha3Terminology::Papiamento, LanguageAlpha2::Dutch_Flemish],
            self::Australia => [LanguageAlpha2::English],
            self::Austria => [LanguageAlpha2::German],
            self::Azerbaijan => [LanguageAlpha2::Azerbaijani],
            self::Bahamas => [LanguageAlpha2::English],
            self::Bahrain => [LanguageAlpha2::Arabic],
            self::Bangladesh => [LanguageAlpha2::Bengali],
            self::Barbados => [LanguageAlpha2::English],
            self::Belarus => [LanguageAlpha2::Belarusian, LanguageAlpha2::Russian],
            self::Belgium => [LanguageAlpha2::Dutch_Flemish, LanguageAlpha2::French, LanguageAlpha2::German],
            self::Belize => [LanguageAlpha2::English],
            self::Benin => [LanguageAlpha2::French],
            self::Bermuda => [LanguageAlpha2::English],
            self::Bhutan => [LanguageAlpha2::Dzongkha],
            self::Bolivia => [LanguageAlpha2::Spanish_Castilian, LanguageAlpha2::Aymara, LanguageAlpha3Extensive::Araona, LanguageAlpha3Extensive::Baure, LanguageAlpha3Extensive::Chiquitano, LanguageAlpha3Extensive::Canichana, LanguageAlpha3Extensive::Cavinena, LanguageAlpha3Extensive::Cayubaba, LanguageAlpha3Extensive::Chacobo, LanguageAlpha3Extensive::Ese_Ejja, LanguageAlpha2::Guarani, LanguageAlpha3Extensive::Guarayu, LanguageAlpha3Extensive::Itonama, LanguageAlpha3Extensive::Leco, LanguageAlpha3Extensive::Machinere, LanguageAlpha3Extensive::Movima, LanguageAlpha3Extensive::Pacahuara, LanguageAlpha3Extensive::Puquina, LanguageAlpha2::Quechua, LanguageAlpha3Extensive::Siriono, LanguageAlpha3Extensive::Tacana, LanguageAlpha3Extensive::Tapiete, LanguageAlpha3Extensive::Toromono, LanguageAlpha3Extensive::Yuki, LanguageAlpha3Extensive::Yuracare],
            self::Bonaire_Sint_Eustatius_and_Saba => [LanguageAlpha2::Dutch_Flemish, LanguageAlpha3Terminology::Papiamento],
            self::Bosnia_and_Herzegovina => [LanguageAlpha2::Bosnian, LanguageAlpha2::Croatian, LanguageAlpha2::Serbian],
            self::Botswana => [LanguageAlpha2::English],
            self::Bouvet_Island => [LanguageAlpha2::Norwegian],
            self::Brazil => [LanguageAlpha2::Portuguese],
            self::British_Indian_Ocean_Territory => [LanguageAlpha2::English],
            self::Brunei_Darussalam => [LanguageAlpha2::Malay],
            self::Bulgaria => [LanguageAlpha2::Bulgarian],
            self::Burkina_Faso => [LanguageAlpha2::French],
            self::Burundi => [LanguageAlpha2::French, LanguageAlpha2::Rundi, LanguageAlpha2::English],
            self::Cabo_Verde => [LanguageAlpha2::Portuguese],
            self::Cambodia => [LanguageAlpha2::Central_Khmer],
            self::Cameroon => [LanguageAlpha2::English, LanguageAlpha2::French],
            self::Canada => [LanguageAlpha2::English, LanguageAlpha2::French],
            self::Cayman_Islands => [LanguageAlpha2::English],
            self::Central_African_Republic => [LanguageAlpha2::French, LanguageAlpha2::Sango],
            self::Chad => [LanguageAlpha2::Arabic, LanguageAlpha2::French],
            self::Chile => [LanguageAlpha2::Spanish_Castilian],
            self::China => [LanguageAlpha2::Chinese],
            self::Christmas_Island => [LanguageAlpha2::English, LanguageAlpha2::Chinese, LanguageAlpha2::Malay],
            self::Cocos_Islands => [LanguageAlpha2::English, LanguageAlpha2::Malay],
            self::Colombia => [LanguageAlpha2::Spanish_Castilian],
            self::Comoros => [LanguageAlpha2::Arabic, LanguageAlpha2::French],
            self::Congo => [LanguageAlpha2::French],
            self::Congo_Democratic_Republic => [LanguageAlpha2::French],
            self::Cook_Islands => [LanguageAlpha2::English, LanguageAlpha2::Maori],
            self::Costa_Rica => [LanguageAlpha2::Spanish_Castilian],
            self::Cote_d_Ivoire => [LanguageAlpha2::French],
            self::Croatia => [LanguageAlpha2::Croatian],
            self::Cuba => [LanguageAlpha2::Spanish_Castilian],
            self::Curacao => [LanguageAlpha3Terminology::Papiamento, LanguageAlpha2::Dutch_Flemish, LanguageAlpha2::English],
            self::Cyprus => [LanguageAlpha2::Greek_Modern_1453, LanguageAlpha2::Turkish],
            self::Czechia => [LanguageAlpha2::Czech, LanguageAlpha2::Slovak],
            self::Denmark => [LanguageAlpha2::Danish],
            self::Djibouti => [LanguageAlpha2::Arabic, LanguageAlpha2::French],
            self::Dominica => [LanguageAlpha2::English],
            self::Dominican_Republic => [LanguageAlpha2::Spanish_Castilian],
            self::Ecuador => [LanguageAlpha2::Spanish_Castilian, LanguageAlpha2::Quechua],
            self::Egypt => [LanguageAlpha2::Arabic],
            self::El_Salvador => [LanguageAlpha2::Spanish_Castilian],
            self::Equatorial_Guinea => [LanguageAlpha2::French, LanguageAlpha2::Portuguese, LanguageAlpha2::Spanish_Castilian],
            self::Eritrea => [LanguageAlpha2::Tigrinya],
            self::Estonia => [LanguageAlpha2::Estonian],
            self::Eswatini => [LanguageAlpha2::English, LanguageAlpha2::Swati],
            self::Ethiopia => [LanguageAlpha2::Afar, LanguageAlpha2::Amharic, LanguageAlpha2::Oromo, LanguageAlpha2::Somali, LanguageAlpha2::Tigrinya],
            self::Falkland_Islands => [LanguageAlpha2::English],
            self::Faroe_Islands => [LanguageAlpha2::Faroese],
            self::Fiji => [LanguageAlpha2::English, LanguageAlpha2::Fijian, LanguageAlpha3Extensive::Fiji_Hindi],
            self::Finland => [LanguageAlpha2::Finnish, LanguageAlpha2::Swedish],
            self::France => [LanguageAlpha2::French],
            self::French_Guiana => [LanguageAlpha2::French],
            self::French_Polynesia => [LanguageAlpha2::French],
            self::French_Southern_Territories => [LanguageAlpha2::French],
            self::Gabon => [LanguageAlpha2::French],
            self::Gambia => [LanguageAlpha2::English],
            self::Georgia => [LanguageAlpha2::Georgian],
            self::Germany => [LanguageAlpha2::German],
            self::Ghana => [LanguageAlpha2::English],
            self::Gibraltar => [LanguageAlpha2::English],
            self::Greece => [LanguageAlpha2::Greek_Modern_1453],
            self::Greenland => [LanguageAlpha2::Kalaallisut_Greenlandic, LanguageAlpha2::Danish],
            self::Grenada => [LanguageAlpha2::English],
            self::Guadeloupe => [LanguageAlpha2::French],
            self::Guam => [LanguageAlpha2::Chamorro, LanguageAlpha2::English],
            self::Guatemala => [LanguageAlpha2::Spanish_Castilian],
            self::Guernsey => [LanguageAlpha2::English],
            self::Guinea => [LanguageAlpha2::French],
            self::Guinea_Bissau => [LanguageAlpha2::Portuguese],
            self::Guyana => [LanguageAlpha2::English],
            self::Haiti => [LanguageAlpha2::French, LanguageAlpha2::Haitian_Haitian_Creole],
            self::Heard_Island_and_McDonald_Islands => [LanguageAlpha2::English],
            self::Holy_See => [LanguageAlpha2::Latin, LanguageAlpha2::Italian],
            self::Honduras => [LanguageAlpha2::Spanish_Castilian],
            self::Hong_Kong => [LanguageAlpha2::Chinese, LanguageAlpha2::English],
            self::Hungary => [LanguageAlpha2::Hungarian],
            self::Iceland => [LanguageAlpha2::Icelandic, LanguageAlpha3Extensive::Icelandic_Sign_Language],
            self::India => [LanguageAlpha2::Hindi, LanguageAlpha2::English],
            self::Indonesia => [LanguageAlpha2::Indonesian],
            self::Iran => [LanguageAlpha2::Persian],
            self::Iraq => [LanguageAlpha2::Arabic, LanguageAlpha2::Kurdish],
            self::Ireland => [LanguageAlpha2::Irish, LanguageAlpha2::English],
            self::Isle_of_Man => [LanguageAlpha2::Manx, LanguageAlpha2::English],
            self::Israel => [LanguageAlpha2::Hebrew],
            self::Italy => [LanguageAlpha2::Italian],
            self::Jamaica => [LanguageAlpha2::English],
            self::Japan => [LanguageAlpha2::Japanese],
            self::Jersey => [LanguageAlpha2::English, LanguageAlpha2::French],
            self::Jordan => [LanguageAlpha2::Arabic],
            self::Kazakhstan => [LanguageAlpha2::Kazakh, LanguageAlpha2::Russian],
            self::Kenya => [LanguageAlpha2::English, LanguageAlpha2::Swahili],
            self::Kiribati => [LanguageAlpha2::English, LanguageAlpha3Terminology::Gilbertese],
            self::Korea_Democratic_Peoples_Republic => [LanguageAlpha2::Korean],
            self::Korea_Republic => [LanguageAlpha2::Korean, LanguageAlpha3Extensive::Korean_Sign_Language],
            self::Kuwait => [LanguageAlpha2::Arabic],
            self::Kyrgyzstan => [LanguageAlpha2::Kirghiz_Kyrgyz, LanguageAlpha2::Russian],
            self::Lao_Peoples_Democratic_Republic => [LanguageAlpha2::Lao],
            self::Latvia => [LanguageAlpha2::Latvian],
            self::Lebanon => [LanguageAlpha2::Arabic],
            self::Lesotho => [LanguageAlpha2::Sotho_Southern, LanguageAlpha2::English],
            self::Liberia => [LanguageAlpha2::English],
            self::Libya => [LanguageAlpha2::Arabic],
            self::Liechtenstein => [LanguageAlpha2::German],
            self::Lithuania => [LanguageAlpha2::Lithuanian],
            self::Luxembourg => [LanguageAlpha2::French, LanguageAlpha2::German, LanguageAlpha2::Luxembourgish_Letzeburgesch],
            self::Macao => [LanguageAlpha2::Chinese, LanguageAlpha2::Portuguese],
            self::Madagascar => [LanguageAlpha2::French, LanguageAlpha2::Malagasy],
            self::Malawi => [LanguageAlpha2::English, LanguageAlpha2::Chichewa_Chewa_Nyanja],
            self::Malaysia => [LanguageAlpha2::Malay],
            self::Maldives => [LanguageAlpha2::Divehi_Dhivehi_Maldivian],
            self::Mali => [LanguageAlpha2::Bambara, LanguageAlpha2::Fulah, LanguageAlpha3Terminology::Songhai_languages, LanguageAlpha3Terminology::Soninke, LanguageAlpha3Terminology::Tamashek],
            self::Malta => [LanguageAlpha2::Maltese],
            self::Marshall_Islands => [LanguageAlpha2::English, LanguageAlpha2::Marshallese],
            self::Martinique => [LanguageAlpha2::French],
            self::Mauritania => [LanguageAlpha2::Arabic],
            self::Mauritius => [LanguageAlpha2::English],
            self::Mayotte => [LanguageAlpha2::French],
            self::Mexico => [LanguageAlpha2::Spanish_Castilian],
            self::Micronesia => [LanguageAlpha2::English],
            self::Moldova => [LanguageAlpha2::Romanian_Moldavian_Moldovan],
            self::Monaco => [LanguageAlpha2::French],
            self::Mongolia => [LanguageAlpha2::Mongolian],
            self::Montenegro => [LanguageAlpha3Terminology::Montenegrin],
            self::Montserrat => [LanguageAlpha2::English],
            self::Morocco => [LanguageAlpha2::Arabic, LanguageAlpha3Terminology::Berber_languages],
            self::Mozambique => [LanguageAlpha2::Portuguese],
            self::Myanmar => [LanguageAlpha2::Burmese],
            self::Namibia => [LanguageAlpha2::English],
            self::Nauru => [LanguageAlpha2::English, LanguageAlpha2::Nauru],
            self::Nepal => [LanguageAlpha2::Nepali],
            self::Netherlands => [LanguageAlpha2::Dutch_Flemish],
            self::New_Caledonia => [LanguageAlpha2::French],
            self::New_Zealand => [LanguageAlpha2::English, LanguageAlpha2::Maori, LanguageAlpha3Extensive::New_Zealand_Sign_Language],
            self::Nicaragua => [LanguageAlpha2::Spanish_Castilian],
            self::Niger => [LanguageAlpha2::French],
            self::Nigeria => [LanguageAlpha2::English],
            self::Niue => [LanguageAlpha2::English, LanguageAlpha3Terminology::Niuean],
            self::Norfolk_Island => [LanguageAlpha2::English, LanguageAlpha3Extensive::Pitcairn_Norfolk],
            self::North_Macedonia => [LanguageAlpha2::Macedonian, LanguageAlpha2::Albanian],
            self::Northern_Mariana_Islands => [LanguageAlpha2::English, LanguageAlpha2::Chamorro, LanguageAlpha3Extensive::Carolinian],
            self::Norway => [LanguageAlpha2::Norwegian, LanguageAlpha2::Northern_Sami],
            self::Oman => [LanguageAlpha2::Arabic],
            self::Pakistan => [LanguageAlpha2::Urdu, LanguageAlpha2::English],
            self::Palau => [LanguageAlpha2::English, LanguageAlpha3Terminology::Palauan],
            self::Palestine => [LanguageAlpha2::Arabic],
            self::Panama => [LanguageAlpha2::Spanish_Castilian],
            self::Papua_New_Guinea => [LanguageAlpha2::English, LanguageAlpha2::Hiri_Motu, LanguageAlpha3Extensive::Papua_New_Guinean_Sign_Language, LanguageAlpha3Terminology::Tok_Pisin],
            self::Paraguay => [LanguageAlpha2::Spanish_Castilian, LanguageAlpha2::Guarani],
            self::Peru => [LanguageAlpha2::Spanish_Castilian, LanguageAlpha2::Quechua, LanguageAlpha2::Aymara, LanguageAlpha3Extensive::Ashaninka],
            self::Philippines => [LanguageAlpha3Terminology::Filipino_Pilipino, LanguageAlpha2::English],
            self::Pitcairn => [LanguageAlpha3Extensive::Pitcairn_Norfolk, LanguageAlpha2::English],
            self::Poland => [LanguageAlpha2::Polish],
            self::Portugal => [LanguageAlpha2::Portuguese],
            self::Puerto_Rico => [LanguageAlpha2::Spanish_Castilian, LanguageAlpha2::English],
            self::Qatar => [LanguageAlpha2::Arabic],
            self::Reunion => [LanguageAlpha2::French],
            self::Romania => [LanguageAlpha2::Romanian_Moldavian_Moldovan],
            self::Russian_Federation => [LanguageAlpha2::Russian],
            self::Rwanda => [LanguageAlpha2::English, LanguageAlpha2::French, LanguageAlpha2::Kinyarwanda, LanguageAlpha2::Swahili],
            self::Saint_Barthelemy => [LanguageAlpha2::French],
            self::Saint_Helena_Ascension_Tristan_da_Cunha => [LanguageAlpha2::English],
            self::Saint_Kitts_and_Nevis => [LanguageAlpha2::English],
            self::Saint_Lucia => [LanguageAlpha2::English],
            self::Saint_Martin_French_part => [LanguageAlpha2::French],
            self::Saint_Pierre_and_Miquelon => [LanguageAlpha2::French],
            self::Saint_Vincent_and_the_Grenadines => [LanguageAlpha2::English],
            self::Samoa => [LanguageAlpha2::English, LanguageAlpha2::Samoan],
            self::San_Marino => [LanguageAlpha2::Italian],
            self::Sao_Tome_and_Principe => [LanguageAlpha2::Portuguese],
            self::Saudi_Arabia => [LanguageAlpha2::Arabic],
            self::Senegal => [LanguageAlpha2::French],
            self::Serbia => [LanguageAlpha2::Serbian],
            self::Seychelles => [LanguageAlpha2::English, LanguageAlpha2::French],
            self::Sierra_Leone => [LanguageAlpha2::English],
            self::Singapore => [LanguageAlpha2::English, LanguageAlpha2::Malay, LanguageAlpha3Extensive::Mandarin_Chinese, LanguageAlpha2::Tamil],
            self::Sint_Maarten_Dutch_part => [LanguageAlpha2::Dutch_Flemish, LanguageAlpha2::English],
            self::Slovakia => [LanguageAlpha2::Slovak],
            self::Slovenia => [LanguageAlpha2::Slovenian],
            self::Solomon_Islands => [LanguageAlpha2::English],
            self::Somalia => [LanguageAlpha2::Somali, LanguageAlpha2::Arabic],
            self::South_Africa => [LanguageAlpha2::Afrikaans, LanguageAlpha2::English, LanguageAlpha2::Ndebele_South_South_Ndebele, LanguageAlpha2::Sotho_Southern, LanguageAlpha2::Swati, LanguageAlpha2::Tsonga, LanguageAlpha2::Tswana, LanguageAlpha2::Venda, LanguageAlpha2::Xhosa, LanguageAlpha2::Zulu],
            self::South_Georgia_South_Sandwich_Islands => [LanguageAlpha2::Ossetian_Ossetic, LanguageAlpha2::Russian],
            self::South_Sudan => [LanguageAlpha2::English],
            self::Spain => [LanguageAlpha2::Spanish_Castilian],
            self::Sri_Lanka => [LanguageAlpha2::Sinhala_Sinhalese, LanguageAlpha2::Tamil],
            self::Sudan => [LanguageAlpha2::Arabic, LanguageAlpha2::English],
            self::Suriname => [LanguageAlpha2::Dutch_Flemish],
            self::Svalbard_Jan_Mayen => [LanguageAlpha2::Norwegian],
            self::Sweden => [LanguageAlpha2::Swedish],
            self::Switzerland => [LanguageAlpha2::French, LanguageAlpha2::German, LanguageAlpha2::Italian, LanguageAlpha2::Romansh],
            self::Syrian_Arab_Republic => [LanguageAlpha2::Arabic],
            self::Taiwan_Province_of_China => [LanguageAlpha3Extensive::Mandarin_Chinese],
            self::Tajikistan => [LanguageAlpha2::Tajik],
            self::Tanzania => [LanguageAlpha2::Swahili, LanguageAlpha2::English],
            self::Thailand => [LanguageAlpha2::Thai],
            self::Timor_Leste => [LanguageAlpha2::Portuguese, LanguageAlpha3Terminology::Tetum],
            self::Togo => [LanguageAlpha2::French],
            self::Tokelau => [LanguageAlpha3Terminology::Tokelau, LanguageAlpha2::Samoan, LanguageAlpha2::English],
            self::Tonga => [LanguageAlpha2::English, LanguageAlpha2::Tonga_Tonga_Islands],
            self::Trinidad_and_Tobago => [LanguageAlpha2::English],
            self::Tunisia => [LanguageAlpha2::Arabic],
            self::Turkey => [LanguageAlpha2::Turkish],
            self::Turkmenistan => [LanguageAlpha2::Turkmen],
            self::Turks_and_Caicos_Islands => [LanguageAlpha2::English],
            self::Tuvalu => [LanguageAlpha3Terminology::Tuvalu, LanguageAlpha2::English],
            self::Uganda => [LanguageAlpha2::English, LanguageAlpha2::Swahili],
            self::Ukraine => [LanguageAlpha2::Ukrainian],
            self::United_Arab_Emirates => [LanguageAlpha2::Arabic],
            self::United_Kingdom => [LanguageAlpha2::English],
            self::United_States_Outlying_Islands => [LanguageAlpha2::English],
            self::United_States_of_America => [LanguageAlpha2::English],
            self::Uruguay => [LanguageAlpha2::Spanish_Castilian],
            self::Uzbekistan => [LanguageAlpha2::Uzbek],
            self::Vanuatu => [LanguageAlpha2::English, LanguageAlpha2::French, LanguageAlpha2::Bislama],
            self::Venezuela => [LanguageAlpha2::Spanish_Castilian],
            self::Viet_Nam => [LanguageAlpha2::Vietnamese],
            self::Virgin_Islands_British => [LanguageAlpha2::English],
            self::Virgin_Islands_U_S => [LanguageAlpha2::English],
            self::Wallis_and_Futuna => [LanguageAlpha2::French],
            self::Western_Sahara => [LanguageAlpha2::Arabic],
            self::Yemen => [LanguageAlpha2::Arabic],
            self::Zambia => [LanguageAlpha2::English],
            self::Zimbabwe => [LanguageAlpha2::English],
        };
    }

    /** @return list<CurrencyAlpha3> */
    public function getCurrenciesAlpha3(): array {
        return match($this) {
            self::Afghanistan => [CurrencyAlpha3::Afghani],
            self::Aland_Islands => [CurrencyAlpha3::Euro],
            self::Albania => [CurrencyAlpha3::Lek],
            self::Algeria => [CurrencyAlpha3::Algerian_Dinar],
            self::American_Samoa => [CurrencyAlpha3::US_Dollar],
            self::Andorra => [CurrencyAlpha3::Euro],
            self::Angola => [CurrencyAlpha3::Kwanza],
            self::Anguilla => [CurrencyAlpha3::East_Caribbean_Dollar],
            self::Antigua_and_Barbuda => [CurrencyAlpha3::East_Caribbean_Dollar],
            self::Argentina => [CurrencyAlpha3::Argentine_Peso],
            self::Armenia => [CurrencyAlpha3::Armenian_Dram],
            self::Aruba => [CurrencyAlpha3::Aruban_Florin],
            self::Australia => [CurrencyAlpha3::Australian_Dollar],
            self::Austria => [CurrencyAlpha3::Euro],
            self::Azerbaijan => [CurrencyAlpha3::Azerbaijan_Manat],
            self::Bahamas => [CurrencyAlpha3::Bahamian_Dollar],
            self::Bahrain => [CurrencyAlpha3::Bahraini_Dinar],
            self::Bangladesh => [CurrencyAlpha3::Taka],
            self::Barbados => [CurrencyAlpha3::Barbados_Dollar],
            self::Belarus => [CurrencyAlpha3::Belarusian_Ruble],
            self::Belgium => [CurrencyAlpha3::Euro],
            self::Belize => [CurrencyAlpha3::Belize_Dollar],
            self::Benin => [CurrencyAlpha3::CFA_Franc_BCEAO],
            self::Bermuda => [CurrencyAlpha3::Bermudian_Dollar],
            self::Bhutan => [
                CurrencyAlpha3::Indian_Rupee,
                CurrencyAlpha3::Ngultrum
            ],
            self::Bolivia => [
                CurrencyAlpha3::Boliviano,
                CurrencyAlpha3::Mvdol
            ],
            self::Bonaire_Sint_Eustatius_and_Saba => [CurrencyAlpha3::US_Dollar],
            self::Bosnia_and_Herzegovina => [CurrencyAlpha3::Convertible_Mark],
            self::Botswana => [CurrencyAlpha3::Pula],
            self::Bouvet_Island => [CurrencyAlpha3::Norwegian_Krone],
            self::Brazil => [CurrencyAlpha3::Brazilian_Real],
            self::British_Indian_Ocean_Territory => [CurrencyAlpha3::US_Dollar],
            self::Brunei_Darussalam => [CurrencyAlpha3::Brunei_Dollar],
            self::Bulgaria => [CurrencyAlpha3::Bulgarian_Lev],
            self::Burkina_Faso => [CurrencyAlpha3::CFA_Franc_BCEAO],
            self::Burundi => [CurrencyAlpha3::Burundi_Franc],
            self::Cabo_Verde => [CurrencyAlpha3::Cabo_Verde_Escudo],
            self::Cambodia => [CurrencyAlpha3::Riel],
            self::Cameroon => [CurrencyAlpha3::CFA_Franc_BEAC],
            self::Canada => [CurrencyAlpha3::Canadian_Dollar],
            self::Cayman_Islands => [CurrencyAlpha3::Cayman_Islands_Dollar],
            self::Central_African_Republic => [CurrencyAlpha3::CFA_Franc_BEAC],
            self::Chad => [CurrencyAlpha3::CFA_Franc_BEAC],
            self::Chile => [
                CurrencyAlpha3::Chilean_Peso,
                CurrencyAlpha3::Unidad_de_Fomento
            ],
            self::China => [CurrencyAlpha3::Yuan_Renminbi],
            self::Christmas_Island => [CurrencyAlpha3::Australian_Dollar],
            self::Cocos_Islands => [CurrencyAlpha3::Australian_Dollar],
            self::Colombia => [
                CurrencyAlpha3::Colombian_Peso,
                CurrencyAlpha3::Unidad_de_Valor_Real
            ],
            self::Comoros => [CurrencyAlpha3::Comorian_Franc],
            self::Congo => [CurrencyAlpha3::CFA_Franc_BEAC],
            self::Congo_Democratic_Republic => [CurrencyAlpha3::Congolese_Franc],
            self::Cook_Islands => [CurrencyAlpha3::New_Zealand_Dollar],
            self::Costa_Rica => [CurrencyAlpha3::Costa_Rican_Colon],
            self::Cote_d_Ivoire => [CurrencyAlpha3::CFA_Franc_BCEAO],
            self::Croatia => [CurrencyAlpha3::Euro],
            self::Cuba => [CurrencyAlpha3::Cuban_Peso],
            self::Curacao => [CurrencyAlpha3::Caribbean_Guilder],
            self::Cyprus => [CurrencyAlpha3::Euro],
            self::Czechia => [CurrencyAlpha3::Czech_Koruna],
            self::Denmark => [CurrencyAlpha3::Danish_Krone],
            self::Djibouti => [CurrencyAlpha3::Djibouti_Franc],
            self::Dominica => [CurrencyAlpha3::East_Caribbean_Dollar],
            self::Dominican_Republic => [CurrencyAlpha3::Dominican_Peso],
            self::Ecuador => [CurrencyAlpha3::US_Dollar],
            self::Egypt => [CurrencyAlpha3::Egyptian_Pound],
            self::El_Salvador => [
                CurrencyAlpha3::El_Salvador_Colon,
                CurrencyAlpha3::US_Dollar
            ],
            self::Equatorial_Guinea => [CurrencyAlpha3::CFA_Franc_BEAC],
            self::Eritrea => [CurrencyAlpha3::Nakfa],
            self::Estonia => [CurrencyAlpha3::Euro],
            self::Eswatini => [CurrencyAlpha3::Lilangeni],
            self::Ethiopia => [CurrencyAlpha3::Ethiopian_Birr],
            self::Falkland_Islands => [CurrencyAlpha3::Falkland_Islands_Pound],
            self::Faroe_Islands => [CurrencyAlpha3::Danish_Krone],
            self::Fiji => [CurrencyAlpha3::Fiji_Dollar],
            self::Finland => [CurrencyAlpha3::Euro],
            self::France => [CurrencyAlpha3::Euro],
            self::French_Guiana => [CurrencyAlpha3::Euro],
            self::French_Polynesia => [CurrencyAlpha3::CFP_Franc],
            self::French_Southern_Territories => [CurrencyAlpha3::Euro],
            self::Gabon => [CurrencyAlpha3::CFA_Franc_BEAC],
            self::Gambia => [CurrencyAlpha3::Dalasi],
            self::Georgia => [CurrencyAlpha3::Lari],
            self::Germany => [CurrencyAlpha3::Euro],
            self::Ghana => [CurrencyAlpha3::Ghana_Cedi],
            self::Gibraltar => [CurrencyAlpha3::Gibraltar_Pound],
            self::Greece => [CurrencyAlpha3::Euro],
            self::Greenland => [CurrencyAlpha3::Danish_Krone],
            self::Grenada => [CurrencyAlpha3::East_Caribbean_Dollar],
            self::Guadeloupe => [CurrencyAlpha3::Euro],
            self::Guam => [CurrencyAlpha3::US_Dollar],
            self::Guatemala => [CurrencyAlpha3::Quetzal],
            self::Guernsey => [CurrencyAlpha3::Pound_Sterling],
            self::Guinea => [CurrencyAlpha3::Guinean_Franc],
            self::Guinea_Bissau => [CurrencyAlpha3::CFA_Franc_BCEAO],
            self::Guyana => [CurrencyAlpha3::Guyana_Dollar],
            self::Haiti => [
                CurrencyAlpha3::Gourde,
                CurrencyAlpha3::US_Dollar
            ],
            self::Heard_Island_and_McDonald_Islands => [CurrencyAlpha3::Australian_Dollar],
            self::Holy_See => [CurrencyAlpha3::Euro],
            self::Honduras => [CurrencyAlpha3::Lempira],
            self::Hong_Kong => [CurrencyAlpha3::Hong_Kong_Dollar],
            self::Hungary => [CurrencyAlpha3::Forint],
            self::Iceland => [CurrencyAlpha3::Iceland_Krona],
            self::India => [CurrencyAlpha3::Indian_Rupee],
            self::Indonesia => [CurrencyAlpha3::Rupiah],
            self::Iran => [CurrencyAlpha3::Iranian_Rial],
            self::Iraq => [CurrencyAlpha3::Iraqi_Dinar],
            self::Ireland => [CurrencyAlpha3::Euro],
            self::Isle_of_Man => [CurrencyAlpha3::Pound_Sterling],
            self::Israel => [CurrencyAlpha3::New_Israeli_Sheqel],
            self::Italy => [CurrencyAlpha3::Euro],
            self::Jamaica => [CurrencyAlpha3::Jamaican_Dollar],
            self::Japan => [CurrencyAlpha3::Yen],
            self::Jersey => [CurrencyAlpha3::Pound_Sterling],
            self::Jordan => [CurrencyAlpha3::Jordanian_Dinar],
            self::Kazakhstan => [CurrencyAlpha3::Tenge],
            self::Kenya => [CurrencyAlpha3::Kenyan_Shilling],
            self::Kiribati => [CurrencyAlpha3::Australian_Dollar],
            self::Korea_Democratic_Peoples_Republic => [CurrencyAlpha3::North_Korean_Won],
            self::Korea_Republic => [CurrencyAlpha3::Won],
            self::Kuwait => [CurrencyAlpha3::Kuwaiti_Dinar],
            self::Kyrgyzstan => [CurrencyAlpha3::Som],
            self::Lao_Peoples_Democratic_Republic => [CurrencyAlpha3::Lao_Kip],
            self::Latvia => [CurrencyAlpha3::Euro],
            self::Lebanon => [CurrencyAlpha3::Lebanese_Pound],
            self::Lesotho => [
                CurrencyAlpha3::Loti,
                CurrencyAlpha3::Rand
            ],
            self::Liberia => [CurrencyAlpha3::Liberian_Dollar],
            self::Libya => [CurrencyAlpha3::Libyan_Dinar],
            self::Liechtenstein => [CurrencyAlpha3::Swiss_Franc],
            self::Lithuania => [CurrencyAlpha3::Euro],
            self::Luxembourg => [CurrencyAlpha3::Euro],
            self::Macao => [CurrencyAlpha3::Pataca],
            self::Madagascar => [CurrencyAlpha3::Malagasy_Ariary],
            self::Malawi => [CurrencyAlpha3::Malawi_Kwacha],
            self::Malaysia => [CurrencyAlpha3::Malaysian_Ringgit],
            self::Maldives => [CurrencyAlpha3::Rufiyaa],
            self::Mali => [CurrencyAlpha3::CFA_Franc_BCEAO],
            self::Malta => [CurrencyAlpha3::Euro],
            self::Marshall_Islands => [CurrencyAlpha3::US_Dollar],
            self::Martinique => [CurrencyAlpha3::Euro],
            self::Mauritania => [CurrencyAlpha3::Ouguiya],
            self::Mauritius => [CurrencyAlpha3::Mauritius_Rupee],
            self::Mayotte => [CurrencyAlpha3::Euro],
            self::Mexico => [
                CurrencyAlpha3::Mexican_Peso,
                CurrencyAlpha3::Mexican_Unidad_de_Inversion_UDI
            ],
            self::Micronesia => [CurrencyAlpha3::US_Dollar],
            self::Moldova => [CurrencyAlpha3::Moldovan_Leu],
            self::Monaco => [CurrencyAlpha3::Euro],
            self::Mongolia => [CurrencyAlpha3::Tugrik],
            self::Montenegro => [CurrencyAlpha3::Euro],
            self::Montserrat => [CurrencyAlpha3::East_Caribbean_Dollar],
            self::Morocco => [CurrencyAlpha3::Moroccan_Dirham],
            self::Mozambique => [CurrencyAlpha3::Mozambique_Metical],
            self::Myanmar => [CurrencyAlpha3::Kyat],
            self::Namibia => [
                CurrencyAlpha3::Namibia_Dollar,
                CurrencyAlpha3::Rand
            ],
            self::Nauru => [CurrencyAlpha3::Australian_Dollar],
            self::Nepal => [CurrencyAlpha3::Nepalese_Rupee],
            self::Netherlands => [CurrencyAlpha3::Euro],
            self::New_Caledonia => [CurrencyAlpha3::CFP_Franc],
            self::New_Zealand => [CurrencyAlpha3::New_Zealand_Dollar],
            self::Nicaragua => [CurrencyAlpha3::Cordoba_Oro],
            self::Niger => [CurrencyAlpha3::CFA_Franc_BCEAO],
            self::Nigeria => [CurrencyAlpha3::Naira],
            self::Niue => [CurrencyAlpha3::New_Zealand_Dollar],
            self::Norfolk_Island => [CurrencyAlpha3::Australian_Dollar],
            self::North_Macedonia => [CurrencyAlpha3::Denar],
            self::Northern_Mariana_Islands => [CurrencyAlpha3::US_Dollar],
            self::Norway => [CurrencyAlpha3::Norwegian_Krone],
            self::Oman => [CurrencyAlpha3::Rial_Omani],
            self::Pakistan => [CurrencyAlpha3::Pakistan_Rupee],
            self::Palau => [CurrencyAlpha3::US_Dollar],
            self::Panama => [
                CurrencyAlpha3::Balboa,
                CurrencyAlpha3::US_Dollar
            ],
            self::Papua_New_Guinea => [CurrencyAlpha3::Kina],
            self::Paraguay => [CurrencyAlpha3::Guarani],
            self::Peru => [CurrencyAlpha3::Sol],
            self::Philippines => [CurrencyAlpha3::Philippine_Peso],
            self::Pitcairn => [CurrencyAlpha3::New_Zealand_Dollar],
            self::Poland => [CurrencyAlpha3::Zloty],
            self::Portugal => [CurrencyAlpha3::Euro],
            self::Puerto_Rico => [CurrencyAlpha3::US_Dollar],
            self::Qatar => [CurrencyAlpha3::Qatari_Rial],
            self::Reunion => [CurrencyAlpha3::Euro],
            self::Romania => [CurrencyAlpha3::Romanian_Leu],
            self::Russian_Federation => [CurrencyAlpha3::Russian_Ruble],
            self::Rwanda => [CurrencyAlpha3::Rwanda_Franc],
            self::Saint_Barthelemy => [CurrencyAlpha3::Euro],
            self::Saint_Helena_Ascension_Tristan_da_Cunha => [CurrencyAlpha3::Saint_Helena_Pound],
            self::Saint_Kitts_and_Nevis => [CurrencyAlpha3::East_Caribbean_Dollar],
            self::Saint_Lucia => [CurrencyAlpha3::East_Caribbean_Dollar],
            self::Saint_Martin_French_part => [CurrencyAlpha3::Euro],
            self::Saint_Pierre_and_Miquelon => [CurrencyAlpha3::Euro],
            self::Saint_Vincent_and_the_Grenadines => [CurrencyAlpha3::East_Caribbean_Dollar],
            self::Samoa => [CurrencyAlpha3::Tala],
            self::San_Marino => [CurrencyAlpha3::Euro],
            self::Sao_Tome_and_Principe => [CurrencyAlpha3::Dobra],
            self::Saudi_Arabia => [CurrencyAlpha3::Saudi_Riyal],
            self::Senegal => [CurrencyAlpha3::CFA_Franc_BCEAO],
            self::Serbia => [CurrencyAlpha3::Serbian_Dinar],
            self::Seychelles => [CurrencyAlpha3::Seychelles_Rupee],
            self::Sierra_Leone => [CurrencyAlpha3::Leone],
            self::Singapore => [CurrencyAlpha3::Singapore_Dollar],
            self::Sint_Maarten_Dutch_part => [CurrencyAlpha3::Caribbean_Guilder],
            self::Slovakia => [CurrencyAlpha3::Euro],
            self::Slovenia => [CurrencyAlpha3::Euro],
            self::Solomon_Islands => [CurrencyAlpha3::Solomon_Islands_Dollar],
            self::Somalia => [CurrencyAlpha3::Somali_Shilling],
            self::South_Africa => [CurrencyAlpha3::Rand],
            self::South_Sudan => [CurrencyAlpha3::South_Sudanese_Pound],
            self::Spain => [CurrencyAlpha3::Euro],
            self::Sri_Lanka => [CurrencyAlpha3::Sri_Lanka_Rupee],
            self::Sudan => [CurrencyAlpha3::Sudanese_Pound],
            self::Suriname => [CurrencyAlpha3::Surinam_Dollar],
            self::Svalbard_Jan_Mayen => [CurrencyAlpha3::Norwegian_Krone],
            self::Sweden => [CurrencyAlpha3::Swedish_Krona],
            self::Switzerland => [
                CurrencyAlpha3::Swiss_Franc,
                CurrencyAlpha3::WIR_Euro,
                CurrencyAlpha3::WIR_Franc
            ],
            self::Syrian_Arab_Republic => [CurrencyAlpha3::Syrian_Pound],
            self::Taiwan_Province_of_China => [CurrencyAlpha3::New_Taiwan_Dollar],
            self::Tajikistan => [CurrencyAlpha3::Somoni],
            self::Tanzania => [CurrencyAlpha3::Tanzanian_Shilling],
            self::Thailand => [CurrencyAlpha3::Baht],
            self::Timor_Leste => [CurrencyAlpha3::US_Dollar],
            self::Togo => [CurrencyAlpha3::CFA_Franc_BCEAO],
            self::Tokelau => [CurrencyAlpha3::New_Zealand_Dollar],
            self::Tonga => [CurrencyAlpha3::Pa_anga],
            self::Trinidad_and_Tobago => [CurrencyAlpha3::Trinidad_and_Tobago_Dollar],
            self::Tunisia => [CurrencyAlpha3::Tunisian_Dinar],
            self::Turkey => [CurrencyAlpha3::Turkish_Lira],
            self::Turkmenistan => [CurrencyAlpha3::Turkmenistan_New_Manat],
            self::Turks_and_Caicos_Islands => [CurrencyAlpha3::US_Dollar],
            self::Tuvalu => [CurrencyAlpha3::Australian_Dollar],
            self::Uganda => [CurrencyAlpha3::Uganda_Shilling],
            self::Ukraine => [CurrencyAlpha3::Hryvnia],
            self::United_Arab_Emirates => [CurrencyAlpha3::UAE_Dirham],
            self::United_Kingdom => [CurrencyAlpha3::Pound_Sterling],
            self::United_States_Outlying_Islands => [CurrencyAlpha3::US_Dollar],
            self::United_States_of_America => [
                CurrencyAlpha3::US_Dollar,
                CurrencyAlpha3::US_Dollar_Next_day
            ],
            self::Uruguay => [
                CurrencyAlpha3::Peso_Uruguayo,
                CurrencyAlpha3::Unidad_Previsional,
                CurrencyAlpha3::Uruguay_Peso_en_Unidades_Indexadas_UI
            ],
            self::Uzbekistan => [CurrencyAlpha3::Uzbekistan_Sum],
            self::Vanuatu => [CurrencyAlpha3::Vatu],
            self::Venezuela => [
                CurrencyAlpha3::Bolivar_Soberano,
                CurrencyAlpha3::Bolivar_Soberano_Old
            ],
            self::Viet_Nam => [CurrencyAlpha3::Dong],
            self::Virgin_Islands_British => [CurrencyAlpha3::US_Dollar],
            self::Virgin_Islands_U_S => [CurrencyAlpha3::US_Dollar],
            self::Wallis_and_Futuna => [CurrencyAlpha3::CFP_Franc],
            self::Western_Sahara => [CurrencyAlpha3::Moroccan_Dirham],
            self::Yemen => [CurrencyAlpha3::Yemeni_Rial],
            self::Zambia => [CurrencyAlpha3::Zambian_Kwacha],
            self::Zimbabwe => [CurrencyAlpha3::Zimbabwe_Gold],
            default => []
        };
    }

    public function getCountryCodeTLD(): CountryCodeTLD {
        return match($this) {
            self::Afghanistan => CountryCodeTLD::af,
            self::Aland_Islands => CountryCodeTLD::ax,
            self::Albania => CountryCodeTLD::al,
            self::Algeria => CountryCodeTLD::dz,
            self::American_Samoa => CountryCodeTLD::as,
            self::Andorra => CountryCodeTLD::ad,
            self::Angola => CountryCodeTLD::ao,
            self::Anguilla => CountryCodeTLD::ai,
            self::Antarctica => CountryCodeTLD::aq,
            self::Antigua_and_Barbuda => CountryCodeTLD::ag,
            self::Argentina => CountryCodeTLD::ar,
            self::Armenia => CountryCodeTLD::am,
            self::Aruba => CountryCodeTLD::aw,
            self::Australia => CountryCodeTLD::au,
            self::Austria => CountryCodeTLD::at,
            self::Azerbaijan => CountryCodeTLD::az,
            self::Bahamas => CountryCodeTLD::bs,
            self::Bahrain => CountryCodeTLD::bh,
            self::Bangladesh => CountryCodeTLD::bd,
            self::Barbados => CountryCodeTLD::bb,
            self::Belarus => CountryCodeTLD::by,
            self::Belgium => CountryCodeTLD::be,
            self::Belize => CountryCodeTLD::bz,
            self::Benin => CountryCodeTLD::bj,
            self::Bermuda => CountryCodeTLD::bm,
            self::Bhutan => CountryCodeTLD::bt,
            self::Bolivia => CountryCodeTLD::bo,
            self::Bonaire_Sint_Eustatius_and_Saba => CountryCodeTLD::bq,
            self::Bosnia_and_Herzegovina => CountryCodeTLD::ba,
            self::Botswana => CountryCodeTLD::bw,
            self::Bouvet_Island => CountryCodeTLD::bv,
            self::Brazil => CountryCodeTLD::br,
            self::British_Indian_Ocean_Territory => CountryCodeTLD::io,
            self::Brunei_Darussalam => CountryCodeTLD::bn,
            self::Bulgaria => CountryCodeTLD::bg,
            self::Burkina_Faso => CountryCodeTLD::bf,
            self::Burundi => CountryCodeTLD::bi,
            self::Cabo_Verde => CountryCodeTLD::cv,
            self::Cambodia => CountryCodeTLD::kh,
            self::Cameroon => CountryCodeTLD::cm,
            self::Canada => CountryCodeTLD::ca,
            self::Cayman_Islands => CountryCodeTLD::ky,
            self::Central_African_Republic => CountryCodeTLD::cf,
            self::Chad => CountryCodeTLD::td,
            self::Chile => CountryCodeTLD::cl,
            self::China => CountryCodeTLD::cn,
            self::Christmas_Island => CountryCodeTLD::cx,
            self::Cocos_Islands => CountryCodeTLD::cc,
            self::Colombia => CountryCodeTLD::co,
            self::Comoros => CountryCodeTLD::km,
            self::Congo => CountryCodeTLD::cg,
            self::Congo_Democratic_Republic => CountryCodeTLD::cd,
            self::Cook_Islands => CountryCodeTLD::ck,
            self::Costa_Rica => CountryCodeTLD::cr,
            self::Cote_d_Ivoire => CountryCodeTLD::ci,
            self::Croatia => CountryCodeTLD::hr,
            self::Cuba => CountryCodeTLD::cu,
            self::Curacao => CountryCodeTLD::cw,
            self::Cyprus => CountryCodeTLD::cy,
            self::Czechia => CountryCodeTLD::cz,
            self::Denmark => CountryCodeTLD::dk,
            self::Djibouti => CountryCodeTLD::dj,
            self::Dominica => CountryCodeTLD::dm,
            self::Dominican_Republic => CountryCodeTLD::do,
            self::Ecuador => CountryCodeTLD::ec,
            self::Egypt => CountryCodeTLD::eg,
            self::El_Salvador => CountryCodeTLD::sv,
            self::Equatorial_Guinea => CountryCodeTLD::gq,
            self::Eritrea => CountryCodeTLD::er,
            self::Estonia => CountryCodeTLD::ee,
            self::Eswatini => CountryCodeTLD::sz,
            self::Ethiopia => CountryCodeTLD::et,
            self::Falkland_Islands => CountryCodeTLD::fk,
            self::Faroe_Islands => CountryCodeTLD::fo,
            self::Fiji => CountryCodeTLD::fj,
            self::Finland => CountryCodeTLD::fi,
            self::France => CountryCodeTLD::fr,
            self::French_Guiana => CountryCodeTLD::gf,
            self::French_Polynesia => CountryCodeTLD::pf,
            self::French_Southern_Territories => CountryCodeTLD::tf,
            self::Gabon => CountryCodeTLD::ga,
            self::Gambia => CountryCodeTLD::gm,
            self::Georgia => CountryCodeTLD::ge,
            self::Germany => CountryCodeTLD::de,
            self::Ghana => CountryCodeTLD::gh,
            self::Gibraltar => CountryCodeTLD::gi,
            self::Greece => CountryCodeTLD::gr,
            self::Greenland => CountryCodeTLD::gl,
            self::Grenada => CountryCodeTLD::gd,
            self::Guadeloupe => CountryCodeTLD::gp,
            self::Guam => CountryCodeTLD::gu,
            self::Guatemala => CountryCodeTLD::gt,
            self::Guernsey => CountryCodeTLD::gg,
            self::Guinea => CountryCodeTLD::gn,
            self::Guinea_Bissau => CountryCodeTLD::gw,
            self::Guyana => CountryCodeTLD::gy,
            self::Haiti => CountryCodeTLD::ht,
            self::Heard_Island_and_McDonald_Islands => CountryCodeTLD::hm,
            self::Holy_See => CountryCodeTLD::va,
            self::Honduras => CountryCodeTLD::hn,
            self::Hong_Kong => CountryCodeTLD::hk,
            self::Hungary => CountryCodeTLD::hu,
            self::Iceland => CountryCodeTLD::is,
            self::India => CountryCodeTLD::in,
            self::Indonesia => CountryCodeTLD::id,
            self::Iran => CountryCodeTLD::ir,
            self::Iraq => CountryCodeTLD::iq,
            self::Ireland => CountryCodeTLD::ie,
            self::Isle_of_Man => CountryCodeTLD::im,
            self::Israel => CountryCodeTLD::il,
            self::Italy => CountryCodeTLD::it,
            self::Jamaica => CountryCodeTLD::jm,
            self::Japan => CountryCodeTLD::jp,
            self::Jersey => CountryCodeTLD::je,
            self::Jordan => CountryCodeTLD::jo,
            self::Kazakhstan => CountryCodeTLD::kz,
            self::Kenya => CountryCodeTLD::ke,
            self::Kiribati => CountryCodeTLD::ki,
            self::Korea_Democratic_Peoples_Republic => CountryCodeTLD::kp,
            self::Korea_Republic => CountryCodeTLD::kr,
            self::Kuwait => CountryCodeTLD::kw,
            self::Kyrgyzstan => CountryCodeTLD::kg,
            self::Lao_Peoples_Democratic_Republic => CountryCodeTLD::la,
            self::Latvia => CountryCodeTLD::lv,
            self::Lebanon => CountryCodeTLD::lb,
            self::Lesotho => CountryCodeTLD::ls,
            self::Liberia => CountryCodeTLD::lr,
            self::Libya => CountryCodeTLD::ly,
            self::Liechtenstein => CountryCodeTLD::li,
            self::Lithuania => CountryCodeTLD::lt,
            self::Luxembourg => CountryCodeTLD::lu,
            self::Macao => CountryCodeTLD::mo,
            self::Madagascar => CountryCodeTLD::mg,
            self::Malawi => CountryCodeTLD::mw,
            self::Malaysia => CountryCodeTLD::my,
            self::Maldives => CountryCodeTLD::mv,
            self::Mali => CountryCodeTLD::ml,
            self::Malta => CountryCodeTLD::mt,
            self::Marshall_Islands => CountryCodeTLD::mh,
            self::Martinique => CountryCodeTLD::mq,
            self::Mauritania => CountryCodeTLD::mr,
            self::Mauritius => CountryCodeTLD::mu,
            self::Mayotte => CountryCodeTLD::yt,
            self::Mexico => CountryCodeTLD::mx,
            self::Micronesia => CountryCodeTLD::fm,
            self::Moldova => CountryCodeTLD::md,
            self::Monaco => CountryCodeTLD::mc,
            self::Mongolia => CountryCodeTLD::mn,
            self::Montenegro => CountryCodeTLD::me,
            self::Montserrat => CountryCodeTLD::ms,
            self::Morocco => CountryCodeTLD::ma,
            self::Mozambique => CountryCodeTLD::mz,
            self::Myanmar => CountryCodeTLD::mm,
            self::Namibia => CountryCodeTLD::na,
            self::Nauru => CountryCodeTLD::nr,
            self::Nepal => CountryCodeTLD::np,
            self::Netherlands => CountryCodeTLD::nl,
            self::New_Caledonia => CountryCodeTLD::nc,
            self::New_Zealand => CountryCodeTLD::nz,
            self::Nicaragua => CountryCodeTLD::ni,
            self::Niger => CountryCodeTLD::ne,
            self::Nigeria => CountryCodeTLD::ng,
            self::Niue => CountryCodeTLD::nu,
            self::Norfolk_Island => CountryCodeTLD::nf,
            self::North_Macedonia => CountryCodeTLD::mk,
            self::Northern_Mariana_Islands => CountryCodeTLD::mp,
            self::Norway => CountryCodeTLD::no,
            self::Oman => CountryCodeTLD::om,
            self::Pakistan => CountryCodeTLD::pk,
            self::Palau => CountryCodeTLD::pw,
            self::Palestine => CountryCodeTLD::ps,
            self::Panama => CountryCodeTLD::pa,
            self::Papua_New_Guinea => CountryCodeTLD::pg,
            self::Paraguay => CountryCodeTLD::py,
            self::Peru => CountryCodeTLD::pe,
            self::Philippines => CountryCodeTLD::ph,
            self::Pitcairn => CountryCodeTLD::pn,
            self::Poland => CountryCodeTLD::pl,
            self::Portugal => CountryCodeTLD::pt,
            self::Puerto_Rico => CountryCodeTLD::pr,
            self::Qatar => CountryCodeTLD::qa,
            self::Reunion => CountryCodeTLD::re,
            self::Romania => CountryCodeTLD::ro,
            self::Russian_Federation => CountryCodeTLD::ru,
            self::Rwanda => CountryCodeTLD::rw,
            self::Saint_Barthelemy => CountryCodeTLD::bl,
            self::Saint_Helena_Ascension_Tristan_da_Cunha => CountryCodeTLD::sh,
            self::Saint_Kitts_and_Nevis => CountryCodeTLD::kn,
            self::Saint_Lucia => CountryCodeTLD::lc,
            self::Saint_Martin_French_part => CountryCodeTLD::mf,
            self::Saint_Pierre_and_Miquelon => CountryCodeTLD::pm,
            self::Saint_Vincent_and_the_Grenadines => CountryCodeTLD::vc,
            self::Samoa => CountryCodeTLD::ws,
            self::San_Marino => CountryCodeTLD::sm,
            self::Sao_Tome_and_Principe => CountryCodeTLD::st,
            self::Saudi_Arabia => CountryCodeTLD::sa,
            self::Senegal => CountryCodeTLD::sn,
            self::Serbia => CountryCodeTLD::rs,
            self::Seychelles => CountryCodeTLD::sc,
            self::Sierra_Leone => CountryCodeTLD::sl,
            self::Singapore => CountryCodeTLD::sg,
            self::Sint_Maarten_Dutch_part => CountryCodeTLD::sx,
            self::Slovakia => CountryCodeTLD::sk,
            self::Slovenia => CountryCodeTLD::si,
            self::Solomon_Islands => CountryCodeTLD::sb,
            self::Somalia => CountryCodeTLD::so,
            self::South_Africa => CountryCodeTLD::za,
            self::South_Georgia_South_Sandwich_Islands => CountryCodeTLD::gs,
            self::South_Sudan => CountryCodeTLD::ss,
            self::Spain => CountryCodeTLD::es,
            self::Sri_Lanka => CountryCodeTLD::lk,
            self::Sudan => CountryCodeTLD::sd,
            self::Suriname => CountryCodeTLD::sr,
            self::Svalbard_Jan_Mayen => CountryCodeTLD::sj,
            self::Sweden => CountryCodeTLD::se,
            self::Switzerland => CountryCodeTLD::ch,
            self::Syrian_Arab_Republic => CountryCodeTLD::sy,
            self::Taiwan_Province_of_China => CountryCodeTLD::tw,
            self::Tajikistan => CountryCodeTLD::tj,
            self::Tanzania => CountryCodeTLD::tz,
            self::Thailand => CountryCodeTLD::th,
            self::Timor_Leste => CountryCodeTLD::tl,
            self::Togo => CountryCodeTLD::tg,
            self::Tokelau => CountryCodeTLD::tk,
            self::Tonga => CountryCodeTLD::to,
            self::Trinidad_and_Tobago => CountryCodeTLD::tt,
            self::Tunisia => CountryCodeTLD::tn,
            self::Turkey => CountryCodeTLD::tr,
            self::Turkmenistan => CountryCodeTLD::tm,
            self::Turks_and_Caicos_Islands => CountryCodeTLD::tc,
            self::Tuvalu => CountryCodeTLD::tv,
            self::Uganda => CountryCodeTLD::ug,
            self::Ukraine => CountryCodeTLD::ua,
            self::United_Arab_Emirates => CountryCodeTLD::ae,
            self::United_Kingdom => CountryCodeTLD::gb,
            self::United_States_Outlying_Islands => CountryCodeTLD::um,
            self::United_States_of_America => CountryCodeTLD::us,
            self::Uruguay => CountryCodeTLD::uy,
            self::Uzbekistan => CountryCodeTLD::uz,
            self::Vanuatu => CountryCodeTLD::vu,
            self::Venezuela => CountryCodeTLD::ve,
            self::Viet_Nam => CountryCodeTLD::vn,
            self::Virgin_Islands_British => CountryCodeTLD::vg,
            self::Virgin_Islands_U_S => CountryCodeTLD::vi,
            self::Wallis_and_Futuna => CountryCodeTLD::wf,
            self::Western_Sahara => CountryCodeTLD::eh,
            self::Yemen => CountryCodeTLD::ye,
            self::Zambia => CountryCodeTLD::zm,
            self::Zimbabwe => CountryCodeTLD::zw,
        };
    }

    public function getInternationalCallPrefix(): InternationalCallPrefix {
        return InternationalCallPrefix::forCountry($this);
    }

    /** @return list<CountrySubdivision> */
    public function getSubdivisions(): array {
        return match($this) {
            self::Afghanistan => [
                CountrySubdivision::Afghanistan_province_Badakhshan,
                CountrySubdivision::Afghanistan_province_Badghis,
                CountrySubdivision::Afghanistan_province_Baghlan,
                CountrySubdivision::Afghanistan_province_Balkh,
                CountrySubdivision::Afghanistan_province_Bamyan,
                CountrySubdivision::Afghanistan_province_Daykundi,
                CountrySubdivision::Afghanistan_province_Farah,
                CountrySubdivision::Afghanistan_province_Faryab,
                CountrySubdivision::Afghanistan_province_Ghazni,
                CountrySubdivision::Afghanistan_province_Ghor,
                CountrySubdivision::Afghanistan_province_Helmand,
                CountrySubdivision::Afghanistan_province_Herat,
                CountrySubdivision::Afghanistan_province_Jowzjan,
                CountrySubdivision::Afghanistan_province_Kabul,
                CountrySubdivision::Afghanistan_province_Kandahar,
                CountrySubdivision::Afghanistan_province_Kapisa,
                CountrySubdivision::Afghanistan_province_Khost,
                CountrySubdivision::Afghanistan_province_Kunar,
                CountrySubdivision::Afghanistan_province_Kunduz,
                CountrySubdivision::Afghanistan_province_Laghman,
                CountrySubdivision::Afghanistan_province_Logar,
                CountrySubdivision::Afghanistan_province_Nangarhar,
                CountrySubdivision::Afghanistan_province_Nimroz,
                CountrySubdivision::Afghanistan_province_Nuristan,
                CountrySubdivision::Afghanistan_province_Paktika,
                CountrySubdivision::Afghanistan_province_Paktiya,
                CountrySubdivision::Afghanistan_province_Panjshayr,
                CountrySubdivision::Afghanistan_province_Parwan,
                CountrySubdivision::Afghanistan_province_Samangan,
                CountrySubdivision::Afghanistan_province_Sar_e_Pul,
                CountrySubdivision::Afghanistan_province_Takhar,
                CountrySubdivision::Afghanistan_province_Uruzgan,
                CountrySubdivision::Afghanistan_province_Wardak,
                CountrySubdivision::Afghanistan_province_Zabul
            ],
            self::Albania => [
                CountrySubdivision::Albania_county_Berat,
                CountrySubdivision::Albania_county_Diber,
                CountrySubdivision::Albania_county_Durres,
                CountrySubdivision::Albania_county_Elbasan,
                CountrySubdivision::Albania_county_Fier,
                CountrySubdivision::Albania_county_Gjirokaster,
                CountrySubdivision::Albania_county_Korce,
                CountrySubdivision::Albania_county_Kukes,
                CountrySubdivision::Albania_county_Lezhe,
                CountrySubdivision::Albania_county_Shkoder,
                CountrySubdivision::Albania_county_Tirane,
                CountrySubdivision::Albania_county_Vlore
            ],
            self::Algeria => [
                CountrySubdivision::Algeria_province_Adrar,
                CountrySubdivision::Algeria_province_Ain_Defla,
                CountrySubdivision::Algeria_province_Ain_Temouchent,
                CountrySubdivision::Algeria_province_Alger,
                CountrySubdivision::Algeria_province_Annaba,
                CountrySubdivision::Algeria_province_Batna,
                CountrySubdivision::Algeria_province_Bechar,
                CountrySubdivision::Algeria_province_Bejaia,
                CountrySubdivision::Algeria_province_Beni_Abbes,
                CountrySubdivision::Algeria_province_Biskra,
                CountrySubdivision::Algeria_province_Blida,
                CountrySubdivision::Algeria_province_Bordj_Badji_Mokhtar,
                CountrySubdivision::Algeria_province_Bordj_Bou_Arreridj,
                CountrySubdivision::Algeria_province_Bouira,
                CountrySubdivision::Algeria_province_Boumerdes,
                CountrySubdivision::Algeria_province_Chlef,
                CountrySubdivision::Algeria_province_Constantine,
                CountrySubdivision::Algeria_province_Djanet,
                CountrySubdivision::Algeria_province_Djelfa,
                CountrySubdivision::Algeria_province_El_Bayadh,
                CountrySubdivision::Algeria_province_El_Meghaier,
                CountrySubdivision::Algeria_province_El_Meniaa,
                CountrySubdivision::Algeria_province_El_Oued,
                CountrySubdivision::Algeria_province_El_Tarf,
                CountrySubdivision::Algeria_province_Ghardaia,
                CountrySubdivision::Algeria_province_Guelma,
                CountrySubdivision::Algeria_province_Illizi,
                CountrySubdivision::Algeria_province_In_Guezzam,
                CountrySubdivision::Algeria_province_In_Salah,
                CountrySubdivision::Algeria_province_Jijel,
                CountrySubdivision::Algeria_province_Khenchela,
                CountrySubdivision::Algeria_province_Laghouat,
                CountrySubdivision::Algeria_province_M_sila,
                CountrySubdivision::Algeria_province_Mascara,
                CountrySubdivision::Algeria_province_Medea,
                CountrySubdivision::Algeria_province_Mila,
                CountrySubdivision::Algeria_province_Mostaganem,
                CountrySubdivision::Algeria_province_Naama,
                CountrySubdivision::Algeria_province_Oran,
                CountrySubdivision::Algeria_province_Ouargla,
                CountrySubdivision::Algeria_province_Ouled_Djellal,
                CountrySubdivision::Algeria_province_Oum_el_Bouaghi,
                CountrySubdivision::Algeria_province_Relizane,
                CountrySubdivision::Algeria_province_Saida,
                CountrySubdivision::Algeria_province_Setif,
                CountrySubdivision::Algeria_province_Sidi_Bel_Abbes,
                CountrySubdivision::Algeria_province_Skikda,
                CountrySubdivision::Algeria_province_Souk_Ahras,
                CountrySubdivision::Algeria_province_Tamanrasset,
                CountrySubdivision::Algeria_province_Tebessa,
                CountrySubdivision::Algeria_province_Tiaret,
                CountrySubdivision::Algeria_province_Timimoun,
                CountrySubdivision::Algeria_province_Tindouf,
                CountrySubdivision::Algeria_province_Tipaza,
                CountrySubdivision::Algeria_province_Tissemsilt,
                CountrySubdivision::Algeria_province_Tizi_Ouzou,
                CountrySubdivision::Algeria_province_Tlemcen,
                CountrySubdivision::Algeria_province_Touggourt
            ],
            self::Andorra => [
                CountrySubdivision::Andorra_parish_Andorra_la_Vella,
                CountrySubdivision::Andorra_parish_Canillo,
                CountrySubdivision::Andorra_parish_Encamp,
                CountrySubdivision::Andorra_parish_Escaldes_Engordany,
                CountrySubdivision::Andorra_parish_La_Massana,
                CountrySubdivision::Andorra_parish_Ordino,
                CountrySubdivision::Andorra_parish_Sant_Julia_de_Loria
            ],
            self::Angola => [
                CountrySubdivision::Angola_province_Bengo,
                CountrySubdivision::Angola_province_Benguela,
                CountrySubdivision::Angola_province_Bie,
                CountrySubdivision::Angola_province_Cabinda,
                CountrySubdivision::Angola_province_Cuando_Cubango,
                CountrySubdivision::Angola_province_Cuanza_Norte,
                CountrySubdivision::Angola_province_Cuanza_Sul,
                CountrySubdivision::Angola_province_Cunene,
                CountrySubdivision::Angola_province_Huambo,
                CountrySubdivision::Angola_province_Huila,
                CountrySubdivision::Angola_province_Luanda,
                CountrySubdivision::Angola_province_Lunda_Norte,
                CountrySubdivision::Angola_province_Lunda_Sul,
                CountrySubdivision::Angola_province_Malange,
                CountrySubdivision::Angola_province_Moxico,
                CountrySubdivision::Angola_province_Namibe,
                CountrySubdivision::Angola_province_Uige,
                CountrySubdivision::Angola_province_Zaire
            ],
            self::Antigua_and_Barbuda => [
                CountrySubdivision::Antigua_Barbuda_dependency_Barbuda,
                CountrySubdivision::Antigua_Barbuda_dependency_Redonda,
                CountrySubdivision::Antigua_Barbuda_parish_Saint_George,
                CountrySubdivision::Antigua_Barbuda_parish_Saint_John,
                CountrySubdivision::Antigua_Barbuda_parish_Saint_Mary,
                CountrySubdivision::Antigua_Barbuda_parish_Saint_Paul,
                CountrySubdivision::Antigua_Barbuda_parish_Saint_Peter,
                CountrySubdivision::Antigua_Barbuda_parish_Saint_Philip
            ],
            self::Argentina => [
                CountrySubdivision::Argentina_city_Ciudad_Autonoma_de_Buenos_Aires,
                CountrySubdivision::Argentina_province_Buenos_Aires,
                CountrySubdivision::Argentina_province_Catamarca,
                CountrySubdivision::Argentina_province_Chaco,
                CountrySubdivision::Argentina_province_Chubut,
                CountrySubdivision::Argentina_province_Cordoba,
                CountrySubdivision::Argentina_province_Corrientes,
                CountrySubdivision::Argentina_province_Entre_Rios,
                CountrySubdivision::Argentina_province_Formosa,
                CountrySubdivision::Argentina_province_Jujuy,
                CountrySubdivision::Argentina_province_La_Pampa,
                CountrySubdivision::Argentina_province_La_Rioja,
                CountrySubdivision::Argentina_province_Mendoza,
                CountrySubdivision::Argentina_province_Misiones,
                CountrySubdivision::Argentina_province_Neuquen,
                CountrySubdivision::Argentina_province_Rio_Negro,
                CountrySubdivision::Argentina_province_Salta,
                CountrySubdivision::Argentina_province_San_Juan,
                CountrySubdivision::Argentina_province_San_Luis,
                CountrySubdivision::Argentina_province_Santa_Cruz,
                CountrySubdivision::Argentina_province_Santa_Fe,
                CountrySubdivision::Argentina_province_Santiago_del_Estero,
                CountrySubdivision::Argentina_province_Tierra_del_Fuego,
                CountrySubdivision::Argentina_province_Tucuman
            ],
            self::Armenia => [
                CountrySubdivision::Armenia_city_Erevan,
                CountrySubdivision::Armenia_region_Aragacotn,
                CountrySubdivision::Armenia_region_Ararat,
                CountrySubdivision::Armenia_region_Armavir,
                CountrySubdivision::Armenia_region_Gegark_unik,
                CountrySubdivision::Armenia_region_Kotayk,
                CountrySubdivision::Armenia_region_Lori,
                CountrySubdivision::Armenia_region_Sirak,
                CountrySubdivision::Armenia_region_Syunik,
                CountrySubdivision::Armenia_region_Tavus,
                CountrySubdivision::Armenia_region_Vayoc_Jor
            ],
            self::Australia => [
                CountrySubdivision::Australia_state_New_South_Wales,
                CountrySubdivision::Australia_state_Queensland,
                CountrySubdivision::Australia_state_South_Australia,
                CountrySubdivision::Australia_state_Tasmania,
                CountrySubdivision::Australia_state_Victoria,
                CountrySubdivision::Australia_state_Western_Australia,
                CountrySubdivision::Australia_territory_Australian_Capital_Territory,
                CountrySubdivision::Australia_territory_Northern_Territory
            ],
            self::Austria => [
                CountrySubdivision::Austria_state_Burgenland,
                CountrySubdivision::Austria_state_Karnten,
                CountrySubdivision::Austria_state_Niederosterreich,
                CountrySubdivision::Austria_state_Oberosterreich,
                CountrySubdivision::Austria_state_Salzburg,
                CountrySubdivision::Austria_state_Steiermark,
                CountrySubdivision::Austria_state_Tirol,
                CountrySubdivision::Austria_state_Vorarlberg,
                CountrySubdivision::Austria_state_Wien
            ],
            self::Azerbaijan => [
                CountrySubdivision::Azerbaijan_autonomous_republic_Naxcivan,
                CountrySubdivision::Azerbaijan_municipality_Baki,
                CountrySubdivision::Azerbaijan_municipality_Gnc,
                CountrySubdivision::Azerbaijan_municipality_Lnkran,
                CountrySubdivision::Azerbaijan_municipality_Mingcevir,
                CountrySubdivision::Azerbaijan_municipality_Naftalan,
                CountrySubdivision::Azerbaijan_municipality_Naxcivan,
                CountrySubdivision::Azerbaijan_municipality_Sirvan,
                CountrySubdivision::Azerbaijan_municipality_Ski,
                CountrySubdivision::Azerbaijan_municipality_Sumqayit,
                CountrySubdivision::Azerbaijan_municipality_Xankndi,
                CountrySubdivision::Azerbaijan_municipality_Yevlax,
                CountrySubdivision::Azerbaijan_rayon_Abseron,
                CountrySubdivision::Azerbaijan_rayon_Agcabdi,
                CountrySubdivision::Azerbaijan_rayon_Agdam,
                CountrySubdivision::Azerbaijan_rayon_Agdas,
                CountrySubdivision::Azerbaijan_rayon_Agstafa,
                CountrySubdivision::Azerbaijan_rayon_Agsu,
                CountrySubdivision::Azerbaijan_rayon_Astara,
                CountrySubdivision::Azerbaijan_rayon_Babk,
                CountrySubdivision::Azerbaijan_rayon_Balakn,
                CountrySubdivision::Azerbaijan_rayon_Beylqan,
                CountrySubdivision::Azerbaijan_rayon_Bilsuvar,
                CountrySubdivision::Azerbaijan_rayon_Brd,
                CountrySubdivision::Azerbaijan_rayon_Cbrayil,
                CountrySubdivision::Azerbaijan_rayon_Clilabad,
                CountrySubdivision::Azerbaijan_rayon_Culfa,
                CountrySubdivision::Azerbaijan_rayon_Dasksn,
                CountrySubdivision::Azerbaijan_rayon_Fuzuli,
                CountrySubdivision::Azerbaijan_rayon_Gdby,
                CountrySubdivision::Azerbaijan_rayon_Goranboy,
                CountrySubdivision::Azerbaijan_rayon_Goycay,
                CountrySubdivision::Azerbaijan_rayon_Goygol,
                CountrySubdivision::Azerbaijan_rayon_Haciqabul,
                CountrySubdivision::Azerbaijan_rayon_Imisli,
                CountrySubdivision::Azerbaijan_rayon_Ismayilli,
                CountrySubdivision::Azerbaijan_rayon_Klbcr,
                CountrySubdivision::Azerbaijan_rayon_Kngrli,
                CountrySubdivision::Azerbaijan_rayon_Kurdmir,
                CountrySubdivision::Azerbaijan_rayon_Lacin,
                CountrySubdivision::Azerbaijan_rayon_Lerik,
                CountrySubdivision::Azerbaijan_rayon_Lnkran,
                CountrySubdivision::Azerbaijan_rayon_Masalli,
                CountrySubdivision::Azerbaijan_rayon_Neftcala,
                CountrySubdivision::Azerbaijan_rayon_Oguz,
                CountrySubdivision::Azerbaijan_rayon_Ordubad,
                CountrySubdivision::Azerbaijan_rayon_Qax,
                CountrySubdivision::Azerbaijan_rayon_Qazax,
                CountrySubdivision::Azerbaijan_rayon_Qbl,
                CountrySubdivision::Azerbaijan_rayon_Qobustan,
                CountrySubdivision::Azerbaijan_rayon_Quba,
                CountrySubdivision::Azerbaijan_rayon_Qubadli,
                CountrySubdivision::Azerbaijan_rayon_Qusar,
                CountrySubdivision::Azerbaijan_rayon_Saatli,
                CountrySubdivision::Azerbaijan_rayon_Sabirabad,
                CountrySubdivision::Azerbaijan_rayon_Sabran,
                CountrySubdivision::Azerbaijan_rayon_Sahbuz,
                CountrySubdivision::Azerbaijan_rayon_Salyan,
                CountrySubdivision::Azerbaijan_rayon_Samaxi,
                CountrySubdivision::Azerbaijan_rayon_Samux,
                CountrySubdivision::Azerbaijan_rayon_Sdrk,
                CountrySubdivision::Azerbaijan_rayon_Siyzn,
                CountrySubdivision::Azerbaijan_rayon_Ski,
                CountrySubdivision::Azerbaijan_rayon_Smkir,
                CountrySubdivision::Azerbaijan_rayon_Srur,
                CountrySubdivision::Azerbaijan_rayon_Susa,
                CountrySubdivision::Azerbaijan_rayon_Tovuz,
                CountrySubdivision::Azerbaijan_rayon_Trtr,
                CountrySubdivision::Azerbaijan_rayon_Ucar,
                CountrySubdivision::Azerbaijan_rayon_Xacmaz,
                CountrySubdivision::Azerbaijan_rayon_Xizi,
                CountrySubdivision::Azerbaijan_rayon_Xocali,
                CountrySubdivision::Azerbaijan_rayon_Xocavnd,
                CountrySubdivision::Azerbaijan_rayon_Yardimli,
                CountrySubdivision::Azerbaijan_rayon_Yevlax,
                CountrySubdivision::Azerbaijan_rayon_Zaqatala,
                CountrySubdivision::Azerbaijan_rayon_Zngilan,
                CountrySubdivision::Azerbaijan_rayon_Zrdab
            ],
            self::Bahamas => [
                CountrySubdivision::Bahamas_district_Acklins,
                CountrySubdivision::Bahamas_district_Berry_Islands,
                CountrySubdivision::Bahamas_district_Bimini,
                CountrySubdivision::Bahamas_district_Black_Point,
                CountrySubdivision::Bahamas_district_Cat_Island,
                CountrySubdivision::Bahamas_district_Central_Abaco,
                CountrySubdivision::Bahamas_district_Central_Andros,
                CountrySubdivision::Bahamas_district_Central_Eleuthera,
                CountrySubdivision::Bahamas_district_City_of_Freeport,
                CountrySubdivision::Bahamas_district_Crooked_Island_and_Long_Cay,
                CountrySubdivision::Bahamas_district_East_Grand_Bahama,
                CountrySubdivision::Bahamas_district_Exuma,
                CountrySubdivision::Bahamas_district_Grand_Cay,
                CountrySubdivision::Bahamas_district_Harbour_Island,
                CountrySubdivision::Bahamas_district_Hope_Town,
                CountrySubdivision::Bahamas_district_Inagua,
                CountrySubdivision::Bahamas_district_Long_Island,
                CountrySubdivision::Bahamas_district_Mangrove_Cay,
                CountrySubdivision::Bahamas_district_Mayaguana,
                CountrySubdivision::Bahamas_district_Moore_s_Island,
                CountrySubdivision::Bahamas_district_North_Abaco,
                CountrySubdivision::Bahamas_district_North_Andros,
                CountrySubdivision::Bahamas_district_North_Eleuthera,
                CountrySubdivision::Bahamas_district_Ragged_Island,
                CountrySubdivision::Bahamas_district_Rum_Cay,
                CountrySubdivision::Bahamas_district_San_Salvador,
                CountrySubdivision::Bahamas_district_South_Abaco,
                CountrySubdivision::Bahamas_district_South_Andros,
                CountrySubdivision::Bahamas_district_South_Eleuthera,
                CountrySubdivision::Bahamas_district_Spanish_Wells,
                CountrySubdivision::Bahamas_district_West_Grand_Bahama,
                CountrySubdivision::Bahamas_island_New_Providence
            ],
            self::Bahrain => [
                CountrySubdivision::Bahrain_governorate_Al_Asimah,
                CountrySubdivision::Bahrain_governorate_Al_Janubiyah,
                CountrySubdivision::Bahrain_governorate_Al_Muharraq,
                CountrySubdivision::Bahrain_governorate_Ash_Shamaliyah
            ],
            self::Bangladesh => [
                CountrySubdivision::Bangladesh_district_Bagerhat,
                CountrySubdivision::Bangladesh_district_Bandarban,
                CountrySubdivision::Bangladesh_district_Barguna,
                CountrySubdivision::Bangladesh_district_Barishal,
                CountrySubdivision::Bangladesh_district_Bhola,
                CountrySubdivision::Bangladesh_district_Bogura,
                CountrySubdivision::Bangladesh_district_Brahmanbaria,
                CountrySubdivision::Bangladesh_district_Chandpur,
                CountrySubdivision::Bangladesh_district_Chapai_Nawabganj,
                CountrySubdivision::Bangladesh_district_Chattogram,
                CountrySubdivision::Bangladesh_district_Chuadanga,
                CountrySubdivision::Bangladesh_district_Cox_s_Bazar,
                CountrySubdivision::Bangladesh_district_Cumilla,
                CountrySubdivision::Bangladesh_district_Dhaka,
                CountrySubdivision::Bangladesh_district_Dinajpur,
                CountrySubdivision::Bangladesh_district_Faridpur,
                CountrySubdivision::Bangladesh_district_Feni,
                CountrySubdivision::Bangladesh_district_Gaibandha,
                CountrySubdivision::Bangladesh_district_Gazipur,
                CountrySubdivision::Bangladesh_district_Gopalganj,
                CountrySubdivision::Bangladesh_district_Habiganj,
                CountrySubdivision::Bangladesh_district_Jamalpur,
                CountrySubdivision::Bangladesh_district_Jashore,
                CountrySubdivision::Bangladesh_district_Jhalakathi,
                CountrySubdivision::Bangladesh_district_Jhenaidah,
                CountrySubdivision::Bangladesh_district_Joypurhat,
                CountrySubdivision::Bangladesh_district_Khagrachhari,
                CountrySubdivision::Bangladesh_district_Khulna,
                CountrySubdivision::Bangladesh_district_Kishoreganj,
                CountrySubdivision::Bangladesh_district_Kurigram,
                CountrySubdivision::Bangladesh_district_Kushtia,
                CountrySubdivision::Bangladesh_district_Lakshmipur,
                CountrySubdivision::Bangladesh_district_Lalmonirhat,
                CountrySubdivision::Bangladesh_district_Madaripur,
                CountrySubdivision::Bangladesh_district_Magura,
                CountrySubdivision::Bangladesh_district_Manikganj,
                CountrySubdivision::Bangladesh_district_Meherpur,
                CountrySubdivision::Bangladesh_district_Moulvibazar,
                CountrySubdivision::Bangladesh_district_Munshiganj,
                CountrySubdivision::Bangladesh_district_Mymensingh,
                CountrySubdivision::Bangladesh_district_Naogaon,
                CountrySubdivision::Bangladesh_district_Narail,
                CountrySubdivision::Bangladesh_district_Narayanganj,
                CountrySubdivision::Bangladesh_district_Narsingdi,
                CountrySubdivision::Bangladesh_district_Natore,
                CountrySubdivision::Bangladesh_district_Netrakona,
                CountrySubdivision::Bangladesh_district_Nilphamari,
                CountrySubdivision::Bangladesh_district_Noakhali,
                CountrySubdivision::Bangladesh_district_Pabna,
                CountrySubdivision::Bangladesh_district_Panchagarh,
                CountrySubdivision::Bangladesh_district_Patuakhali,
                CountrySubdivision::Bangladesh_district_Pirojpur,
                CountrySubdivision::Bangladesh_district_Rajbari,
                CountrySubdivision::Bangladesh_district_Rajshahi,
                CountrySubdivision::Bangladesh_district_Rangamati,
                CountrySubdivision::Bangladesh_district_Rangpur,
                CountrySubdivision::Bangladesh_district_Satkhira,
                CountrySubdivision::Bangladesh_district_Shariatpur,
                CountrySubdivision::Bangladesh_district_Sherpur,
                CountrySubdivision::Bangladesh_district_Sirajganj,
                CountrySubdivision::Bangladesh_district_Sunamganj,
                CountrySubdivision::Bangladesh_district_Sylhet,
                CountrySubdivision::Bangladesh_district_Tangail,
                CountrySubdivision::Bangladesh_district_Thakurgaon,
                CountrySubdivision::Bangladesh_division_Barishal,
                CountrySubdivision::Bangladesh_division_Chattogram,
                CountrySubdivision::Bangladesh_division_Dhaka,
                CountrySubdivision::Bangladesh_division_Khulna,
                CountrySubdivision::Bangladesh_division_Mymensingh,
                CountrySubdivision::Bangladesh_division_Rajshahi,
                CountrySubdivision::Bangladesh_division_Rangpur,
                CountrySubdivision::Bangladesh_division_Sylhet
            ],
            self::Barbados => [
                CountrySubdivision::Barbados_parish_Christ_Church,
                CountrySubdivision::Barbados_parish_Saint_Andrew,
                CountrySubdivision::Barbados_parish_Saint_George,
                CountrySubdivision::Barbados_parish_Saint_James,
                CountrySubdivision::Barbados_parish_Saint_John,
                CountrySubdivision::Barbados_parish_Saint_Joseph,
                CountrySubdivision::Barbados_parish_Saint_Lucy,
                CountrySubdivision::Barbados_parish_Saint_Michael,
                CountrySubdivision::Barbados_parish_Saint_Peter,
                CountrySubdivision::Barbados_parish_Saint_Philip,
                CountrySubdivision::Barbados_parish_Saint_Thomas
            ],
            self::Belarus => [
                CountrySubdivision::Belarus_city_Gorod_Minsk,
                CountrySubdivision::Belarus_oblast_Bresckaja_voblasc,
                CountrySubdivision::Belarus_oblast_Gomel_skaja_oblast,
                CountrySubdivision::Belarus_oblast_Grodnenskaja_oblast,
                CountrySubdivision::Belarus_oblast_Mahiliouskaja_voblasc,
                CountrySubdivision::Belarus_oblast_Minskaja_oblast,
                CountrySubdivision::Belarus_oblast_Viciebskaja_voblasc
            ],
            self::Belgium => [
                CountrySubdivision::Belgium_province_Antwerpen,
                CountrySubdivision::Belgium_province_Brabant_wallon,
                CountrySubdivision::Belgium_province_Hainaut,
                CountrySubdivision::Belgium_province_Liege,
                CountrySubdivision::Belgium_province_Limburg,
                CountrySubdivision::Belgium_province_Luxembourg,
                CountrySubdivision::Belgium_province_Namur,
                CountrySubdivision::Belgium_province_Oost_Vlaanderen,
                CountrySubdivision::Belgium_province_Vlaams_Brabant,
                CountrySubdivision::Belgium_province_West_Vlaanderen,
                CountrySubdivision::Belgium_region_Brussels_Hoofdstedelijk_Gewest,
                CountrySubdivision::Belgium_region_Vlaams_Gewest,
                CountrySubdivision::Belgium_region_wallonne_Region
            ],
            self::Belize => [
                CountrySubdivision::Belize_district_Belize,
                CountrySubdivision::Belize_district_Cayo,
                CountrySubdivision::Belize_district_Corozal,
                CountrySubdivision::Belize_district_Orange_Walk,
                CountrySubdivision::Belize_district_Stann_Creek,
                CountrySubdivision::Belize_district_Toledo
            ],
            self::Benin => [
                CountrySubdivision::Benin_department_Alibori,
                CountrySubdivision::Benin_department_Atacora,
                CountrySubdivision::Benin_department_Atlantique,
                CountrySubdivision::Benin_department_Borgou,
                CountrySubdivision::Benin_department_Collines,
                CountrySubdivision::Benin_department_Couffo,
                CountrySubdivision::Benin_department_Donga,
                CountrySubdivision::Benin_department_Littoral,
                CountrySubdivision::Benin_department_Mono,
                CountrySubdivision::Benin_department_Oueme,
                CountrySubdivision::Benin_department_Plateau,
                CountrySubdivision::Benin_department_Zou
            ],
            self::Bhutan => [
                CountrySubdivision::Bhutan_district_Bumthang,
                CountrySubdivision::Bhutan_district_Chhukha,
                CountrySubdivision::Bhutan_district_Dagana,
                CountrySubdivision::Bhutan_district_Gasa,
                CountrySubdivision::Bhutan_district_Haa,
                CountrySubdivision::Bhutan_district_Lhuentse,
                CountrySubdivision::Bhutan_district_Monggar,
                CountrySubdivision::Bhutan_district_Paro,
                CountrySubdivision::Bhutan_district_Pema_Gatshel,
                CountrySubdivision::Bhutan_district_Punakha,
                CountrySubdivision::Bhutan_district_Samdrup_Jongkhar,
                CountrySubdivision::Bhutan_district_Samtse,
                CountrySubdivision::Bhutan_district_Sarpang,
                CountrySubdivision::Bhutan_district_Thimphu,
                CountrySubdivision::Bhutan_district_Trashi_Yangtse,
                CountrySubdivision::Bhutan_district_Trashigang,
                CountrySubdivision::Bhutan_district_Trongsa,
                CountrySubdivision::Bhutan_district_Tsirang,
                CountrySubdivision::Bhutan_district_Wangdue_Phodrang,
                CountrySubdivision::Bhutan_district_Zhemgang
            ],
            self::Bolivia => [
                CountrySubdivision::Bolivia_department_Chuquisaca,
                CountrySubdivision::Bolivia_department_Cochabamba,
                CountrySubdivision::Bolivia_department_El_Beni,
                CountrySubdivision::Bolivia_department_La_Paz,
                CountrySubdivision::Bolivia_department_Oruro,
                CountrySubdivision::Bolivia_department_Pando,
                CountrySubdivision::Bolivia_department_Potosi,
                CountrySubdivision::Bolivia_department_Santa_Cruz,
                CountrySubdivision::Bolivia_department_Tarija
            ],
            self::Bonaire_Sint_Eustatius_and_Saba => [
                CountrySubdivision::Caribbean_Netherlands_special_municipality_Bonaire,
                CountrySubdivision::Caribbean_Netherlands_special_municipality_Saba,
                CountrySubdivision::Caribbean_Netherlands_special_municipality_Sint_Eustatius
            ],
            self::Bosnia_and_Herzegovina => [
                CountrySubdivision::Bosnia_Herzegovina_district_with_special_status_Brcko_distrikt,
                CountrySubdivision::Bosnia_Herzegovina_entity_Federacija_Bosne_i_Hercegovine,
                CountrySubdivision::Bosnia_Herzegovina_entity_Republika_Srpska
            ],
            self::Botswana => [
                CountrySubdivision::Botswana_city_Francistown,
                CountrySubdivision::Botswana_city_Gaborone,
                CountrySubdivision::Botswana_district_Central,
                CountrySubdivision::Botswana_district_Chobe,
                CountrySubdivision::Botswana_district_Ghanzi,
                CountrySubdivision::Botswana_district_Kgalagadi,
                CountrySubdivision::Botswana_district_Kgatleng,
                CountrySubdivision::Botswana_district_Kweneng,
                CountrySubdivision::Botswana_district_North_East,
                CountrySubdivision::Botswana_district_North_West,
                CountrySubdivision::Botswana_district_South_East,
                CountrySubdivision::Botswana_district_Southern,
                CountrySubdivision::Botswana_town_Jwaneng,
                CountrySubdivision::Botswana_town_Lobatse,
                CountrySubdivision::Botswana_town_Selibe_Phikwe,
                CountrySubdivision::Botswana_town_Sowa_Town
            ],
            self::Brazil => [
                CountrySubdivision::Brazil_federal_district_Distrito_Federal,
                CountrySubdivision::Brazil_state_Acre,
                CountrySubdivision::Brazil_state_Alagoas,
                CountrySubdivision::Brazil_state_Amapa,
                CountrySubdivision::Brazil_state_Amazonas,
                CountrySubdivision::Brazil_state_Bahia,
                CountrySubdivision::Brazil_state_Ceara,
                CountrySubdivision::Brazil_state_Espirito_Santo,
                CountrySubdivision::Brazil_state_Goias,
                CountrySubdivision::Brazil_state_Maranhao,
                CountrySubdivision::Brazil_state_Mato_Grosso,
                CountrySubdivision::Brazil_state_Mato_Grosso_do_Sul,
                CountrySubdivision::Brazil_state_Minas_Gerais,
                CountrySubdivision::Brazil_state_Para,
                CountrySubdivision::Brazil_state_Paraiba,
                CountrySubdivision::Brazil_state_Parana,
                CountrySubdivision::Brazil_state_Pernambuco,
                CountrySubdivision::Brazil_state_Piaui,
                CountrySubdivision::Brazil_state_Rio_Grande_do_Norte,
                CountrySubdivision::Brazil_state_Rio_Grande_do_Sul,
                CountrySubdivision::Brazil_state_Rio_de_Janeiro,
                CountrySubdivision::Brazil_state_Rondonia,
                CountrySubdivision::Brazil_state_Roraima,
                CountrySubdivision::Brazil_state_Santa_Catarina,
                CountrySubdivision::Brazil_state_Sao_Paulo,
                CountrySubdivision::Brazil_state_Sergipe,
                CountrySubdivision::Brazil_state_Tocantins
            ],
            self::Brunei_Darussalam => [
                CountrySubdivision::Brunei_district_Belait,
                CountrySubdivision::Brunei_district_Brunei_dan_Muara,
                CountrySubdivision::Brunei_district_Temburong,
                CountrySubdivision::Brunei_district_Tutong
            ],
            self::Bulgaria => [
                CountrySubdivision::Bulgaria_district_Blagoevgrad,
                CountrySubdivision::Bulgaria_district_Burgas,
                CountrySubdivision::Bulgaria_district_Dobrich,
                CountrySubdivision::Bulgaria_district_Gabrovo,
                CountrySubdivision::Bulgaria_district_Haskovo,
                CountrySubdivision::Bulgaria_district_Kardzhali,
                CountrySubdivision::Bulgaria_district_Kyustendil,
                CountrySubdivision::Bulgaria_district_Lovech,
                CountrySubdivision::Bulgaria_district_Montana,
                CountrySubdivision::Bulgaria_district_Pazardzhik,
                CountrySubdivision::Bulgaria_district_Pernik,
                CountrySubdivision::Bulgaria_district_Pleven,
                CountrySubdivision::Bulgaria_district_Plovdiv,
                CountrySubdivision::Bulgaria_district_Razgrad,
                CountrySubdivision::Bulgaria_district_Ruse,
                CountrySubdivision::Bulgaria_district_Shumen,
                CountrySubdivision::Bulgaria_district_Silistra,
                CountrySubdivision::Bulgaria_district_Sliven,
                CountrySubdivision::Bulgaria_district_Smolyan,
                CountrySubdivision::Bulgaria_district_Sofia,
                CountrySubdivision::Bulgaria_district_Sofia_stolitsa,
                CountrySubdivision::Bulgaria_district_Stara_Zagora,
                CountrySubdivision::Bulgaria_district_Targovishte,
                CountrySubdivision::Bulgaria_district_Varna,
                CountrySubdivision::Bulgaria_district_Veliko_Tarnovo,
                CountrySubdivision::Bulgaria_district_Vidin,
                CountrySubdivision::Bulgaria_district_Vratsa,
                CountrySubdivision::Bulgaria_district_Yambol
            ],
            self::Burkina_Faso => [
                CountrySubdivision::Burkina_Faso_province_Bale,
                CountrySubdivision::Burkina_Faso_province_Bam,
                CountrySubdivision::Burkina_Faso_province_Banwa,
                CountrySubdivision::Burkina_Faso_province_Bazega,
                CountrySubdivision::Burkina_Faso_province_Bougouriba,
                CountrySubdivision::Burkina_Faso_province_Boulgou,
                CountrySubdivision::Burkina_Faso_province_Boulkiemde,
                CountrySubdivision::Burkina_Faso_province_Comoe,
                CountrySubdivision::Burkina_Faso_province_Ganzourgou,
                CountrySubdivision::Burkina_Faso_province_Gnagna,
                CountrySubdivision::Burkina_Faso_province_Gourma,
                CountrySubdivision::Burkina_Faso_province_Houet,
                CountrySubdivision::Burkina_Faso_province_Ioba,
                CountrySubdivision::Burkina_Faso_province_Kadiogo,
                CountrySubdivision::Burkina_Faso_province_Kenedougou,
                CountrySubdivision::Burkina_Faso_province_Komondjari,
                CountrySubdivision::Burkina_Faso_province_Kompienga,
                CountrySubdivision::Burkina_Faso_province_Kossi,
                CountrySubdivision::Burkina_Faso_province_Koulpelogo,
                CountrySubdivision::Burkina_Faso_province_Kouritenga,
                CountrySubdivision::Burkina_Faso_province_Kourweogo,
                CountrySubdivision::Burkina_Faso_province_Leraba,
                CountrySubdivision::Burkina_Faso_province_Loroum,
                CountrySubdivision::Burkina_Faso_province_Mouhoun,
                CountrySubdivision::Burkina_Faso_province_Nahouri,
                CountrySubdivision::Burkina_Faso_province_Namentenga,
                CountrySubdivision::Burkina_Faso_province_Nayala,
                CountrySubdivision::Burkina_Faso_province_Noumbiel,
                CountrySubdivision::Burkina_Faso_province_Oubritenga,
                CountrySubdivision::Burkina_Faso_province_Oudalan,
                CountrySubdivision::Burkina_Faso_province_Passore,
                CountrySubdivision::Burkina_Faso_province_Poni,
                CountrySubdivision::Burkina_Faso_province_Sanguie,
                CountrySubdivision::Burkina_Faso_province_Sanmatenga,
                CountrySubdivision::Burkina_Faso_province_Seno,
                CountrySubdivision::Burkina_Faso_province_Sissili,
                CountrySubdivision::Burkina_Faso_province_Soum,
                CountrySubdivision::Burkina_Faso_province_Sourou,
                CountrySubdivision::Burkina_Faso_province_Tapoa,
                CountrySubdivision::Burkina_Faso_province_Tuy,
                CountrySubdivision::Burkina_Faso_province_Yagha,
                CountrySubdivision::Burkina_Faso_province_Yatenga,
                CountrySubdivision::Burkina_Faso_province_Ziro,
                CountrySubdivision::Burkina_Faso_province_Zondoma,
                CountrySubdivision::Burkina_Faso_province_Zoundweogo,
                CountrySubdivision::Burkina_Faso_region_Boucle_du_Mouhoun,
                CountrySubdivision::Burkina_Faso_region_Cascades,
                CountrySubdivision::Burkina_Faso_region_Centre,
                CountrySubdivision::Burkina_Faso_region_Centre_Est,
                CountrySubdivision::Burkina_Faso_region_Centre_Nord,
                CountrySubdivision::Burkina_Faso_region_Centre_Ouest,
                CountrySubdivision::Burkina_Faso_region_Centre_Sud,
                CountrySubdivision::Burkina_Faso_region_Est,
                CountrySubdivision::Burkina_Faso_region_Hauts_Bassins,
                CountrySubdivision::Burkina_Faso_region_Nord,
                CountrySubdivision::Burkina_Faso_region_Plateau_Central,
                CountrySubdivision::Burkina_Faso_region_Sahel,
                CountrySubdivision::Burkina_Faso_region_Sud_Ouest
            ],
            self::Burundi => [
                CountrySubdivision::Burundi_province_Bubanza,
                CountrySubdivision::Burundi_province_Bujumbura_Mairie,
                CountrySubdivision::Burundi_province_Bujumbura_Rural,
                CountrySubdivision::Burundi_province_Bururi,
                CountrySubdivision::Burundi_province_Cankuzo,
                CountrySubdivision::Burundi_province_Cibitoke,
                CountrySubdivision::Burundi_province_Gitega,
                CountrySubdivision::Burundi_province_Karuzi,
                CountrySubdivision::Burundi_province_Kayanza,
                CountrySubdivision::Burundi_province_Kirundo,
                CountrySubdivision::Burundi_province_Makamba,
                CountrySubdivision::Burundi_province_Muramvya,
                CountrySubdivision::Burundi_province_Muyinga,
                CountrySubdivision::Burundi_province_Mwaro,
                CountrySubdivision::Burundi_province_Ngozi,
                CountrySubdivision::Burundi_province_Rumonge,
                CountrySubdivision::Burundi_province_Rutana,
                CountrySubdivision::Burundi_province_Ruyigi
            ],
            self::Cabo_Verde => [
                CountrySubdivision::Cape_Verde_geographical_region_Ilhas_de_Barlavento,
                CountrySubdivision::Cape_Verde_geographical_region_Ilhas_de_Sotavento,
                CountrySubdivision::Cape_Verde_municipality_Boa_Vista,
                CountrySubdivision::Cape_Verde_municipality_Brava,
                CountrySubdivision::Cape_Verde_municipality_Maio,
                CountrySubdivision::Cape_Verde_municipality_Mosteiros,
                CountrySubdivision::Cape_Verde_municipality_Paul,
                CountrySubdivision::Cape_Verde_municipality_Porto_Novo,
                CountrySubdivision::Cape_Verde_municipality_Praia,
                CountrySubdivision::Cape_Verde_municipality_Ribeira_Brava,
                CountrySubdivision::Cape_Verde_municipality_Ribeira_Grande,
                CountrySubdivision::Cape_Verde_municipality_Ribeira_Grande_de_Santiago,
                CountrySubdivision::Cape_Verde_municipality_Sal,
                CountrySubdivision::Cape_Verde_municipality_Santa_Catarina,
                CountrySubdivision::Cape_Verde_municipality_Santa_Catarina_do_Fogo,
                CountrySubdivision::Cape_Verde_municipality_Santa_Cruz,
                CountrySubdivision::Cape_Verde_municipality_Sao_Domingos,
                CountrySubdivision::Cape_Verde_municipality_Sao_Filipe,
                CountrySubdivision::Cape_Verde_municipality_Sao_Lourenco_dos_Orgaos,
                CountrySubdivision::Cape_Verde_municipality_Sao_Miguel,
                CountrySubdivision::Cape_Verde_municipality_Sao_Salvador_do_Mundo,
                CountrySubdivision::Cape_Verde_municipality_Sao_Vicente,
                CountrySubdivision::Cape_Verde_municipality_Tarrafal,
                CountrySubdivision::Cape_Verde_municipality_Tarrafal_de_Sao_Nicolau
            ],
            self::Cambodia => [
                CountrySubdivision::Cambodia_autonomous_municipality_Phnom_Penh,
                CountrySubdivision::Cambodia_province_Baat_Dambang,
                CountrySubdivision::Cambodia_province_Banteay_Mean_Choay,
                CountrySubdivision::Cambodia_province_Kaeb,
                CountrySubdivision::Cambodia_province_Kampong_Chaam,
                CountrySubdivision::Cambodia_province_Kampong_Chhnang,
                CountrySubdivision::Cambodia_province_Kampong_Spueu,
                CountrySubdivision::Cambodia_province_Kampong_Thum,
                CountrySubdivision::Cambodia_province_Kampot,
                CountrySubdivision::Cambodia_province_Kandaal,
                CountrySubdivision::Cambodia_province_Kaoh_Kong,
                CountrySubdivision::Cambodia_province_Kracheh,
                CountrySubdivision::Cambodia_province_Mondol_Kiri,
                CountrySubdivision::Cambodia_province_Otdar_Mean_Chey,
                CountrySubdivision::Cambodia_province_Pailin,
                CountrySubdivision::Cambodia_province_Pousaat,
                CountrySubdivision::Cambodia_province_Preah_Sihanouk,
                CountrySubdivision::Cambodia_province_Preah_Vihear,
                CountrySubdivision::Cambodia_province_Prey_Veaeng,
                CountrySubdivision::Cambodia_province_Rotanak_Kiri,
                CountrySubdivision::Cambodia_province_Siem_Reab,
                CountrySubdivision::Cambodia_province_Stoeng_Treng,
                CountrySubdivision::Cambodia_province_Svaay_Rieng,
                CountrySubdivision::Cambodia_province_Taakaev,
                CountrySubdivision::Cambodia_province_Tbong_Khmum
            ],
            self::Cameroon => [
                CountrySubdivision::Cameroon_region_Adamaoua,
                CountrySubdivision::Cameroon_region_Centre,
                CountrySubdivision::Cameroon_region_East,
                CountrySubdivision::Cameroon_region_Extreme_Nord,
                CountrySubdivision::Cameroon_region_Littoral,
                CountrySubdivision::Cameroon_region_Nord,
                CountrySubdivision::Cameroon_region_Nord_Ouest,
                CountrySubdivision::Cameroon_region_Ouest,
                CountrySubdivision::Cameroon_region_South,
                CountrySubdivision::Cameroon_region_South_West
            ],
            self::Canada => [
                CountrySubdivision::Canada_province_Alberta,
                CountrySubdivision::Canada_province_British_Columbia,
                CountrySubdivision::Canada_province_Manitoba,
                CountrySubdivision::Canada_province_New_Brunswick,
                CountrySubdivision::Canada_province_Newfoundland_and_Labrador,
                CountrySubdivision::Canada_province_Nouvelle_Ecosse,
                CountrySubdivision::Canada_province_Ontario,
                CountrySubdivision::Canada_province_Prince_Edward_Island,
                CountrySubdivision::Canada_province_Quebec,
                CountrySubdivision::Canada_province_Saskatchewan,
                CountrySubdivision::Canada_territory_Northwest_Territories,
                CountrySubdivision::Canada_territory_Nunavut,
                CountrySubdivision::Canada_territory_Yukon
            ],
            self::Central_African_Republic => [
                CountrySubdivision::Central_African_Republic_commune_Bangui,
                CountrySubdivision::Central_African_Republic_economic_prefecture_Gribingui,
                CountrySubdivision::Central_African_Republic_economic_prefecture_Sangha,
                CountrySubdivision::Central_African_Republic_prefecture_Bamingui_Bangoran,
                CountrySubdivision::Central_African_Republic_prefecture_Basse_Kotto,
                CountrySubdivision::Central_African_Republic_prefecture_Haut_Mbomou,
                CountrySubdivision::Central_African_Republic_prefecture_Haute_Kotto,
                CountrySubdivision::Central_African_Republic_prefecture_Haute_Sangha_Mambere_Kadei,
                CountrySubdivision::Central_African_Republic_prefecture_Kemo_Giribingi,
                CountrySubdivision::Central_African_Republic_prefecture_Lobaye,
                CountrySubdivision::Central_African_Republic_prefecture_Mbomou,
                CountrySubdivision::Central_African_Republic_prefecture_Nana_Mambere,
                CountrySubdivision::Central_African_Republic_prefecture_Ombella_Mpoko,
                CountrySubdivision::Central_African_Republic_prefecture_Ouaka,
                CountrySubdivision::Central_African_Republic_prefecture_Ouham,
                CountrySubdivision::Central_African_Republic_prefecture_Ouham_Pende,
                CountrySubdivision::Central_African_Republic_prefecture_Vakaga
            ],
            self::Chad => [
                CountrySubdivision::Chad_province_Al_Batha,
                CountrySubdivision::Chad_province_Al_Buhayrah,
                CountrySubdivision::Chad_province_Bahr_el_Ghazal,
                CountrySubdivision::Chad_province_Borkou,
                CountrySubdivision::Chad_province_Chari_Baguirmi,
                CountrySubdivision::Chad_province_Ennedi_Est,
                CountrySubdivision::Chad_province_Ennedi_Ouest,
                CountrySubdivision::Chad_province_Guera,
                CountrySubdivision::Chad_province_Hadjer_Lamis,
                CountrySubdivision::Chad_province_Kanem,
                CountrySubdivision::Chad_province_Logone_Occidental,
                CountrySubdivision::Chad_province_Logone_Oriental,
                CountrySubdivision::Chad_province_Madinat_Injamina,
                CountrySubdivision::Chad_province_Mandoul,
                CountrySubdivision::Chad_province_Mayo_Kebbi_Est,
                CountrySubdivision::Chad_province_Mayo_Kebbi_Ouest,
                CountrySubdivision::Chad_province_Moyen_Chari,
                CountrySubdivision::Chad_province_Ouaddai,
                CountrySubdivision::Chad_province_Salamat,
                CountrySubdivision::Chad_province_Sila,
                CountrySubdivision::Chad_province_Tandjile,
                CountrySubdivision::Chad_province_Tibasti,
                CountrySubdivision::Chad_province_Wadi_Fira
            ],
            self::Chile => [
                CountrySubdivision::Chile_region_Aisen_del_General_Carlos_Ibanez_del_Campo,
                CountrySubdivision::Chile_region_Antofagasta,
                CountrySubdivision::Chile_region_Arica_y_Parinacota,
                CountrySubdivision::Chile_region_Atacama,
                CountrySubdivision::Chile_region_Biobio,
                CountrySubdivision::Chile_region_Coquimbo,
                CountrySubdivision::Chile_region_La_Araucania,
                CountrySubdivision::Chile_region_Libertador_General_Bernardo_O_Higgins,
                CountrySubdivision::Chile_region_Los_Lagos,
                CountrySubdivision::Chile_region_Los_Rios,
                CountrySubdivision::Chile_region_Magallanes,
                CountrySubdivision::Chile_region_Maule,
                CountrySubdivision::Chile_region_Nuble,
                CountrySubdivision::Chile_region_Region_Metropolitana_de_Santiago,
                CountrySubdivision::Chile_region_Tarapaca,
                CountrySubdivision::Chile_region_Valparaiso
            ],
            self::China => [
                CountrySubdivision::China_autonomous_region_Guangxi_Zhuangzu_Zizhiqu,
                CountrySubdivision::China_autonomous_region_Nei_Mongol_Zizhiqu,
                CountrySubdivision::China_autonomous_region_Ningxia_Huizu_Zizhiqu,
                CountrySubdivision::China_autonomous_region_Xinjiang_Uygur_Zizhiqu,
                CountrySubdivision::China_autonomous_region_Xizang_Zizhiqu,
                CountrySubdivision::China_municipality_Beijing_Shi,
                CountrySubdivision::China_municipality_Chongqing_Shi,
                CountrySubdivision::China_municipality_Shanghai_Shi,
                CountrySubdivision::China_municipality_Tianjin_Shi,
                CountrySubdivision::China_province_Anhui_Sheng,
                CountrySubdivision::China_province_Fujian_Sheng,
                CountrySubdivision::China_province_Gansu_Sheng,
                CountrySubdivision::China_province_Guangdong_Sheng,
                CountrySubdivision::China_province_Guizhou_Sheng,
                CountrySubdivision::China_province_Hainan_Sheng,
                CountrySubdivision::China_province_Hebei_Sheng,
                CountrySubdivision::China_province_Heilongjiang_Sheng,
                CountrySubdivision::China_province_Henan_Sheng,
                CountrySubdivision::China_province_Hubei_Sheng,
                CountrySubdivision::China_province_Hunan_Sheng,
                CountrySubdivision::China_province_Jiangsu_Sheng,
                CountrySubdivision::China_province_Jiangxi_Sheng,
                CountrySubdivision::China_province_Jilin_Sheng,
                CountrySubdivision::China_province_Liaoning_Sheng,
                CountrySubdivision::China_province_Qinghai_Sheng,
                CountrySubdivision::China_province_Shaanxi_Sheng,
                CountrySubdivision::China_province_Shandong_Sheng,
                CountrySubdivision::China_province_Shanxi_Sheng,
                CountrySubdivision::China_province_Sichuan_Sheng,
                CountrySubdivision::China_province_Taiwan_Sheng,
                CountrySubdivision::China_province_Yunnan_Sheng,
                CountrySubdivision::China_province_Zhejiang_Sheng,
                CountrySubdivision::China_special_administrative_region_Aomen_Tebiexingzhengqu,
                CountrySubdivision::China_special_administrative_region_Hong_Kong_SAR
            ],
            self::Colombia => [
                CountrySubdivision::Colombia_capital_district_Distrito_Capital_de_Bogota,
                CountrySubdivision::Colombia_department_Amazonas,
                CountrySubdivision::Colombia_department_Antioquia,
                CountrySubdivision::Colombia_department_Arauca,
                CountrySubdivision::Colombia_department_Atlantico,
                CountrySubdivision::Colombia_department_Bolivar,
                CountrySubdivision::Colombia_department_Boyaca,
                CountrySubdivision::Colombia_department_Caldas,
                CountrySubdivision::Colombia_department_Caqueta,
                CountrySubdivision::Colombia_department_Casanare,
                CountrySubdivision::Colombia_department_Cauca,
                CountrySubdivision::Colombia_department_Cesar,
                CountrySubdivision::Colombia_department_Choco,
                CountrySubdivision::Colombia_department_Cordoba,
                CountrySubdivision::Colombia_department_Cundinamarca,
                CountrySubdivision::Colombia_department_Guainia,
                CountrySubdivision::Colombia_department_Guaviare,
                CountrySubdivision::Colombia_department_Huila,
                CountrySubdivision::Colombia_department_La_Guajira,
                CountrySubdivision::Colombia_department_Magdalena,
                CountrySubdivision::Colombia_department_Meta,
                CountrySubdivision::Colombia_department_Narino,
                CountrySubdivision::Colombia_department_Norte_de_Santander,
                CountrySubdivision::Colombia_department_Putumayo,
                CountrySubdivision::Colombia_department_Quindio,
                CountrySubdivision::Colombia_department_Risaralda,
                CountrySubdivision::Colombia_department_San_Andres_Providencia_y_Santa_Catalina,
                CountrySubdivision::Colombia_department_Santander,
                CountrySubdivision::Colombia_department_Sucre,
                CountrySubdivision::Colombia_department_Tolima,
                CountrySubdivision::Colombia_department_Valle_del_Cauca,
                CountrySubdivision::Colombia_department_Vaupes,
                CountrySubdivision::Colombia_department_Vichada
            ],
            self::Comoros => [
                CountrySubdivision::Comoros_island_Andjazidja,
                CountrySubdivision::Comoros_island_Andjouan,
                CountrySubdivision::Comoros_island_Moheli
            ],
            self::Congo => [
                CountrySubdivision::Congo_Brazzaville_department_Bouenza,
                CountrySubdivision::Congo_Brazzaville_department_Brazzaville,
                CountrySubdivision::Congo_Brazzaville_department_Cuvette,
                CountrySubdivision::Congo_Brazzaville_department_Cuvette_Ouest,
                CountrySubdivision::Congo_Brazzaville_department_Kouilou,
                CountrySubdivision::Congo_Brazzaville_department_Lekoumou,
                CountrySubdivision::Congo_Brazzaville_department_Likouala,
                CountrySubdivision::Congo_Brazzaville_department_Niari,
                CountrySubdivision::Congo_Brazzaville_department_Plateaux,
                CountrySubdivision::Congo_Brazzaville_department_Pointe_Noire,
                CountrySubdivision::Congo_Brazzaville_department_Pool,
                CountrySubdivision::Congo_Brazzaville_department_Sangha
            ],
            self::Congo_Democratic_Republic => [
                CountrySubdivision::Congo_Kinshasa_city_Kinshasa,
                CountrySubdivision::Congo_Kinshasa_province_Bas_Uele,
                CountrySubdivision::Congo_Kinshasa_province_Equateur,
                CountrySubdivision::Congo_Kinshasa_province_Haut_Katanga,
                CountrySubdivision::Congo_Kinshasa_province_Haut_Lomami,
                CountrySubdivision::Congo_Kinshasa_province_Haut_Uele,
                CountrySubdivision::Congo_Kinshasa_province_Ituri,
                CountrySubdivision::Congo_Kinshasa_province_Kasai,
                CountrySubdivision::Congo_Kinshasa_province_Kasai_Central,
                CountrySubdivision::Congo_Kinshasa_province_Kasai_Oriental,
                CountrySubdivision::Congo_Kinshasa_province_Kongo_Central,
                CountrySubdivision::Congo_Kinshasa_province_Kwango,
                CountrySubdivision::Congo_Kinshasa_province_Kwilu,
                CountrySubdivision::Congo_Kinshasa_province_Lomami,
                CountrySubdivision::Congo_Kinshasa_province_Lualaba,
                CountrySubdivision::Congo_Kinshasa_province_Mai_Ndombe,
                CountrySubdivision::Congo_Kinshasa_province_Maniema,
                CountrySubdivision::Congo_Kinshasa_province_Mongala,
                CountrySubdivision::Congo_Kinshasa_province_Nord_Kivu,
                CountrySubdivision::Congo_Kinshasa_province_Nord_Ubangi,
                CountrySubdivision::Congo_Kinshasa_province_Sankuru,
                CountrySubdivision::Congo_Kinshasa_province_Sud_Kivu,
                CountrySubdivision::Congo_Kinshasa_province_Sud_Ubangi,
                CountrySubdivision::Congo_Kinshasa_province_Tanganyika,
                CountrySubdivision::Congo_Kinshasa_province_Tshopo,
                CountrySubdivision::Congo_Kinshasa_province_Tshuapa
            ],
            self::Costa_Rica => [
                CountrySubdivision::Costa_Rica_province_Alajuela,
                CountrySubdivision::Costa_Rica_province_Cartago,
                CountrySubdivision::Costa_Rica_province_Guanacaste,
                CountrySubdivision::Costa_Rica_province_Heredia,
                CountrySubdivision::Costa_Rica_province_Limon,
                CountrySubdivision::Costa_Rica_province_Puntarenas,
                CountrySubdivision::Costa_Rica_province_San_Jose
            ],
            self::Cote_d_Ivoire => [
                CountrySubdivision::Cote_d_Ivoire_autonomous_district_Abidjan,
                CountrySubdivision::Cote_d_Ivoire_autonomous_district_Yamoussoukro,
                CountrySubdivision::Cote_d_Ivoire_district_Bas_Sassandra,
                CountrySubdivision::Cote_d_Ivoire_district_Comoe,
                CountrySubdivision::Cote_d_Ivoire_district_Denguele,
                CountrySubdivision::Cote_d_Ivoire_district_Goh_Djiboua,
                CountrySubdivision::Cote_d_Ivoire_district_Lacs,
                CountrySubdivision::Cote_d_Ivoire_district_Lagunes,
                CountrySubdivision::Cote_d_Ivoire_district_Montagnes,
                CountrySubdivision::Cote_d_Ivoire_district_Sassandra_Marahoue,
                CountrySubdivision::Cote_d_Ivoire_district_Savanes,
                CountrySubdivision::Cote_d_Ivoire_district_Vallee_du_Bandama,
                CountrySubdivision::Cote_d_Ivoire_district_Woroba,
                CountrySubdivision::Cote_d_Ivoire_district_Zanzan
            ],
            self::Croatia => [
                CountrySubdivision::Croatia_city_Grad_Zagreb,
                CountrySubdivision::Croatia_county_Bjelovarsko_bilogorska_zupanija,
                CountrySubdivision::Croatia_county_Brodsko_posavska_zupanija,
                CountrySubdivision::Croatia_county_Dubrovacko_neretvanska_zupanija,
                CountrySubdivision::Croatia_county_Istarska_zupanija,
                CountrySubdivision::Croatia_county_Karlovacka_zupanija,
                CountrySubdivision::Croatia_county_Koprivnicko_krizevacka_zupanija,
                CountrySubdivision::Croatia_county_Krapinsko_zagorska_zupanija,
                CountrySubdivision::Croatia_county_Licko_senjska_zupanija,
                CountrySubdivision::Croatia_county_Medimurska_zupanija,
                CountrySubdivision::Croatia_county_Osjecko_baranjska_zupanija,
                CountrySubdivision::Croatia_county_Pozesko_slavonska_zupanija,
                CountrySubdivision::Croatia_county_Primorsko_goranska_zupanija,
                CountrySubdivision::Croatia_county_Sibensko_kninska_zupanija,
                CountrySubdivision::Croatia_county_Sisacko_moslavacka_zupanija,
                CountrySubdivision::Croatia_county_Splitsko_dalmatinska_zupanija,
                CountrySubdivision::Croatia_county_Varazdinska_zupanija,
                CountrySubdivision::Croatia_county_Viroviticko_podravska_zupanija,
                CountrySubdivision::Croatia_county_Vukovarsko_srijemska_zupanija,
                CountrySubdivision::Croatia_county_Zadarska_zupanija,
                CountrySubdivision::Croatia_county_Zagrebacka_zupanija
            ],
            self::Cuba => [
                CountrySubdivision::Cuba_province_Artemisa,
                CountrySubdivision::Cuba_province_Camaguey,
                CountrySubdivision::Cuba_province_Ciego_de_Avila,
                CountrySubdivision::Cuba_province_Cienfuegos,
                CountrySubdivision::Cuba_province_Granma,
                CountrySubdivision::Cuba_province_Guantanamo,
                CountrySubdivision::Cuba_province_Holguin,
                CountrySubdivision::Cuba_province_La_Habana,
                CountrySubdivision::Cuba_province_Las_Tunas,
                CountrySubdivision::Cuba_province_Matanzas,
                CountrySubdivision::Cuba_province_Mayabeque,
                CountrySubdivision::Cuba_province_Pinar_del_Rio,
                CountrySubdivision::Cuba_province_Sancti_Spiritus,
                CountrySubdivision::Cuba_province_Santiago_de_Cuba,
                CountrySubdivision::Cuba_province_Villa_Clara,
                CountrySubdivision::Cuba_special_municipality_Isla_de_la_Juventud
            ],
            self::Cyprus => [
                CountrySubdivision::Cyprus_district_Ammochostos,
                CountrySubdivision::Cyprus_district_Baf,
                CountrySubdivision::Cyprus_district_Girne,
                CountrySubdivision::Cyprus_district_Larnaka,
                CountrySubdivision::Cyprus_district_Lefkosia,
                CountrySubdivision::Cyprus_district_Lemesos
            ],
            self::Czechia => [
                CountrySubdivision::Czechia_capital_city_Praha_Hlavni_mesto,
                CountrySubdivision::Czechia_district_Benesov,
                CountrySubdivision::Czechia_district_Beroun,
                CountrySubdivision::Czechia_district_Blansko,
                CountrySubdivision::Czechia_district_Breclav,
                CountrySubdivision::Czechia_district_Brno_mesto,
                CountrySubdivision::Czechia_district_Brno_venkov,
                CountrySubdivision::Czechia_district_Bruntal,
                CountrySubdivision::Czechia_district_Ceska_Lipa,
                CountrySubdivision::Czechia_district_Ceske_Budejovice,
                CountrySubdivision::Czechia_district_Cesky_Krumlov,
                CountrySubdivision::Czechia_district_Cheb,
                CountrySubdivision::Czechia_district_Chomutov,
                CountrySubdivision::Czechia_district_Chrudim,
                CountrySubdivision::Czechia_district_Decin,
                CountrySubdivision::Czechia_district_Domazlice,
                CountrySubdivision::Czechia_district_Frydek_Mistek,
                CountrySubdivision::Czechia_district_Havlickuv_Brod,
                CountrySubdivision::Czechia_district_Hodonin,
                CountrySubdivision::Czechia_district_Hradec_Kralove,
                CountrySubdivision::Czechia_district_Jablonec_nad_Nisou,
                CountrySubdivision::Czechia_district_Jesenik,
                CountrySubdivision::Czechia_district_Jicin,
                CountrySubdivision::Czechia_district_Jihlava,
                CountrySubdivision::Czechia_district_Jindrichuv_Hradec,
                CountrySubdivision::Czechia_district_Karlovy_Vary,
                CountrySubdivision::Czechia_district_Karvina,
                CountrySubdivision::Czechia_district_Kladno,
                CountrySubdivision::Czechia_district_Klatovy,
                CountrySubdivision::Czechia_district_Kolin,
                CountrySubdivision::Czechia_district_Kromeriz,
                CountrySubdivision::Czechia_district_Kutna_Hora,
                CountrySubdivision::Czechia_district_Liberec,
                CountrySubdivision::Czechia_district_Litomerice,
                CountrySubdivision::Czechia_district_Louny,
                CountrySubdivision::Czechia_district_Melnik,
                CountrySubdivision::Czechia_district_Mlada_Boleslav,
                CountrySubdivision::Czechia_district_Most,
                CountrySubdivision::Czechia_district_Nachod,
                CountrySubdivision::Czechia_district_Novy_Jicin,
                CountrySubdivision::Czechia_district_Nymburk,
                CountrySubdivision::Czechia_district_Olomouc,
                CountrySubdivision::Czechia_district_Opava,
                CountrySubdivision::Czechia_district_Ostrava_mesto,
                CountrySubdivision::Czechia_district_Pardubice,
                CountrySubdivision::Czechia_district_Pelhrimov,
                CountrySubdivision::Czechia_district_Pisek,
                CountrySubdivision::Czechia_district_Plzen_jih,
                CountrySubdivision::Czechia_district_Plzen_mesto,
                CountrySubdivision::Czechia_district_Plzen_sever,
                CountrySubdivision::Czechia_district_Prachatice,
                CountrySubdivision::Czechia_district_Praha_vychod,
                CountrySubdivision::Czechia_district_Praha_zapad,
                CountrySubdivision::Czechia_district_Prerov,
                CountrySubdivision::Czechia_district_Pribram,
                CountrySubdivision::Czechia_district_Prostejov,
                CountrySubdivision::Czechia_district_Rakovnik,
                CountrySubdivision::Czechia_district_Rokycany,
                CountrySubdivision::Czechia_district_Rychnov_nad_Kneznou,
                CountrySubdivision::Czechia_district_Semily,
                CountrySubdivision::Czechia_district_Sokolov,
                CountrySubdivision::Czechia_district_Strakonice,
                CountrySubdivision::Czechia_district_Sumperk,
                CountrySubdivision::Czechia_district_Svitavy,
                CountrySubdivision::Czechia_district_Tabor,
                CountrySubdivision::Czechia_district_Tachov,
                CountrySubdivision::Czechia_district_Teplice,
                CountrySubdivision::Czechia_district_Trebic,
                CountrySubdivision::Czechia_district_Trutnov,
                CountrySubdivision::Czechia_district_Uherske_Hradiste,
                CountrySubdivision::Czechia_district_Usti_nad_Labem,
                CountrySubdivision::Czechia_district_Usti_nad_Orlici,
                CountrySubdivision::Czechia_district_Vsetin,
                CountrySubdivision::Czechia_district_Vyskov,
                CountrySubdivision::Czechia_district_Zdar_nad_Sazavou,
                CountrySubdivision::Czechia_district_Zlin,
                CountrySubdivision::Czechia_district_Znojmo,
                CountrySubdivision::Czechia_region_Jihocesky_kraj,
                CountrySubdivision::Czechia_region_Jihomoravsky_kraj,
                CountrySubdivision::Czechia_region_Karlovarsky_kraj,
                CountrySubdivision::Czechia_region_Kraj_Vysocina,
                CountrySubdivision::Czechia_region_Kralovehradecky_kraj,
                CountrySubdivision::Czechia_region_Liberecky_kraj,
                CountrySubdivision::Czechia_region_Moravskoslezsky_kraj,
                CountrySubdivision::Czechia_region_Olomoucky_kraj,
                CountrySubdivision::Czechia_region_Pardubicky_kraj,
                CountrySubdivision::Czechia_region_Plzensky_kraj,
                CountrySubdivision::Czechia_region_Stredocesky_kraj,
                CountrySubdivision::Czechia_region_Ustecky_kraj,
                CountrySubdivision::Czechia_region_Zlinsky_kraj
            ],
            self::Denmark => [
                CountrySubdivision::Denmark_region_Hovedstaden,
                CountrySubdivision::Denmark_region_Midtjylland,
                CountrySubdivision::Denmark_region_Nordjylland,
                CountrySubdivision::Denmark_region_Sjaelland,
                CountrySubdivision::Denmark_region_Syddanmark
            ],
            self::Djibouti => [
                CountrySubdivision::Djibouti_city_Djibouti,
                CountrySubdivision::Djibouti_region_Ali_Sabieh,
                CountrySubdivision::Djibouti_region_Arta,
                CountrySubdivision::Djibouti_region_Awbuk,
                CountrySubdivision::Djibouti_region_Dikhil,
                CountrySubdivision::Djibouti_region_Tadjourah
            ],
            self::Dominica => [
                CountrySubdivision::Dominica_parish_Saint_Andrew,
                CountrySubdivision::Dominica_parish_Saint_David,
                CountrySubdivision::Dominica_parish_Saint_George,
                CountrySubdivision::Dominica_parish_Saint_John,
                CountrySubdivision::Dominica_parish_Saint_Joseph,
                CountrySubdivision::Dominica_parish_Saint_Luke,
                CountrySubdivision::Dominica_parish_Saint_Mark,
                CountrySubdivision::Dominica_parish_Saint_Patrick,
                CountrySubdivision::Dominica_parish_Saint_Paul,
                CountrySubdivision::Dominica_parish_Saint_Peter
            ],
            self::Dominican_Republic => [
                CountrySubdivision::Dominican_Republic_district_Distrito_Nacional_Santo_Domingo,
                CountrySubdivision::Dominican_Republic_province_Azua,
                CountrySubdivision::Dominican_Republic_province_Baoruco,
                CountrySubdivision::Dominican_Republic_province_Barahona,
                CountrySubdivision::Dominican_Republic_province_Dajabon,
                CountrySubdivision::Dominican_Republic_province_Duarte,
                CountrySubdivision::Dominican_Republic_province_El_Seibo,
                CountrySubdivision::Dominican_Republic_province_Elias_Pina,
                CountrySubdivision::Dominican_Republic_province_Espaillat,
                CountrySubdivision::Dominican_Republic_province_Hato_Mayor,
                CountrySubdivision::Dominican_Republic_province_Hermanas_Mirabal,
                CountrySubdivision::Dominican_Republic_province_Independencia,
                CountrySubdivision::Dominican_Republic_province_La_Altagracia,
                CountrySubdivision::Dominican_Republic_province_La_Romana,
                CountrySubdivision::Dominican_Republic_province_La_Vega,
                CountrySubdivision::Dominican_Republic_province_Maria_Trinidad_Sanchez,
                CountrySubdivision::Dominican_Republic_province_Monsenor_Nouel,
                CountrySubdivision::Dominican_Republic_province_Monte_Cristi,
                CountrySubdivision::Dominican_Republic_province_Monte_Plata,
                CountrySubdivision::Dominican_Republic_province_Pedernales,
                CountrySubdivision::Dominican_Republic_province_Peravia,
                CountrySubdivision::Dominican_Republic_province_Puerto_Plata,
                CountrySubdivision::Dominican_Republic_province_Samana,
                CountrySubdivision::Dominican_Republic_province_San_Cristobal,
                CountrySubdivision::Dominican_Republic_province_San_Jose_de_Ocoa,
                CountrySubdivision::Dominican_Republic_province_San_Juan,
                CountrySubdivision::Dominican_Republic_province_San_Pedro_de_Macoris,
                CountrySubdivision::Dominican_Republic_province_Sanchez_Ramirez,
                CountrySubdivision::Dominican_Republic_province_Santiago,
                CountrySubdivision::Dominican_Republic_province_Santiago_Rodriguez,
                CountrySubdivision::Dominican_Republic_province_Santo_Domingo,
                CountrySubdivision::Dominican_Republic_province_Valverde,
                CountrySubdivision::Dominican_Republic_region_Cibao_Nordeste,
                CountrySubdivision::Dominican_Republic_region_Cibao_Noroeste,
                CountrySubdivision::Dominican_Republic_region_Cibao_Norte,
                CountrySubdivision::Dominican_Republic_region_Cibao_Sur,
                CountrySubdivision::Dominican_Republic_region_El_Valle,
                CountrySubdivision::Dominican_Republic_region_Enriquillo,
                CountrySubdivision::Dominican_Republic_region_Higuamo,
                CountrySubdivision::Dominican_Republic_region_Ozama,
                CountrySubdivision::Dominican_Republic_region_Valdesia,
                CountrySubdivision::Dominican_Republic_region_Yuma
            ],
            self::Ecuador => [
                CountrySubdivision::Ecuador_province_Azuay,
                CountrySubdivision::Ecuador_province_Bolivar,
                CountrySubdivision::Ecuador_province_Canar,
                CountrySubdivision::Ecuador_province_Carchi,
                CountrySubdivision::Ecuador_province_Chimborazo,
                CountrySubdivision::Ecuador_province_Cotopaxi,
                CountrySubdivision::Ecuador_province_El_Oro,
                CountrySubdivision::Ecuador_province_Esmeraldas,
                CountrySubdivision::Ecuador_province_Galapagos,
                CountrySubdivision::Ecuador_province_Guayas,
                CountrySubdivision::Ecuador_province_Imbabura,
                CountrySubdivision::Ecuador_province_Loja,
                CountrySubdivision::Ecuador_province_Los_Rios,
                CountrySubdivision::Ecuador_province_Manabi,
                CountrySubdivision::Ecuador_province_Morona_Santiago,
                CountrySubdivision::Ecuador_province_Napo,
                CountrySubdivision::Ecuador_province_Orellana,
                CountrySubdivision::Ecuador_province_Pastaza,
                CountrySubdivision::Ecuador_province_Pichincha,
                CountrySubdivision::Ecuador_province_Santa_Elena,
                CountrySubdivision::Ecuador_province_Santo_Domingo_de_los_Tsachilas,
                CountrySubdivision::Ecuador_province_Sucumbios,
                CountrySubdivision::Ecuador_province_Tungurahua,
                CountrySubdivision::Ecuador_province_Zamora_Chinchipe
            ],
            self::Egypt => [
                CountrySubdivision::Egypt_governorate_Ad_Daqahliyah,
                CountrySubdivision::Egypt_governorate_Al_Bahr_al_Ahmar,
                CountrySubdivision::Egypt_governorate_Al_Buhayrah,
                CountrySubdivision::Egypt_governorate_Al_Fayyum,
                CountrySubdivision::Egypt_governorate_Al_Gharbiyah,
                CountrySubdivision::Egypt_governorate_Al_Iskandariyah,
                CountrySubdivision::Egypt_governorate_Al_Isma_iliyah,
                CountrySubdivision::Egypt_governorate_Al_Jizah,
                CountrySubdivision::Egypt_governorate_Al_Minufiyah,
                CountrySubdivision::Egypt_governorate_Al_Minya,
                CountrySubdivision::Egypt_governorate_Al_Qahirah,
                CountrySubdivision::Egypt_governorate_Al_Qalyubiyah,
                CountrySubdivision::Egypt_governorate_Al_Uqsur,
                CountrySubdivision::Egypt_governorate_Al_Wadi_al_Jadid,
                CountrySubdivision::Egypt_governorate_As_Suways,
                CountrySubdivision::Egypt_governorate_Ash_Sharqiyah,
                CountrySubdivision::Egypt_governorate_Aswan,
                CountrySubdivision::Egypt_governorate_Asyut,
                CountrySubdivision::Egypt_governorate_Bani_Suwayf,
                CountrySubdivision::Egypt_governorate_Bur_Sa_id,
                CountrySubdivision::Egypt_governorate_Dumyat,
                CountrySubdivision::Egypt_governorate_Janub_Sina,
                CountrySubdivision::Egypt_governorate_Kafr_ash_Shaykh,
                CountrySubdivision::Egypt_governorate_Matruh,
                CountrySubdivision::Egypt_governorate_Qina,
                CountrySubdivision::Egypt_governorate_Shamal_Sina,
                CountrySubdivision::Egypt_governorate_Suhaj
            ],
            self::El_Salvador => [
                CountrySubdivision::El_Salvador_department_Ahuachapan,
                CountrySubdivision::El_Salvador_department_Cabanas,
                CountrySubdivision::El_Salvador_department_Chalatenango,
                CountrySubdivision::El_Salvador_department_Cuscatlan,
                CountrySubdivision::El_Salvador_department_La_Libertad,
                CountrySubdivision::El_Salvador_department_La_Paz,
                CountrySubdivision::El_Salvador_department_La_Union,
                CountrySubdivision::El_Salvador_department_Morazan,
                CountrySubdivision::El_Salvador_department_San_Miguel,
                CountrySubdivision::El_Salvador_department_San_Salvador,
                CountrySubdivision::El_Salvador_department_San_Vicente,
                CountrySubdivision::El_Salvador_department_Santa_Ana,
                CountrySubdivision::El_Salvador_department_Sonsonate,
                CountrySubdivision::El_Salvador_department_Usulutan
            ],
            self::Equatorial_Guinea => [
                CountrySubdivision::Equatorial_Guinea_province_Annobon,
                CountrySubdivision::Equatorial_Guinea_province_Bioko_Nord,
                CountrySubdivision::Equatorial_Guinea_province_Bioko_Sud,
                CountrySubdivision::Equatorial_Guinea_province_Centro_Sud,
                CountrySubdivision::Equatorial_Guinea_province_Djibloho,
                CountrySubdivision::Equatorial_Guinea_province_Kie_Ntem,
                CountrySubdivision::Equatorial_Guinea_province_Litoral,
                CountrySubdivision::Equatorial_Guinea_province_Wele_Nzas,
                CountrySubdivision::Equatorial_Guinea_region_Regiao_Continental,
                CountrySubdivision::Equatorial_Guinea_region_Regiao_Insular
            ],
            self::Eritrea => [
                CountrySubdivision::Eritrea_region_Al_Awsat,
                CountrySubdivision::Eritrea_region_Al_Janubi,
                CountrySubdivision::Eritrea_region_Ansaba,
                CountrySubdivision::Eritrea_region_Debubawi_K_eyyih_Bahri,
                CountrySubdivision::Eritrea_region_Gash_Barka,
                CountrySubdivision::Eritrea_region_Semienawi_K_eyyih_Bahri
            ],
            self::Estonia => [
                CountrySubdivision::Estonia_county_Harjumaa,
                CountrySubdivision::Estonia_county_Hiiumaa,
                CountrySubdivision::Estonia_county_Ida_Virumaa,
                CountrySubdivision::Estonia_county_Jarvamaa,
                CountrySubdivision::Estonia_county_Jogevamaa,
                CountrySubdivision::Estonia_county_Laane_Virumaa,
                CountrySubdivision::Estonia_county_Laanemaa,
                CountrySubdivision::Estonia_county_Parnumaa,
                CountrySubdivision::Estonia_county_Polvamaa,
                CountrySubdivision::Estonia_county_Raplamaa,
                CountrySubdivision::Estonia_county_Saaremaa,
                CountrySubdivision::Estonia_county_Tartumaa,
                CountrySubdivision::Estonia_county_Valgamaa,
                CountrySubdivision::Estonia_county_Viljandimaa,
                CountrySubdivision::Estonia_county_Vorumaa,
                CountrySubdivision::Estonia_rural_municipality_Alutaguse,
                CountrySubdivision::Estonia_rural_municipality_Anija,
                CountrySubdivision::Estonia_rural_municipality_Antsla,
                CountrySubdivision::Estonia_rural_municipality_Elva,
                CountrySubdivision::Estonia_rural_municipality_Haademeeste,
                CountrySubdivision::Estonia_rural_municipality_Haljala,
                CountrySubdivision::Estonia_rural_municipality_Harku,
                CountrySubdivision::Estonia_rural_municipality_Hiiumaa,
                CountrySubdivision::Estonia_rural_municipality_Jarva,
                CountrySubdivision::Estonia_rural_municipality_Joelahtme,
                CountrySubdivision::Estonia_rural_municipality_Jogeva,
                CountrySubdivision::Estonia_rural_municipality_Johvi,
                CountrySubdivision::Estonia_rural_municipality_Kadrina,
                CountrySubdivision::Estonia_rural_municipality_Kambja,
                CountrySubdivision::Estonia_rural_municipality_Kanepi,
                CountrySubdivision::Estonia_rural_municipality_Kastre,
                CountrySubdivision::Estonia_rural_municipality_Kehtna,
                CountrySubdivision::Estonia_rural_municipality_Kihnu,
                CountrySubdivision::Estonia_rural_municipality_Kiili,
                CountrySubdivision::Estonia_rural_municipality_Kohila,
                CountrySubdivision::Estonia_rural_municipality_Kose,
                CountrySubdivision::Estonia_rural_municipality_Kuusalu,
                CountrySubdivision::Estonia_rural_municipality_Laane_Harju,
                CountrySubdivision::Estonia_rural_municipality_Laane_Nigula,
                CountrySubdivision::Estonia_rural_municipality_Laaneranna,
                CountrySubdivision::Estonia_rural_municipality_Luganuse,
                CountrySubdivision::Estonia_rural_municipality_Luunja,
                CountrySubdivision::Estonia_rural_municipality_Marjamaa,
                CountrySubdivision::Estonia_rural_municipality_Muhu,
                CountrySubdivision::Estonia_rural_municipality_Mulgi,
                CountrySubdivision::Estonia_rural_municipality_Mustvee,
                CountrySubdivision::Estonia_rural_municipality_Noo,
                CountrySubdivision::Estonia_rural_municipality_Otepaa,
                CountrySubdivision::Estonia_rural_municipality_Peipsiaare,
                CountrySubdivision::Estonia_rural_municipality_Pohja_Parnumaa,
                CountrySubdivision::Estonia_rural_municipality_Pohja_Sakala,
                CountrySubdivision::Estonia_rural_municipality_Poltsamaa,
                CountrySubdivision::Estonia_rural_municipality_Polva,
                CountrySubdivision::Estonia_rural_municipality_Raasiku,
                CountrySubdivision::Estonia_rural_municipality_Rae,
                CountrySubdivision::Estonia_rural_municipality_Rakvere,
                CountrySubdivision::Estonia_rural_municipality_Rapina,
                CountrySubdivision::Estonia_rural_municipality_Rapla,
                CountrySubdivision::Estonia_rural_municipality_Rouge,
                CountrySubdivision::Estonia_rural_municipality_Ruhnu,
                CountrySubdivision::Estonia_rural_municipality_Saarde,
                CountrySubdivision::Estonia_rural_municipality_Saaremaa,
                CountrySubdivision::Estonia_rural_municipality_Saku,
                CountrySubdivision::Estonia_rural_municipality_Saue,
                CountrySubdivision::Estonia_rural_municipality_Setomaa,
                CountrySubdivision::Estonia_rural_municipality_Tapa,
                CountrySubdivision::Estonia_rural_municipality_Tartu,
                CountrySubdivision::Estonia_rural_municipality_Toila,
                CountrySubdivision::Estonia_rural_municipality_Tori,
                CountrySubdivision::Estonia_rural_municipality_Torva,
                CountrySubdivision::Estonia_rural_municipality_Turi,
                CountrySubdivision::Estonia_rural_municipality_Vaike_Maarja,
                CountrySubdivision::Estonia_rural_municipality_Valga,
                CountrySubdivision::Estonia_rural_municipality_Viimsi,
                CountrySubdivision::Estonia_rural_municipality_Viljandi,
                CountrySubdivision::Estonia_rural_municipality_Vinni,
                CountrySubdivision::Estonia_rural_municipality_Viru_Nigula,
                CountrySubdivision::Estonia_rural_municipality_Vormsi,
                CountrySubdivision::Estonia_rural_municipality_Voru,
                CountrySubdivision::Estonia_urban_municipality_Haapsalu,
                CountrySubdivision::Estonia_urban_municipality_Keila,
                CountrySubdivision::Estonia_urban_municipality_Kohtla_Jarve,
                CountrySubdivision::Estonia_urban_municipality_Loksa,
                CountrySubdivision::Estonia_urban_municipality_Maardu,
                CountrySubdivision::Estonia_urban_municipality_Narva,
                CountrySubdivision::Estonia_urban_municipality_Narva_Joesuu,
                CountrySubdivision::Estonia_urban_municipality_Paide,
                CountrySubdivision::Estonia_urban_municipality_Parnu,
                CountrySubdivision::Estonia_urban_municipality_Rakvere,
                CountrySubdivision::Estonia_urban_municipality_Sillamae,
                CountrySubdivision::Estonia_urban_municipality_Tallinn,
                CountrySubdivision::Estonia_urban_municipality_Tartu,
                CountrySubdivision::Estonia_urban_municipality_Viljandi,
                CountrySubdivision::Estonia_urban_municipality_Voru
            ],
            self::Eswatini => [
                CountrySubdivision::Eswatini_region_Hhohho,
                CountrySubdivision::Eswatini_region_Lubombo,
                CountrySubdivision::Eswatini_region_Manzini,
                CountrySubdivision::Eswatini_region_Shiselweni
            ],
            self::Ethiopia => [
                CountrySubdivision::Ethiopia_administration_Addis_Ababa,
                CountrySubdivision::Ethiopia_administration_Dire_Dawa,
                CountrySubdivision::Ethiopia_regional_state_Afar,
                CountrySubdivision::Ethiopia_regional_state_Amara,
                CountrySubdivision::Ethiopia_regional_state_Benshangul_Gumaz,
                CountrySubdivision::Ethiopia_regional_state_Gambela_Peoples,
                CountrySubdivision::Ethiopia_regional_state_Harari_People,
                CountrySubdivision::Ethiopia_regional_state_Oromia,
                CountrySubdivision::Ethiopia_regional_state_Sidama,
                CountrySubdivision::Ethiopia_regional_state_Somali,
                CountrySubdivision::Ethiopia_regional_state_Southern_Nations_Nationalities_and_Peoples,
                CountrySubdivision::Ethiopia_regional_state_Southwest_Ethiopia_Peoples,
                CountrySubdivision::Ethiopia_regional_state_Tigrai
            ],
            self::Fiji => [
                CountrySubdivision::Fiji_dependency_Rotuma,
                CountrySubdivision::Fiji_division_Central,
                CountrySubdivision::Fiji_division_Eastern,
                CountrySubdivision::Fiji_division_Northern,
                CountrySubdivision::Fiji_division_Western,
                CountrySubdivision::Fiji_province_Ba,
                CountrySubdivision::Fiji_province_Bua,
                CountrySubdivision::Fiji_province_Cakaudrove,
                CountrySubdivision::Fiji_province_Kadavu,
                CountrySubdivision::Fiji_province_Lau,
                CountrySubdivision::Fiji_province_Lomaiviti,
                CountrySubdivision::Fiji_province_Macuata,
                CountrySubdivision::Fiji_province_Nadroga_and_Navosa,
                CountrySubdivision::Fiji_province_Naitasiri,
                CountrySubdivision::Fiji_province_Namosi,
                CountrySubdivision::Fiji_province_Ra,
                CountrySubdivision::Fiji_province_Rewa,
                CountrySubdivision::Fiji_province_Serua,
                CountrySubdivision::Fiji_province_Tailevu
            ],
            self::Finland => [
                CountrySubdivision::Finland_region_Ahvenanmaan_maakunta,
                CountrySubdivision::Finland_region_Birkaland,
                CountrySubdivision::Finland_region_Egentliga_Finland,
                CountrySubdivision::Finland_region_Egentliga_Tavastland,
                CountrySubdivision::Finland_region_Etela_Karjala,
                CountrySubdivision::Finland_region_Etela_Pohjanmaa,
                CountrySubdivision::Finland_region_Etela_Savo,
                CountrySubdivision::Finland_region_Kainuu,
                CountrySubdivision::Finland_region_Keski_Pohjanmaa,
                CountrySubdivision::Finland_region_Keski_Suomi,
                CountrySubdivision::Finland_region_Kymenlaakso,
                CountrySubdivision::Finland_region_Lappi,
                CountrySubdivision::Finland_region_Norra_Karelen,
                CountrySubdivision::Finland_region_Norra_Osterbotten,
                CountrySubdivision::Finland_region_Norra_Savolax,
                CountrySubdivision::Finland_region_Nyland,
                CountrySubdivision::Finland_region_Paijanne_Tavastland,
                CountrySubdivision::Finland_region_Pohjanmaa,
                CountrySubdivision::Finland_region_Satakunta
            ],
            self::France => [
                CountrySubdivision::France_European_collectivity_Alsace,
                CountrySubdivision::France_dependency_Clipperton,
                CountrySubdivision::France_metropolitan_collectivity_with_special_status_Corse,
                CountrySubdivision::France_metropolitan_collectivity_with_special_status_Metropole_de_Lyon,
                CountrySubdivision::France_metropolitan_collectivity_with_special_status_Paris,
                CountrySubdivision::France_metropolitan_department_Ain,
                CountrySubdivision::France_metropolitan_department_Aisne,
                CountrySubdivision::France_metropolitan_department_Allier,
                CountrySubdivision::France_metropolitan_department_Alpes_Maritimes,
                CountrySubdivision::France_metropolitan_department_Alpes_de_Haute_Provence,
                CountrySubdivision::France_metropolitan_department_Ardeche,
                CountrySubdivision::France_metropolitan_department_Ardennes,
                CountrySubdivision::France_metropolitan_department_Ariege,
                CountrySubdivision::France_metropolitan_department_Aube,
                CountrySubdivision::France_metropolitan_department_Aude,
                CountrySubdivision::France_metropolitan_department_Aveyron,
                CountrySubdivision::France_metropolitan_department_Bas_Rhin,
                CountrySubdivision::France_metropolitan_department_Bouches_du_Rhone,
                CountrySubdivision::France_metropolitan_department_Calvados,
                CountrySubdivision::France_metropolitan_department_Cantal,
                CountrySubdivision::France_metropolitan_department_Charente,
                CountrySubdivision::France_metropolitan_department_Charente_Maritime,
                CountrySubdivision::France_metropolitan_department_Cher,
                CountrySubdivision::France_metropolitan_department_Correze,
                CountrySubdivision::France_metropolitan_department_Corse_du_Sud,
                CountrySubdivision::France_metropolitan_department_Cote_d_Or,
                CountrySubdivision::France_metropolitan_department_Cotes_d_Armor,
                CountrySubdivision::France_metropolitan_department_Creuse,
                CountrySubdivision::France_metropolitan_department_Deux_Sevres,
                CountrySubdivision::France_metropolitan_department_Dordogne,
                CountrySubdivision::France_metropolitan_department_Doubs,
                CountrySubdivision::France_metropolitan_department_Drome,
                CountrySubdivision::France_metropolitan_department_Essonne,
                CountrySubdivision::France_metropolitan_department_Eure,
                CountrySubdivision::France_metropolitan_department_Eure_et_Loir,
                CountrySubdivision::France_metropolitan_department_Finistere,
                CountrySubdivision::France_metropolitan_department_Gard,
                CountrySubdivision::France_metropolitan_department_Gers,
                CountrySubdivision::France_metropolitan_department_Gironde,
                CountrySubdivision::France_metropolitan_department_Haut_Rhin,
                CountrySubdivision::France_metropolitan_department_Haute_Corse,
                CountrySubdivision::France_metropolitan_department_Haute_Garonne,
                CountrySubdivision::France_metropolitan_department_Haute_Loire,
                CountrySubdivision::France_metropolitan_department_Haute_Marne,
                CountrySubdivision::France_metropolitan_department_Haute_Saone,
                CountrySubdivision::France_metropolitan_department_Haute_Savoie,
                CountrySubdivision::France_metropolitan_department_Haute_Vienne,
                CountrySubdivision::France_metropolitan_department_Hautes_Alpes,
                CountrySubdivision::France_metropolitan_department_Hautes_Pyrenees,
                CountrySubdivision::France_metropolitan_department_Hauts_de_Seine,
                CountrySubdivision::France_metropolitan_department_Herault,
                CountrySubdivision::France_metropolitan_department_Ille_et_Vilaine,
                CountrySubdivision::France_metropolitan_department_Indre,
                CountrySubdivision::France_metropolitan_department_Indre_et_Loire,
                CountrySubdivision::France_metropolitan_department_Isere,
                CountrySubdivision::France_metropolitan_department_Jura,
                CountrySubdivision::France_metropolitan_department_Landes,
                CountrySubdivision::France_metropolitan_department_Loir_et_Cher,
                CountrySubdivision::France_metropolitan_department_Loire,
                CountrySubdivision::France_metropolitan_department_Loire_Atlantique,
                CountrySubdivision::France_metropolitan_department_Loiret,
                CountrySubdivision::France_metropolitan_department_Lot,
                CountrySubdivision::France_metropolitan_department_Lot_et_Garonne,
                CountrySubdivision::France_metropolitan_department_Lozere,
                CountrySubdivision::France_metropolitan_department_Maine_et_Loire,
                CountrySubdivision::France_metropolitan_department_Manche,
                CountrySubdivision::France_metropolitan_department_Marne,
                CountrySubdivision::France_metropolitan_department_Mayenne,
                CountrySubdivision::France_metropolitan_department_Meurthe_et_Moselle,
                CountrySubdivision::France_metropolitan_department_Meuse,
                CountrySubdivision::France_metropolitan_department_Morbihan,
                CountrySubdivision::France_metropolitan_department_Moselle,
                CountrySubdivision::France_metropolitan_department_Nievre,
                CountrySubdivision::France_metropolitan_department_Nord,
                CountrySubdivision::France_metropolitan_department_Oise,
                CountrySubdivision::France_metropolitan_department_Orne,
                CountrySubdivision::France_metropolitan_department_Pas_de_Calais,
                CountrySubdivision::France_metropolitan_department_Puy_de_Dome,
                CountrySubdivision::France_metropolitan_department_Pyrenees_Atlantiques,
                CountrySubdivision::France_metropolitan_department_Pyrenees_Orientales,
                CountrySubdivision::France_metropolitan_department_Rhone,
                CountrySubdivision::France_metropolitan_department_Saone_et_Loire,
                CountrySubdivision::France_metropolitan_department_Sarthe,
                CountrySubdivision::France_metropolitan_department_Savoie,
                CountrySubdivision::France_metropolitan_department_Seine_Maritime,
                CountrySubdivision::France_metropolitan_department_Seine_Saint_Denis,
                CountrySubdivision::France_metropolitan_department_Seine_et_Marne,
                CountrySubdivision::France_metropolitan_department_Somme,
                CountrySubdivision::France_metropolitan_department_Tarn,
                CountrySubdivision::France_metropolitan_department_Tarn_et_Garonne,
                CountrySubdivision::France_metropolitan_department_Territoire_de_Belfort,
                CountrySubdivision::France_metropolitan_department_Val_d_Oise,
                CountrySubdivision::France_metropolitan_department_Val_de_Marne,
                CountrySubdivision::France_metropolitan_department_Var,
                CountrySubdivision::France_metropolitan_department_Vaucluse,
                CountrySubdivision::France_metropolitan_department_Vendee,
                CountrySubdivision::France_metropolitan_department_Vienne,
                CountrySubdivision::France_metropolitan_department_Vosges,
                CountrySubdivision::France_metropolitan_department_Yonne,
                CountrySubdivision::France_metropolitan_department_Yvelines,
                CountrySubdivision::France_metropolitan_region_Auvergne_Rhone_Alpes,
                CountrySubdivision::France_metropolitan_region_Bourgogne_Franche_Comte,
                CountrySubdivision::France_metropolitan_region_Bretagne,
                CountrySubdivision::France_metropolitan_region_Centre_Val_de_Loire,
                CountrySubdivision::France_metropolitan_region_Grand_Est,
                CountrySubdivision::France_metropolitan_region_Hauts_de_France,
                CountrySubdivision::France_metropolitan_region_Ile_de_France,
                CountrySubdivision::France_metropolitan_region_Normandie,
                CountrySubdivision::France_metropolitan_region_Nouvelle_Aquitaine,
                CountrySubdivision::France_metropolitan_region_Occitanie,
                CountrySubdivision::France_metropolitan_region_Pays_de_la_Loire,
                CountrySubdivision::France_metropolitan_region_Provence_Alpes_Cote_d_Azur,
                CountrySubdivision::France_overseas_collectivity_Polynesie_francaise,
                CountrySubdivision::France_overseas_collectivity_Saint_Barthelemy,
                CountrySubdivision::France_overseas_collectivity_Saint_Martin,
                CountrySubdivision::France_overseas_collectivity_Saint_Pierre_et_Miquelon,
                CountrySubdivision::France_overseas_collectivity_Wallis_et_Futuna,
                CountrySubdivision::France_overseas_collectivity_with_special_status_Nouvelle_Caledonie,
                CountrySubdivision::France_overseas_departmental_collectivity_Guadeloupe,
                CountrySubdivision::France_overseas_departmental_collectivity_La_Reunion,
                CountrySubdivision::France_overseas_departmental_collectivity_Mayotte,
                CountrySubdivision::France_overseas_territory_Terres_australes_francaises,
                CountrySubdivision::France_overseas_unique_territorial_collectivity_Guyane_francaise,
                CountrySubdivision::France_overseas_unique_territorial_collectivity_Martinique
            ],
            self::Gabon => [
                CountrySubdivision::Gabon_province_Estuaire,
                CountrySubdivision::Gabon_province_Haut_Ogooue,
                CountrySubdivision::Gabon_province_Moyen_Ogooue,
                CountrySubdivision::Gabon_province_Ngounie,
                CountrySubdivision::Gabon_province_Nyanga,
                CountrySubdivision::Gabon_province_Ogooue_Ivindo,
                CountrySubdivision::Gabon_province_Ogooue_Lolo,
                CountrySubdivision::Gabon_province_Ogooue_Maritime,
                CountrySubdivision::Gabon_province_Woleu_Ntem
            ],
            self::Gambia => [
                CountrySubdivision::Gambia_city_Banjul,
                CountrySubdivision::Gambia_division_Central_River,
                CountrySubdivision::Gambia_division_Lower_River,
                CountrySubdivision::Gambia_division_North_Bank,
                CountrySubdivision::Gambia_division_Upper_River,
                CountrySubdivision::Gambia_division_Western
            ],
            self::Georgia => [
                CountrySubdivision::Georgia_autonomous_republic_Abkhazia,
                CountrySubdivision::Georgia_autonomous_republic_Ajaria,
                CountrySubdivision::Georgia_city_Tbilisi,
                CountrySubdivision::Georgia_region_Guria,
                CountrySubdivision::Georgia_region_Imereti,
                CountrySubdivision::Georgia_region_K_akheti,
                CountrySubdivision::Georgia_region_Kvemo_Kartli,
                CountrySubdivision::Georgia_region_Mtskheta_Mtianeti,
                CountrySubdivision::Georgia_region_Rach_a_Lechkhumi_Kvemo_Svaneti,
                CountrySubdivision::Georgia_region_Samegrelo_Zemo_Svaneti,
                CountrySubdivision::Georgia_region_Samtskhe_Javakheti,
                CountrySubdivision::Georgia_region_Shida_Kartli
            ],
            self::Germany => [
                CountrySubdivision::Germany_Land_Baden_Wurttemberg,
                CountrySubdivision::Germany_Land_Bayern,
                CountrySubdivision::Germany_Land_Berlin,
                CountrySubdivision::Germany_Land_Brandenburg,
                CountrySubdivision::Germany_Land_Bremen,
                CountrySubdivision::Germany_Land_Hamburg,
                CountrySubdivision::Germany_Land_Hessen,
                CountrySubdivision::Germany_Land_Mecklenburg_Vorpommern,
                CountrySubdivision::Germany_Land_Niedersachsen,
                CountrySubdivision::Germany_Land_Nordrhein_Westfalen,
                CountrySubdivision::Germany_Land_Rheinland_Pfalz,
                CountrySubdivision::Germany_Land_Saarland,
                CountrySubdivision::Germany_Land_Sachsen,
                CountrySubdivision::Germany_Land_Sachsen_Anhalt,
                CountrySubdivision::Germany_Land_Schleswig_Holstein,
                CountrySubdivision::Germany_Land_Thuringen
            ],
            self::Ghana => [
                CountrySubdivision::Ghana_region_Ahafo,
                CountrySubdivision::Ghana_region_Ashanti,
                CountrySubdivision::Ghana_region_Bono,
                CountrySubdivision::Ghana_region_Bono_East,
                CountrySubdivision::Ghana_region_Central,
                CountrySubdivision::Ghana_region_Eastern,
                CountrySubdivision::Ghana_region_Greater_Accra,
                CountrySubdivision::Ghana_region_North_East,
                CountrySubdivision::Ghana_region_Northern,
                CountrySubdivision::Ghana_region_Oti,
                CountrySubdivision::Ghana_region_Savannah,
                CountrySubdivision::Ghana_region_Upper_East,
                CountrySubdivision::Ghana_region_Upper_West,
                CountrySubdivision::Ghana_region_Volta,
                CountrySubdivision::Ghana_region_Western,
                CountrySubdivision::Ghana_region_Western_North
            ],
            self::Greece => [
                CountrySubdivision::Greece_administrative_region_Anatoliki_Makedonia_kai_Thraki,
                CountrySubdivision::Greece_administrative_region_Attiki,
                CountrySubdivision::Greece_administrative_region_Dytiki_Ellada,
                CountrySubdivision::Greece_administrative_region_Dytiki_Makedonia,
                CountrySubdivision::Greece_administrative_region_Ionia_Nisia,
                CountrySubdivision::Greece_administrative_region_Ipeiros,
                CountrySubdivision::Greece_administrative_region_Kentriki_Makedonia,
                CountrySubdivision::Greece_administrative_region_Kriti,
                CountrySubdivision::Greece_administrative_region_Notio_Aigaio,
                CountrySubdivision::Greece_administrative_region_Peloponnisos,
                CountrySubdivision::Greece_administrative_region_Sterea_Ellada,
                CountrySubdivision::Greece_administrative_region_Thessalia,
                CountrySubdivision::Greece_administrative_region_Voreio_Aigaio,
                CountrySubdivision::Greece_self_governed_part_Agion_Oros
            ],
            self::Greenland => [
                CountrySubdivision::Greenland_municipality_Avannaata_Kommunia,
                CountrySubdivision::Greenland_municipality_Kommune_Kujalleq,
                CountrySubdivision::Greenland_municipality_Kommune_Qeqertalik,
                CountrySubdivision::Greenland_municipality_Kommuneqarfik_Sermersooq,
                CountrySubdivision::Greenland_municipality_Qeqqata_Kommunia
            ],
            self::Grenada => [
                CountrySubdivision::Grenada_dependency_Southern_Grenadine_Islands,
                CountrySubdivision::Grenada_parish_Saint_Andrew,
                CountrySubdivision::Grenada_parish_Saint_David,
                CountrySubdivision::Grenada_parish_Saint_George,
                CountrySubdivision::Grenada_parish_Saint_John,
                CountrySubdivision::Grenada_parish_Saint_Mark,
                CountrySubdivision::Grenada_parish_Saint_Patrick
            ],
            self::Guatemala => [
                CountrySubdivision::Guatemala_department_Alta_Verapaz,
                CountrySubdivision::Guatemala_department_Baja_Verapaz,
                CountrySubdivision::Guatemala_department_Chimaltenango,
                CountrySubdivision::Guatemala_department_Chiquimula,
                CountrySubdivision::Guatemala_department_El_Progreso,
                CountrySubdivision::Guatemala_department_Escuintla,
                CountrySubdivision::Guatemala_department_Guatemala,
                CountrySubdivision::Guatemala_department_Huehuetenango,
                CountrySubdivision::Guatemala_department_Izabal,
                CountrySubdivision::Guatemala_department_Jalapa,
                CountrySubdivision::Guatemala_department_Jutiapa,
                CountrySubdivision::Guatemala_department_Peten,
                CountrySubdivision::Guatemala_department_Quetzaltenango,
                CountrySubdivision::Guatemala_department_Quiche,
                CountrySubdivision::Guatemala_department_Retalhuleu,
                CountrySubdivision::Guatemala_department_Sacatepequez,
                CountrySubdivision::Guatemala_department_San_Marcos,
                CountrySubdivision::Guatemala_department_Santa_Rosa,
                CountrySubdivision::Guatemala_department_Solola,
                CountrySubdivision::Guatemala_department_Suchitepequez,
                CountrySubdivision::Guatemala_department_Totonicapan,
                CountrySubdivision::Guatemala_department_Zacapa
            ],
            self::Guinea => [
                CountrySubdivision::Guinea_administrative_region_Boke,
                CountrySubdivision::Guinea_administrative_region_Faranah,
                CountrySubdivision::Guinea_administrative_region_Kankan,
                CountrySubdivision::Guinea_administrative_region_Kindia,
                CountrySubdivision::Guinea_administrative_region_Labe,
                CountrySubdivision::Guinea_administrative_region_Mamou,
                CountrySubdivision::Guinea_administrative_region_Nzerekore,
                CountrySubdivision::Guinea_governorate_Conakry,
                CountrySubdivision::Guinea_prefecture_Beyla,
                CountrySubdivision::Guinea_prefecture_Boffa,
                CountrySubdivision::Guinea_prefecture_Boke,
                CountrySubdivision::Guinea_prefecture_Coyah,
                CountrySubdivision::Guinea_prefecture_Dabola,
                CountrySubdivision::Guinea_prefecture_Dalaba,
                CountrySubdivision::Guinea_prefecture_Dinguiraye,
                CountrySubdivision::Guinea_prefecture_Dubreka,
                CountrySubdivision::Guinea_prefecture_Faranah,
                CountrySubdivision::Guinea_prefecture_Forecariah,
                CountrySubdivision::Guinea_prefecture_Fria,
                CountrySubdivision::Guinea_prefecture_Gaoual,
                CountrySubdivision::Guinea_prefecture_Guekedou,
                CountrySubdivision::Guinea_prefecture_Kankan,
                CountrySubdivision::Guinea_prefecture_Kerouane,
                CountrySubdivision::Guinea_prefecture_Kindia,
                CountrySubdivision::Guinea_prefecture_Kissidougou,
                CountrySubdivision::Guinea_prefecture_Koubia,
                CountrySubdivision::Guinea_prefecture_Koundara,
                CountrySubdivision::Guinea_prefecture_Kouroussa,
                CountrySubdivision::Guinea_prefecture_Labe,
                CountrySubdivision::Guinea_prefecture_Lelouma,
                CountrySubdivision::Guinea_prefecture_Lola,
                CountrySubdivision::Guinea_prefecture_Macenta,
                CountrySubdivision::Guinea_prefecture_Mali,
                CountrySubdivision::Guinea_prefecture_Mamou,
                CountrySubdivision::Guinea_prefecture_Mandiana,
                CountrySubdivision::Guinea_prefecture_Nzerekore,
                CountrySubdivision::Guinea_prefecture_Pita,
                CountrySubdivision::Guinea_prefecture_Siguiri,
                CountrySubdivision::Guinea_prefecture_Telimele,
                CountrySubdivision::Guinea_prefecture_Tougue,
                CountrySubdivision::Guinea_prefecture_Yomou
            ],
            self::Guinea_Bissau => [
                CountrySubdivision::Guinea_Bissau_autonomous_sector_Bissau,
                CountrySubdivision::Guinea_Bissau_province_Leste,
                CountrySubdivision::Guinea_Bissau_province_Norte,
                CountrySubdivision::Guinea_Bissau_province_Sul,
                CountrySubdivision::Guinea_Bissau_region_Bafata,
                CountrySubdivision::Guinea_Bissau_region_Biombo,
                CountrySubdivision::Guinea_Bissau_region_Bolama_Bijagos,
                CountrySubdivision::Guinea_Bissau_region_Cacheu,
                CountrySubdivision::Guinea_Bissau_region_Gabu,
                CountrySubdivision::Guinea_Bissau_region_Oio,
                CountrySubdivision::Guinea_Bissau_region_Quinara,
                CountrySubdivision::Guinea_Bissau_region_Tombali
            ],
            self::Guyana => [
                CountrySubdivision::Guyana_region_Barima_Waini,
                CountrySubdivision::Guyana_region_Cuyuni_Mazaruni,
                CountrySubdivision::Guyana_region_Demerara_Mahaica,
                CountrySubdivision::Guyana_region_East_Berbice_Corentyne,
                CountrySubdivision::Guyana_region_Essequibo_Islands_West_Demerara,
                CountrySubdivision::Guyana_region_Mahaica_Berbice,
                CountrySubdivision::Guyana_region_Pomeroon_Supenaam,
                CountrySubdivision::Guyana_region_Potaro_Siparuni,
                CountrySubdivision::Guyana_region_Upper_Demerara_Berbice,
                CountrySubdivision::Guyana_region_Upper_Takutu_Upper_Essequibo
            ],
            self::Haiti => [
                CountrySubdivision::Haiti_department_Artibonite,
                CountrySubdivision::Haiti_department_Centre,
                CountrySubdivision::Haiti_department_Grandans,
                CountrySubdivision::Haiti_department_Lwes,
                CountrySubdivision::Haiti_department_Nip,
                CountrySubdivision::Haiti_department_Nord,
                CountrySubdivision::Haiti_department_Nord_Est,
                CountrySubdivision::Haiti_department_Nord_Ouest,
                CountrySubdivision::Haiti_department_Sid,
                CountrySubdivision::Haiti_department_Sides
            ],
            self::Honduras => [
                CountrySubdivision::Honduras_department_Atlantida,
                CountrySubdivision::Honduras_department_Choluteca,
                CountrySubdivision::Honduras_department_Colon,
                CountrySubdivision::Honduras_department_Comayagua,
                CountrySubdivision::Honduras_department_Copan,
                CountrySubdivision::Honduras_department_Cortes,
                CountrySubdivision::Honduras_department_El_Paraiso,
                CountrySubdivision::Honduras_department_Francisco_Morazan,
                CountrySubdivision::Honduras_department_Gracias_a_Dios,
                CountrySubdivision::Honduras_department_Intibuca,
                CountrySubdivision::Honduras_department_Islas_de_la_Bahia,
                CountrySubdivision::Honduras_department_La_Paz,
                CountrySubdivision::Honduras_department_Lempira,
                CountrySubdivision::Honduras_department_Ocotepeque,
                CountrySubdivision::Honduras_department_Olancho,
                CountrySubdivision::Honduras_department_Santa_Barbara,
                CountrySubdivision::Honduras_department_Valle,
                CountrySubdivision::Honduras_department_Yoro
            ],
            self::Hungary => [
                CountrySubdivision::Hungary_capital_city_Budapest,
                CountrySubdivision::Hungary_city_with_county_rights_Bekescsaba,
                CountrySubdivision::Hungary_city_with_county_rights_Debrecen,
                CountrySubdivision::Hungary_city_with_county_rights_Dunaujvaros,
                CountrySubdivision::Hungary_city_with_county_rights_Eger,
                CountrySubdivision::Hungary_city_with_county_rights_Erd,
                CountrySubdivision::Hungary_city_with_county_rights_Gyor,
                CountrySubdivision::Hungary_city_with_county_rights_Hodmezovasarhely,
                CountrySubdivision::Hungary_city_with_county_rights_Kaposvar,
                CountrySubdivision::Hungary_city_with_county_rights_Kecskemet,
                CountrySubdivision::Hungary_city_with_county_rights_Miskolc,
                CountrySubdivision::Hungary_city_with_county_rights_Nagykanizsa,
                CountrySubdivision::Hungary_city_with_county_rights_Nyiregyhaza,
                CountrySubdivision::Hungary_city_with_county_rights_Pecs,
                CountrySubdivision::Hungary_city_with_county_rights_Salgotarjan,
                CountrySubdivision::Hungary_city_with_county_rights_Sopron,
                CountrySubdivision::Hungary_city_with_county_rights_Szeged,
                CountrySubdivision::Hungary_city_with_county_rights_Szekesfehervar,
                CountrySubdivision::Hungary_city_with_county_rights_Szekszard,
                CountrySubdivision::Hungary_city_with_county_rights_Szolnok,
                CountrySubdivision::Hungary_city_with_county_rights_Szombathely,
                CountrySubdivision::Hungary_city_with_county_rights_Tatabanya,
                CountrySubdivision::Hungary_city_with_county_rights_Veszprem,
                CountrySubdivision::Hungary_city_with_county_rights_Zalaegerszeg,
                CountrySubdivision::Hungary_county_Bacs_Kiskun,
                CountrySubdivision::Hungary_county_Baranya,
                CountrySubdivision::Hungary_county_Bekes,
                CountrySubdivision::Hungary_county_Borsod_Abauj_Zemplen,
                CountrySubdivision::Hungary_county_Csongrad_Csanad,
                CountrySubdivision::Hungary_county_Fejer,
                CountrySubdivision::Hungary_county_Gyor_Moson_Sopron,
                CountrySubdivision::Hungary_county_Hajdu_Bihar,
                CountrySubdivision::Hungary_county_Heves,
                CountrySubdivision::Hungary_county_Jasz_Nagykun_Szolnok,
                CountrySubdivision::Hungary_county_Komarom_Esztergom,
                CountrySubdivision::Hungary_county_Nograd,
                CountrySubdivision::Hungary_county_Pest,
                CountrySubdivision::Hungary_county_Somogy,
                CountrySubdivision::Hungary_county_Szabolcs_Szatmar_Bereg,
                CountrySubdivision::Hungary_county_Tolna,
                CountrySubdivision::Hungary_county_Vas,
                CountrySubdivision::Hungary_county_Veszprem,
                CountrySubdivision::Hungary_county_Zala
            ],
            self::Iceland => [
                CountrySubdivision::Iceland_municipality_Akraneskaupstadur,
                CountrySubdivision::Iceland_municipality_Akureyrarbaer,
                CountrySubdivision::Iceland_municipality_Arneshreppur,
                CountrySubdivision::Iceland_municipality_Asahreppur,
                CountrySubdivision::Iceland_municipality_Blaskogabyggd,
                CountrySubdivision::Iceland_municipality_Bolungarvikurkaupstadur,
                CountrySubdivision::Iceland_municipality_Borgarbyggd,
                CountrySubdivision::Iceland_municipality_Dalabyggd,
                CountrySubdivision::Iceland_municipality_Dalvikurbyggd,
                CountrySubdivision::Iceland_municipality_Eyja_og_Miklaholtshreppur,
                CountrySubdivision::Iceland_municipality_Eyjafjardarsveit,
                CountrySubdivision::Iceland_municipality_Fjallabyggd,
                CountrySubdivision::Iceland_municipality_Fjardabyggd,
                CountrySubdivision::Iceland_municipality_Fljotsdalshreppur,
                CountrySubdivision::Iceland_municipality_Floahreppur,
                CountrySubdivision::Iceland_municipality_Gardabaer,
                CountrySubdivision::Iceland_municipality_Grimsnes_og_Grafningshreppur,
                CountrySubdivision::Iceland_municipality_Grindavikurbaer,
                CountrySubdivision::Iceland_municipality_Grundarfjardarbaer,
                CountrySubdivision::Iceland_municipality_Grytubakkahreppur,
                CountrySubdivision::Iceland_municipality_Hafnarfjardarkaupstadur,
                CountrySubdivision::Iceland_municipality_Horgarsveit,
                CountrySubdivision::Iceland_municipality_Hrunamannahreppur,
                CountrySubdivision::Iceland_municipality_Hunabyggd,
                CountrySubdivision::Iceland_municipality_Hunathing_vestra,
                CountrySubdivision::Iceland_municipality_Hvalfjardarsveit,
                CountrySubdivision::Iceland_municipality_Hveragerdisbaer,
                CountrySubdivision::Iceland_municipality_Isafjardarbaer,
                CountrySubdivision::Iceland_municipality_Kaldrananeshreppur,
                CountrySubdivision::Iceland_municipality_Kjosarhreppur,
                CountrySubdivision::Iceland_municipality_Kopavogsbaer,
                CountrySubdivision::Iceland_municipality_Langanesbyggd,
                CountrySubdivision::Iceland_municipality_Mosfellsbaer,
                CountrySubdivision::Iceland_municipality_Mulathing,
                CountrySubdivision::Iceland_municipality_Myrdalshreppur,
                CountrySubdivision::Iceland_municipality_Nordurthing,
                CountrySubdivision::Iceland_municipality_Rangarthing_eystra,
                CountrySubdivision::Iceland_municipality_Rangarthing_ytra,
                CountrySubdivision::Iceland_municipality_Reykholahreppur,
                CountrySubdivision::Iceland_municipality_Reykjanesbaer,
                CountrySubdivision::Iceland_municipality_Reykjavikurborg,
                CountrySubdivision::Iceland_municipality_Seltjarnarnesbaer,
                CountrySubdivision::Iceland_municipality_Skaftarhreppur,
                CountrySubdivision::Iceland_municipality_Skagabyggd,
                CountrySubdivision::Iceland_municipality_Skagafjordur,
                CountrySubdivision::Iceland_municipality_Skeida_og_Gnupverjahreppur,
                CountrySubdivision::Iceland_municipality_Skorradalshreppur,
                CountrySubdivision::Iceland_municipality_Snaefellsbaer,
                CountrySubdivision::Iceland_municipality_Strandabyggd,
                CountrySubdivision::Iceland_municipality_Stykkisholmsbaer,
                CountrySubdivision::Iceland_municipality_Sudavikurhreppur,
                CountrySubdivision::Iceland_municipality_Sudurnesjabaer,
                CountrySubdivision::Iceland_municipality_Svalbardsstrandarhreppur,
                CountrySubdivision::Iceland_municipality_Sveitarfelagid_Arborg,
                CountrySubdivision::Iceland_municipality_Sveitarfelagid_Hornafjordur,
                CountrySubdivision::Iceland_municipality_Sveitarfelagid_Olfus,
                CountrySubdivision::Iceland_municipality_Sveitarfelagid_Skagastrond,
                CountrySubdivision::Iceland_municipality_Sveitarfelagid_Vogar,
                CountrySubdivision::Iceland_municipality_THingeyjarsveit,
                CountrySubdivision::Iceland_municipality_Talknafjardarhreppur,
                CountrySubdivision::Iceland_municipality_Tjorneshreppur,
                CountrySubdivision::Iceland_municipality_Vestmannaeyjabaer,
                CountrySubdivision::Iceland_municipality_Vesturbyggd,
                CountrySubdivision::Iceland_municipality_Vopnafjardarhreppur,
                CountrySubdivision::Iceland_region_Austurland,
                CountrySubdivision::Iceland_region_Hofudborgarsvaedi,
                CountrySubdivision::Iceland_region_Nordurland_eystra,
                CountrySubdivision::Iceland_region_Nordurland_vestra,
                CountrySubdivision::Iceland_region_Sudurland,
                CountrySubdivision::Iceland_region_Sudurnes,
                CountrySubdivision::Iceland_region_Vestfirdir,
                CountrySubdivision::Iceland_region_Vesturland
            ],
            self::India => [
                CountrySubdivision::India_Union_territory_Andaman_and_Nicobar_Islands,
                CountrySubdivision::India_Union_territory_Chandigarh,
                CountrySubdivision::India_Union_territory_Dadra_and_Nagar_Haveli_and_Daman_and_Diu,
                CountrySubdivision::India_Union_territory_Delhi,
                CountrySubdivision::India_Union_territory_Jammu_and_Kashmir,
                CountrySubdivision::India_Union_territory_Ladakh,
                CountrySubdivision::India_Union_territory_Lakshadweep,
                CountrySubdivision::India_Union_territory_Puducherry,
                CountrySubdivision::India_state_Andhra_Pradesh,
                CountrySubdivision::India_state_Arunachal_Pradesh,
                CountrySubdivision::India_state_Assam,
                CountrySubdivision::India_state_Bihar,
                CountrySubdivision::India_state_Chhattisgarh,
                CountrySubdivision::India_state_Goa,
                CountrySubdivision::India_state_Gujarat,
                CountrySubdivision::India_state_Haryana,
                CountrySubdivision::India_state_Himachal_Pradesh,
                CountrySubdivision::India_state_Jharkhand,
                CountrySubdivision::India_state_Karnataka,
                CountrySubdivision::India_state_Kerala,
                CountrySubdivision::India_state_Madhya_Pradesh,
                CountrySubdivision::India_state_Maharashtra,
                CountrySubdivision::India_state_Manipur,
                CountrySubdivision::India_state_Meghalaya,
                CountrySubdivision::India_state_Mizoram,
                CountrySubdivision::India_state_Nagaland,
                CountrySubdivision::India_state_Odisha,
                CountrySubdivision::India_state_Punjab,
                CountrySubdivision::India_state_Rajasthan,
                CountrySubdivision::India_state_Sikkim,
                CountrySubdivision::India_state_Tamil_Nadu,
                CountrySubdivision::India_state_Telangana,
                CountrySubdivision::India_state_Tripura,
                CountrySubdivision::India_state_Uttar_Pradesh,
                CountrySubdivision::India_state_Uttarakhand,
                CountrySubdivision::India_state_West_Bengal
            ],
            self::Indonesia => [
                CountrySubdivision::Indonesia_capital_district_Jakarta_Raya,
                CountrySubdivision::Indonesia_geographical_unit_Jawa,
                CountrySubdivision::Indonesia_geographical_unit_Kalimantan,
                CountrySubdivision::Indonesia_geographical_unit_Maluku,
                CountrySubdivision::Indonesia_geographical_unit_Nusa_Tenggara,
                CountrySubdivision::Indonesia_geographical_unit_Papua,
                CountrySubdivision::Indonesia_geographical_unit_Sulawesi,
                CountrySubdivision::Indonesia_geographical_unit_Sumatera,
                CountrySubdivision::Indonesia_province_Aceh,
                CountrySubdivision::Indonesia_province_Bali,
                CountrySubdivision::Indonesia_province_Banten,
                CountrySubdivision::Indonesia_province_Bengkulu,
                CountrySubdivision::Indonesia_province_Gorontalo,
                CountrySubdivision::Indonesia_province_Jambi,
                CountrySubdivision::Indonesia_province_Jawa_Barat,
                CountrySubdivision::Indonesia_province_Jawa_Tengah,
                CountrySubdivision::Indonesia_province_Jawa_Timur,
                CountrySubdivision::Indonesia_province_Kalimantan_Barat,
                CountrySubdivision::Indonesia_province_Kalimantan_Selatan,
                CountrySubdivision::Indonesia_province_Kalimantan_Tengah,
                CountrySubdivision::Indonesia_province_Kalimantan_Timur,
                CountrySubdivision::Indonesia_province_Kalimantan_Utara,
                CountrySubdivision::Indonesia_province_Kepulauan_Bangka_Belitung,
                CountrySubdivision::Indonesia_province_Kepulauan_Riau,
                CountrySubdivision::Indonesia_province_Lampung,
                CountrySubdivision::Indonesia_province_Maluku,
                CountrySubdivision::Indonesia_province_Maluku_Utara,
                CountrySubdivision::Indonesia_province_Nusa_Tenggara_Barat,
                CountrySubdivision::Indonesia_province_Nusa_Tenggara_Timur,
                CountrySubdivision::Indonesia_province_Papua,
                CountrySubdivision::Indonesia_province_Papua_Barat,
                CountrySubdivision::Indonesia_province_Papua_Barat_Daya,
                CountrySubdivision::Indonesia_province_Papua_Pengunungan,
                CountrySubdivision::Indonesia_province_Papua_Selatan,
                CountrySubdivision::Indonesia_province_Papua_Tengah,
                CountrySubdivision::Indonesia_province_Riau,
                CountrySubdivision::Indonesia_province_Sulawesi_Barat,
                CountrySubdivision::Indonesia_province_Sulawesi_Selatan,
                CountrySubdivision::Indonesia_province_Sulawesi_Tengah,
                CountrySubdivision::Indonesia_province_Sulawesi_Tenggara,
                CountrySubdivision::Indonesia_province_Sulawesi_Utara,
                CountrySubdivision::Indonesia_province_Sumatera_Barat,
                CountrySubdivision::Indonesia_province_Sumatera_Selatan,
                CountrySubdivision::Indonesia_province_Sumatera_Utara,
                CountrySubdivision::Indonesia_special_region_Yogyakarta
            ],
            self::Iran => [
                CountrySubdivision::Iran_province_Alborz,
                CountrySubdivision::Iran_province_Ardabil,
                CountrySubdivision::Iran_province_Azarbayjan_e_Gharbi,
                CountrySubdivision::Iran_province_Azarbayjan_e_Sharqi,
                CountrySubdivision::Iran_province_Bushehr,
                CountrySubdivision::Iran_province_Chahar_Mahal_va_Bakhtiari,
                CountrySubdivision::Iran_province_Esfahan,
                CountrySubdivision::Iran_province_Fars,
                CountrySubdivision::Iran_province_Gilan,
                CountrySubdivision::Iran_province_Golestan,
                CountrySubdivision::Iran_province_Hamadan,
                CountrySubdivision::Iran_province_Hormozgan,
                CountrySubdivision::Iran_province_Ilam,
                CountrySubdivision::Iran_province_Kerman,
                CountrySubdivision::Iran_province_Kermanshah,
                CountrySubdivision::Iran_province_Khorasan_e_Jonubi,
                CountrySubdivision::Iran_province_Khorasan_e_Razavi,
                CountrySubdivision::Iran_province_Khorasan_e_Shomali,
                CountrySubdivision::Iran_province_Khuzestan,
                CountrySubdivision::Iran_province_Kohgiluyeh_va_Bowyer_Ahmad,
                CountrySubdivision::Iran_province_Kordestan,
                CountrySubdivision::Iran_province_Lorestan,
                CountrySubdivision::Iran_province_Markazi,
                CountrySubdivision::Iran_province_Mazandaran,
                CountrySubdivision::Iran_province_Qazvin,
                CountrySubdivision::Iran_province_Qom,
                CountrySubdivision::Iran_province_Semnan,
                CountrySubdivision::Iran_province_Sistan_va_Baluchestan,
                CountrySubdivision::Iran_province_Tehran,
                CountrySubdivision::Iran_province_Yazd,
                CountrySubdivision::Iran_province_Zanjan
            ],
            self::Iraq => [
                CountrySubdivision::Iraq_governorate_Al_Anbar,
                CountrySubdivision::Iraq_governorate_Al_Basrah,
                CountrySubdivision::Iraq_governorate_Al_Muthanna,
                CountrySubdivision::Iraq_governorate_Al_Qadisiyah,
                CountrySubdivision::Iraq_governorate_An_Najaf,
                CountrySubdivision::Iraq_governorate_Arbil,
                CountrySubdivision::Iraq_governorate_As_Sulaymaniyah,
                CountrySubdivision::Iraq_governorate_Babil,
                CountrySubdivision::Iraq_governorate_Baghdad,
                CountrySubdivision::Iraq_governorate_Dahuk,
                CountrySubdivision::Iraq_governorate_Dhi_Qar,
                CountrySubdivision::Iraq_governorate_Diyala,
                CountrySubdivision::Iraq_governorate_Karbala,
                CountrySubdivision::Iraq_governorate_Kirkuk,
                CountrySubdivision::Iraq_governorate_Maysan,
                CountrySubdivision::Iraq_governorate_Ninawa,
                CountrySubdivision::Iraq_governorate_Salah_ad_Din,
                CountrySubdivision::Iraq_governorate_Wasit,
                CountrySubdivision::Iraq_region_Herem_i_Kurdistan
            ],
            self::Ireland => [
                CountrySubdivision::Ireland_county_An_Cabhan,
                CountrySubdivision::Ireland_county_An_Clar,
                CountrySubdivision::Ireland_county_An_Iarmhi,
                CountrySubdivision::Ireland_county_An_Longfort,
                CountrySubdivision::Ireland_county_An_Mhi,
                CountrySubdivision::Ireland_county_Baile_Atha_Cliath,
                CountrySubdivision::Ireland_county_Carlow,
                CountrySubdivision::Ireland_county_Ciarrai,
                CountrySubdivision::Ireland_county_Cill_Chainnigh,
                CountrySubdivision::Ireland_county_Cill_Dara,
                CountrySubdivision::Ireland_county_Cill_Mhantain,
                CountrySubdivision::Ireland_county_Corcaigh,
                CountrySubdivision::Ireland_county_Donegal,
                CountrySubdivision::Ireland_county_Gaillimh,
                CountrySubdivision::Ireland_county_Laois,
                CountrySubdivision::Ireland_county_Leitrim,
                CountrySubdivision::Ireland_county_Limerick,
                CountrySubdivision::Ireland_county_Loch_Garman,
                CountrySubdivision::Ireland_county_Louth,
                CountrySubdivision::Ireland_county_Maigh_Eo,
                CountrySubdivision::Ireland_county_Monaghan,
                CountrySubdivision::Ireland_county_Offaly,
                CountrySubdivision::Ireland_county_Port_Lairge,
                CountrySubdivision::Ireland_county_Ros_Comain,
                CountrySubdivision::Ireland_county_Sligeach,
                CountrySubdivision::Ireland_county_Tiobraid_Arann,
                CountrySubdivision::Ireland_province_An_Mhumhain,
                CountrySubdivision::Ireland_province_Connacht,
                CountrySubdivision::Ireland_province_Laighin,
                CountrySubdivision::Ireland_province_Ulaidh
            ],
            self::Israel => [
                CountrySubdivision::Israel_district_Al_Awsat,
                CountrySubdivision::Israel_district_Al_Janubi,
                CountrySubdivision::Israel_district_Al_Quds,
                CountrySubdivision::Israel_district_Ash_Shamali,
                CountrySubdivision::Israel_district_Hefa,
                CountrySubdivision::Israel_district_Tall_Abib
            ],
            self::Italy => [
                CountrySubdivision::Italy_autonomous_province_Bolzano,
                CountrySubdivision::Italy_autonomous_province_Trento,
                CountrySubdivision::Italy_autonomous_region_Friuli_Venezia_Giulia,
                CountrySubdivision::Italy_autonomous_region_Sardegna,
                CountrySubdivision::Italy_autonomous_region_Sicilia,
                CountrySubdivision::Italy_autonomous_region_Trentino_Alto_Adige,
                CountrySubdivision::Italy_autonomous_region_Val_d_Aoste,
                CountrySubdivision::Italy_decentralized_regional_entity_Gorizia,
                CountrySubdivision::Italy_decentralized_regional_entity_Pordenone,
                CountrySubdivision::Italy_decentralized_regional_entity_Trieste,
                CountrySubdivision::Italy_decentralized_regional_entity_Udine,
                CountrySubdivision::Italy_free_municipal_consortium_Agrigento,
                CountrySubdivision::Italy_free_municipal_consortium_Caltanissetta,
                CountrySubdivision::Italy_free_municipal_consortium_Enna,
                CountrySubdivision::Italy_free_municipal_consortium_Ragusa,
                CountrySubdivision::Italy_free_municipal_consortium_Siracusa,
                CountrySubdivision::Italy_free_municipal_consortium_Trapani,
                CountrySubdivision::Italy_metropolitan_city_Bari,
                CountrySubdivision::Italy_metropolitan_city_Bologna,
                CountrySubdivision::Italy_metropolitan_city_Cagliari,
                CountrySubdivision::Italy_metropolitan_city_Catania,
                CountrySubdivision::Italy_metropolitan_city_Firenze,
                CountrySubdivision::Italy_metropolitan_city_Genova,
                CountrySubdivision::Italy_metropolitan_city_Messina,
                CountrySubdivision::Italy_metropolitan_city_Milano,
                CountrySubdivision::Italy_metropolitan_city_Napoli,
                CountrySubdivision::Italy_metropolitan_city_Palermo,
                CountrySubdivision::Italy_metropolitan_city_Reggio_Calabria,
                CountrySubdivision::Italy_metropolitan_city_Roma,
                CountrySubdivision::Italy_metropolitan_city_Torino,
                CountrySubdivision::Italy_metropolitan_city_Venezia,
                CountrySubdivision::Italy_province_Alessandria,
                CountrySubdivision::Italy_province_Ancona,
                CountrySubdivision::Italy_province_Arezzo,
                CountrySubdivision::Italy_province_Ascoli_Piceno,
                CountrySubdivision::Italy_province_Asti,
                CountrySubdivision::Italy_province_Avellino,
                CountrySubdivision::Italy_province_Barletta_Andria_Trani,
                CountrySubdivision::Italy_province_Belluno,
                CountrySubdivision::Italy_province_Benevento,
                CountrySubdivision::Italy_province_Bergamo,
                CountrySubdivision::Italy_province_Biella,
                CountrySubdivision::Italy_province_Brescia,
                CountrySubdivision::Italy_province_Brindisi,
                CountrySubdivision::Italy_province_Campobasso,
                CountrySubdivision::Italy_province_Caserta,
                CountrySubdivision::Italy_province_Catanzaro,
                CountrySubdivision::Italy_province_Chieti,
                CountrySubdivision::Italy_province_Como,
                CountrySubdivision::Italy_province_Cosenza,
                CountrySubdivision::Italy_province_Cremona,
                CountrySubdivision::Italy_province_Crotone,
                CountrySubdivision::Italy_province_Cuneo,
                CountrySubdivision::Italy_province_Fermo,
                CountrySubdivision::Italy_province_Ferrara,
                CountrySubdivision::Italy_province_Foggia,
                CountrySubdivision::Italy_province_Forli_Cesena,
                CountrySubdivision::Italy_province_Frosinone,
                CountrySubdivision::Italy_province_Grosseto,
                CountrySubdivision::Italy_province_Imperia,
                CountrySubdivision::Italy_province_Isernia,
                CountrySubdivision::Italy_province_L_Aquila,
                CountrySubdivision::Italy_province_La_Spezia,
                CountrySubdivision::Italy_province_Latina,
                CountrySubdivision::Italy_province_Lecce,
                CountrySubdivision::Italy_province_Lecco,
                CountrySubdivision::Italy_province_Livorno,
                CountrySubdivision::Italy_province_Lodi,
                CountrySubdivision::Italy_province_Lucca,
                CountrySubdivision::Italy_province_Macerata,
                CountrySubdivision::Italy_province_Mantova,
                CountrySubdivision::Italy_province_Massa_Carrara,
                CountrySubdivision::Italy_province_Matera,
                CountrySubdivision::Italy_province_Modena,
                CountrySubdivision::Italy_province_Monza_e_Brianza,
                CountrySubdivision::Italy_province_Novara,
                CountrySubdivision::Italy_province_Nuoro,
                CountrySubdivision::Italy_province_Oristano,
                CountrySubdivision::Italy_province_Padova,
                CountrySubdivision::Italy_province_Parma,
                CountrySubdivision::Italy_province_Pavia,
                CountrySubdivision::Italy_province_Perugia,
                CountrySubdivision::Italy_province_Pesaro_e_Urbino,
                CountrySubdivision::Italy_province_Pescara,
                CountrySubdivision::Italy_province_Piacenza,
                CountrySubdivision::Italy_province_Pisa,
                CountrySubdivision::Italy_province_Pistoia,
                CountrySubdivision::Italy_province_Potenza,
                CountrySubdivision::Italy_province_Prato,
                CountrySubdivision::Italy_province_Ravenna,
                CountrySubdivision::Italy_province_Reggio_Emilia,
                CountrySubdivision::Italy_province_Rieti,
                CountrySubdivision::Italy_province_Rimini,
                CountrySubdivision::Italy_province_Rovigo,
                CountrySubdivision::Italy_province_Salerno,
                CountrySubdivision::Italy_province_Sassari,
                CountrySubdivision::Italy_province_Savona,
                CountrySubdivision::Italy_province_Siena,
                CountrySubdivision::Italy_province_Sondrio,
                CountrySubdivision::Italy_province_Sud_Sardegna,
                CountrySubdivision::Italy_province_Taranto,
                CountrySubdivision::Italy_province_Teramo,
                CountrySubdivision::Italy_province_Terni,
                CountrySubdivision::Italy_province_Treviso,
                CountrySubdivision::Italy_province_Varese,
                CountrySubdivision::Italy_province_Verbano_Cusio_Ossola,
                CountrySubdivision::Italy_province_Vercelli,
                CountrySubdivision::Italy_province_Verona,
                CountrySubdivision::Italy_province_Vibo_Valentia,
                CountrySubdivision::Italy_province_Vicenza,
                CountrySubdivision::Italy_province_Viterbo,
                CountrySubdivision::Italy_region_Abruzzo,
                CountrySubdivision::Italy_region_Basilicata,
                CountrySubdivision::Italy_region_Calabria,
                CountrySubdivision::Italy_region_Campania,
                CountrySubdivision::Italy_region_Emilia_Romagna,
                CountrySubdivision::Italy_region_Lazio,
                CountrySubdivision::Italy_region_Liguria,
                CountrySubdivision::Italy_region_Lombardia,
                CountrySubdivision::Italy_region_Marche,
                CountrySubdivision::Italy_region_Molise,
                CountrySubdivision::Italy_region_Piemonte,
                CountrySubdivision::Italy_region_Puglia,
                CountrySubdivision::Italy_region_Toscana,
                CountrySubdivision::Italy_region_Umbria,
                CountrySubdivision::Italy_region_Veneto
            ],
            self::Jamaica => [
                CountrySubdivision::Jamaica_parish_Clarendon,
                CountrySubdivision::Jamaica_parish_Hanover,
                CountrySubdivision::Jamaica_parish_Kingston,
                CountrySubdivision::Jamaica_parish_Manchester,
                CountrySubdivision::Jamaica_parish_Portland,
                CountrySubdivision::Jamaica_parish_Saint_Andrew,
                CountrySubdivision::Jamaica_parish_Saint_Ann,
                CountrySubdivision::Jamaica_parish_Saint_Catherine,
                CountrySubdivision::Jamaica_parish_Saint_Elizabeth,
                CountrySubdivision::Jamaica_parish_Saint_James,
                CountrySubdivision::Jamaica_parish_Saint_Mary,
                CountrySubdivision::Jamaica_parish_Saint_Thomas,
                CountrySubdivision::Jamaica_parish_Trelawny,
                CountrySubdivision::Jamaica_parish_Westmoreland
            ],
            self::Japan => [
                CountrySubdivision::Japan_prefecture_Aichi,
                CountrySubdivision::Japan_prefecture_Akita,
                CountrySubdivision::Japan_prefecture_Aomori,
                CountrySubdivision::Japan_prefecture_Chiba,
                CountrySubdivision::Japan_prefecture_Ehime,
                CountrySubdivision::Japan_prefecture_Fukui,
                CountrySubdivision::Japan_prefecture_Fukuoka,
                CountrySubdivision::Japan_prefecture_Fukushima,
                CountrySubdivision::Japan_prefecture_Gifu,
                CountrySubdivision::Japan_prefecture_Gunma,
                CountrySubdivision::Japan_prefecture_Hiroshima,
                CountrySubdivision::Japan_prefecture_Hokkaido,
                CountrySubdivision::Japan_prefecture_Hyogo,
                CountrySubdivision::Japan_prefecture_Ibaraki,
                CountrySubdivision::Japan_prefecture_Ishikawa,
                CountrySubdivision::Japan_prefecture_Iwate,
                CountrySubdivision::Japan_prefecture_Kagawa,
                CountrySubdivision::Japan_prefecture_Kagoshima,
                CountrySubdivision::Japan_prefecture_Kanagawa,
                CountrySubdivision::Japan_prefecture_Kochi,
                CountrySubdivision::Japan_prefecture_Kumamoto,
                CountrySubdivision::Japan_prefecture_Kyoto,
                CountrySubdivision::Japan_prefecture_Mie,
                CountrySubdivision::Japan_prefecture_Miyagi,
                CountrySubdivision::Japan_prefecture_Miyazaki,
                CountrySubdivision::Japan_prefecture_Nagano,
                CountrySubdivision::Japan_prefecture_Nagasaki,
                CountrySubdivision::Japan_prefecture_Nara,
                CountrySubdivision::Japan_prefecture_Niigata,
                CountrySubdivision::Japan_prefecture_Oita,
                CountrySubdivision::Japan_prefecture_Okayama,
                CountrySubdivision::Japan_prefecture_Okinawa,
                CountrySubdivision::Japan_prefecture_Osaka,
                CountrySubdivision::Japan_prefecture_Saga,
                CountrySubdivision::Japan_prefecture_Saitama,
                CountrySubdivision::Japan_prefecture_Shiga,
                CountrySubdivision::Japan_prefecture_Shimane,
                CountrySubdivision::Japan_prefecture_Shizuoka,
                CountrySubdivision::Japan_prefecture_Tochigi,
                CountrySubdivision::Japan_prefecture_Tokushima,
                CountrySubdivision::Japan_prefecture_Tokyo,
                CountrySubdivision::Japan_prefecture_Tottori,
                CountrySubdivision::Japan_prefecture_Toyama,
                CountrySubdivision::Japan_prefecture_Wakayama,
                CountrySubdivision::Japan_prefecture_Yamagata,
                CountrySubdivision::Japan_prefecture_Yamaguchi,
                CountrySubdivision::Japan_prefecture_Yamanashi
            ],
            self::Jordan => [
                CountrySubdivision::Jordan_governorate_Ajlun,
                CountrySubdivision::Jordan_governorate_Al_Aqabah,
                CountrySubdivision::Jordan_governorate_Al_Asimah,
                CountrySubdivision::Jordan_governorate_Al_Balqa,
                CountrySubdivision::Jordan_governorate_Al_Karak,
                CountrySubdivision::Jordan_governorate_Al_Mafraq,
                CountrySubdivision::Jordan_governorate_At_Tafilah,
                CountrySubdivision::Jordan_governorate_Az_Zarqa,
                CountrySubdivision::Jordan_governorate_Irbid,
                CountrySubdivision::Jordan_governorate_Jarash,
                CountrySubdivision::Jordan_governorate_Ma_an,
                CountrySubdivision::Jordan_governorate_Madaba
            ],
            self::Kazakhstan => [
                CountrySubdivision::Kazakhstan_city_Almaty,
                CountrySubdivision::Kazakhstan_city_Astana,
                CountrySubdivision::Kazakhstan_city_Shymkent,
                CountrySubdivision::Kazakhstan_region_Abajskaja_oblast,
                CountrySubdivision::Kazakhstan_region_Akmolinskaja_oblast,
                CountrySubdivision::Kazakhstan_region_Aktjubinskaja_oblast,
                CountrySubdivision::Kazakhstan_region_Almatinskaja_oblast,
                CountrySubdivision::Kazakhstan_region_Atyrauskaja_oblast,
                CountrySubdivision::Kazakhstan_region_Batys_Qazaqstan_oblysy,
                CountrySubdivision::Kazakhstan_region_Karagandinskaja_oblast,
                CountrySubdivision::Kazakhstan_region_Kostanajskaja_oblast,
                CountrySubdivision::Kazakhstan_region_Kyzylordinskaja_oblast,
                CountrySubdivision::Kazakhstan_region_Mangghystau_oblysy,
                CountrySubdivision::Kazakhstan_region_Pavlodar_oblysy,
                CountrySubdivision::Kazakhstan_region_Severo_Kazahstanskaja_oblast,
                CountrySubdivision::Kazakhstan_region_Shyghys_Qazaqstan_oblysy,
                CountrySubdivision::Kazakhstan_region_Turkestankaya_oblast,
                CountrySubdivision::Kazakhstan_region_Ulytauskaja_oblast,
                CountrySubdivision::Kazakhstan_region_Zhambyl_oblysy,
                CountrySubdivision::Kazakhstan_region_Zhetisu_oblysy
            ],
            self::Kenya => [
                CountrySubdivision::Kenya_county_Baringo,
                CountrySubdivision::Kenya_county_Bomet,
                CountrySubdivision::Kenya_county_Bungoma,
                CountrySubdivision::Kenya_county_Busia,
                CountrySubdivision::Kenya_county_Elgeyo_Marakwet,
                CountrySubdivision::Kenya_county_Embu,
                CountrySubdivision::Kenya_county_Garissa,
                CountrySubdivision::Kenya_county_Homa_Bay,
                CountrySubdivision::Kenya_county_Isiolo,
                CountrySubdivision::Kenya_county_Kajiado,
                CountrySubdivision::Kenya_county_Kakamega,
                CountrySubdivision::Kenya_county_Kericho,
                CountrySubdivision::Kenya_county_Kiambu,
                CountrySubdivision::Kenya_county_Kilifi,
                CountrySubdivision::Kenya_county_Kirinyaga,
                CountrySubdivision::Kenya_county_Kisii,
                CountrySubdivision::Kenya_county_Kisumu,
                CountrySubdivision::Kenya_county_Kitui,
                CountrySubdivision::Kenya_county_Kwale,
                CountrySubdivision::Kenya_county_Laikipia,
                CountrySubdivision::Kenya_county_Lamu,
                CountrySubdivision::Kenya_county_Machakos,
                CountrySubdivision::Kenya_county_Makueni,
                CountrySubdivision::Kenya_county_Mandera,
                CountrySubdivision::Kenya_county_Marsabit,
                CountrySubdivision::Kenya_county_Meru,
                CountrySubdivision::Kenya_county_Migori,
                CountrySubdivision::Kenya_county_Mombasa,
                CountrySubdivision::Kenya_county_Murang_a,
                CountrySubdivision::Kenya_county_Nairobi_City,
                CountrySubdivision::Kenya_county_Nakuru,
                CountrySubdivision::Kenya_county_Nandi,
                CountrySubdivision::Kenya_county_Narok,
                CountrySubdivision::Kenya_county_Nyamira,
                CountrySubdivision::Kenya_county_Nyandarua,
                CountrySubdivision::Kenya_county_Nyeri,
                CountrySubdivision::Kenya_county_Samburu,
                CountrySubdivision::Kenya_county_Siaya,
                CountrySubdivision::Kenya_county_Taita_Taveta,
                CountrySubdivision::Kenya_county_Tana_River,
                CountrySubdivision::Kenya_county_Tharaka_Nithi,
                CountrySubdivision::Kenya_county_Trans_Nzoia,
                CountrySubdivision::Kenya_county_Turkana,
                CountrySubdivision::Kenya_county_Uasin_Gishu,
                CountrySubdivision::Kenya_county_Vihiga,
                CountrySubdivision::Kenya_county_Wajir,
                CountrySubdivision::Kenya_county_West_Pokot
            ],
            self::Kiribati => [
                CountrySubdivision::Kiribati_group_of_islands_20_inhabited_islands_Gilbert_Islands,
                CountrySubdivision::Kiribati_group_of_islands_20_inhabited_islands_Line_Islands,
                CountrySubdivision::Kiribati_group_of_islands_20_inhabited_islands_Phoenix_Islands
            ],
            self::Korea_Democratic_Peoples_Republic => [
                CountrySubdivision::North_Korea_capital_city_P_yongyang,
                CountrySubdivision::North_Korea_metropolitan_city_Kaeseong,
                CountrySubdivision::North_Korea_metropolitan_city_Nampho,
                CountrySubdivision::North_Korea_province_Chagang_do,
                CountrySubdivision::North_Korea_province_Hamgyong_bukto,
                CountrySubdivision::North_Korea_province_Hamgyong_namdo,
                CountrySubdivision::North_Korea_province_Hwanghae_bukto,
                CountrySubdivision::North_Korea_province_Hwanghae_namdo,
                CountrySubdivision::North_Korea_province_Kangweonto,
                CountrySubdivision::North_Korea_province_P_yongan_bukto,
                CountrySubdivision::North_Korea_province_P_yongan_namdo,
                CountrySubdivision::North_Korea_province_Ryanggang_do,
                CountrySubdivision::North_Korea_special_city_Raseon
            ],
            self::Korea_Republic => [
                CountrySubdivision::South_Korea_metropolitan_city_Busan_gwangyeoksi,
                CountrySubdivision::South_Korea_metropolitan_city_Daegu_gwangyeoksi,
                CountrySubdivision::South_Korea_metropolitan_city_Daejeon_gwangyeoksi,
                CountrySubdivision::South_Korea_metropolitan_city_Gwangju_gwangyeoksi,
                CountrySubdivision::South_Korea_metropolitan_city_Incheon_gwangyeoksi,
                CountrySubdivision::South_Korea_metropolitan_city_Ulsan_gwangyeoksi,
                CountrySubdivision::South_Korea_province_Chungcheongbuk_do,
                CountrySubdivision::South_Korea_province_Chungcheongnam_do,
                CountrySubdivision::South_Korea_province_Gyeonggi_do,
                CountrySubdivision::South_Korea_province_Gyeongsangbuk_do,
                CountrySubdivision::South_Korea_province_Gyeongsangnam_do,
                CountrySubdivision::South_Korea_province_Jeollabuk_do,
                CountrySubdivision::South_Korea_province_Jeollanam_do,
                CountrySubdivision::South_Korea_special_city_Seoul_teukbyeolsi,
                CountrySubdivision::South_Korea_special_self_governing_city_Sejong,
                CountrySubdivision::South_Korea_special_self_governing_province_Gangwon_teukbyeoljachido,
                CountrySubdivision::South_Korea_special_self_governing_province_Jeju_teukbyeoljachido
            ],
            self::Kuwait => [
                CountrySubdivision::Kuwait_governorate_Al_Ahmadi,
                CountrySubdivision::Kuwait_governorate_Al_Asimah,
                CountrySubdivision::Kuwait_governorate_Al_Farwaniyah,
                CountrySubdivision::Kuwait_governorate_Al_Jahra,
                CountrySubdivision::Kuwait_governorate_Hawalli,
                CountrySubdivision::Kuwait_governorate_Mubarak_al_Kabir
            ],
            self::Kyrgyzstan => [
                CountrySubdivision::Kyrgyzstan_city_Bishkek_Shaary,
                CountrySubdivision::Kyrgyzstan_city_Gorod_Osh,
                CountrySubdivision::Kyrgyzstan_region_Batken,
                CountrySubdivision::Kyrgyzstan_region_Chuyskaya_oblast,
                CountrySubdivision::Kyrgyzstan_region_Dzhalal_Abadskaya_oblast,
                CountrySubdivision::Kyrgyzstan_region_Issyk_Kul_skaja_oblast,
                CountrySubdivision::Kyrgyzstan_region_Naryn,
                CountrySubdivision::Kyrgyzstan_region_Osh,
                CountrySubdivision::Kyrgyzstan_region_Talas
            ],
            self::Lao_Peoples_Democratic_Republic => [
                CountrySubdivision::Laos_prefecture_Viangchan,
                CountrySubdivision::Laos_province_Attapu,
                CountrySubdivision::Laos_province_Bokeo,
                CountrySubdivision::Laos_province_Bolikhamxai,
                CountrySubdivision::Laos_province_Champasak,
                CountrySubdivision::Laos_province_Houaphan,
                CountrySubdivision::Laos_province_Khammouan,
                CountrySubdivision::Laos_province_Louang_Namtha,
                CountrySubdivision::Laos_province_Louangphabang,
                CountrySubdivision::Laos_province_Oudomxai,
                CountrySubdivision::Laos_province_Phongsali,
                CountrySubdivision::Laos_province_Salavan,
                CountrySubdivision::Laos_province_Savannakhet,
                CountrySubdivision::Laos_province_Viangchan,
                CountrySubdivision::Laos_province_Xaignabouli,
                CountrySubdivision::Laos_province_Xaisomboun,
                CountrySubdivision::Laos_province_Xekong,
                CountrySubdivision::Laos_province_Xiangkhouang
            ],
            self::Latvia => [
                CountrySubdivision::Latvia_municipality_Adazu_novads,
                CountrySubdivision::Latvia_municipality_Aizkraukles_novads,
                CountrySubdivision::Latvia_municipality_Aluksnes_novads,
                CountrySubdivision::Latvia_municipality_Augsdaugavas_novads,
                CountrySubdivision::Latvia_municipality_Balvu_novads,
                CountrySubdivision::Latvia_municipality_Bauskas_novads,
                CountrySubdivision::Latvia_municipality_Cesu_novads,
                CountrySubdivision::Latvia_municipality_Dienvidkurzemes_Novads,
                CountrySubdivision::Latvia_municipality_Dobeles_novads,
                CountrySubdivision::Latvia_municipality_Gulbenes_novads,
                CountrySubdivision::Latvia_municipality_Jekabpils_novads,
                CountrySubdivision::Latvia_municipality_Jelgavas_novads,
                CountrySubdivision::Latvia_municipality_Kekavas_novads,
                CountrySubdivision::Latvia_municipality_Kraslavas_novads,
                CountrySubdivision::Latvia_municipality_Kuldigas_novads,
                CountrySubdivision::Latvia_municipality_Limbazu_novads,
                CountrySubdivision::Latvia_municipality_Livanu_novads,
                CountrySubdivision::Latvia_municipality_Ludzas_novads,
                CountrySubdivision::Latvia_municipality_Madonas_novads,
                CountrySubdivision::Latvia_municipality_Marupes_novads,
                CountrySubdivision::Latvia_municipality_Ogres_novads,
                CountrySubdivision::Latvia_municipality_Olaines_novads,
                CountrySubdivision::Latvia_municipality_Preilu_novads,
                CountrySubdivision::Latvia_municipality_Rezeknes_novads,
                CountrySubdivision::Latvia_municipality_Ropazu_novads,
                CountrySubdivision::Latvia_municipality_Salaspils_novads,
                CountrySubdivision::Latvia_municipality_Saldus_novads,
                CountrySubdivision::Latvia_municipality_Saulkrastu_novads,
                CountrySubdivision::Latvia_municipality_Siguldas_novads,
                CountrySubdivision::Latvia_municipality_Smiltenes_novads,
                CountrySubdivision::Latvia_municipality_Talsu_novads,
                CountrySubdivision::Latvia_municipality_Tukuma_novads,
                CountrySubdivision::Latvia_municipality_Valkas_novads,
                CountrySubdivision::Latvia_municipality_Valmieras_Novads,
                CountrySubdivision::Latvia_municipality_Varaklanu_novads,
                CountrySubdivision::Latvia_municipality_Ventspils_novads,
                CountrySubdivision::Latvia_state_city_Daugavpils,
                CountrySubdivision::Latvia_state_city_Jelgava,
                CountrySubdivision::Latvia_state_city_Jurmala,
                CountrySubdivision::Latvia_state_city_Liepaja,
                CountrySubdivision::Latvia_state_city_Rezekne,
                CountrySubdivision::Latvia_state_city_Riga,
                CountrySubdivision::Latvia_state_city_Ventspils
            ],
            self::Lebanon => [
                CountrySubdivision::Lebanon_governorate_Aakkar,
                CountrySubdivision::Lebanon_governorate_Al_Biqa,
                CountrySubdivision::Lebanon_governorate_Al_Janub,
                CountrySubdivision::Lebanon_governorate_An_Nabatiyah,
                CountrySubdivision::Lebanon_governorate_Ash_Shimal,
                CountrySubdivision::Lebanon_governorate_Baalbek_Hermel,
                CountrySubdivision::Lebanon_governorate_Bayrut,
                CountrySubdivision::Lebanon_governorate_Jabal_Lubnan
            ],
            self::Lesotho => [
                CountrySubdivision::Lesotho_district_Berea,
                CountrySubdivision::Lesotho_district_Botha_Bothe,
                CountrySubdivision::Lesotho_district_Leribe,
                CountrySubdivision::Lesotho_district_Mafeteng,
                CountrySubdivision::Lesotho_district_Maseru,
                CountrySubdivision::Lesotho_district_Mohale_s_Hoek,
                CountrySubdivision::Lesotho_district_Mokhotlong,
                CountrySubdivision::Lesotho_district_Qacha_s_Nek,
                CountrySubdivision::Lesotho_district_Quthing,
                CountrySubdivision::Lesotho_district_Thaba_Tseka
            ],
            self::Liberia => [
                CountrySubdivision::Liberia_county_Bomi,
                CountrySubdivision::Liberia_county_Bong,
                CountrySubdivision::Liberia_county_Gbarpolu,
                CountrySubdivision::Liberia_county_Grand_Bassa,
                CountrySubdivision::Liberia_county_Grand_Cape_Mount,
                CountrySubdivision::Liberia_county_Grand_Gedeh,
                CountrySubdivision::Liberia_county_Grand_Kru,
                CountrySubdivision::Liberia_county_Lofa,
                CountrySubdivision::Liberia_county_Margibi,
                CountrySubdivision::Liberia_county_Maryland,
                CountrySubdivision::Liberia_county_Montserrado,
                CountrySubdivision::Liberia_county_Nimba,
                CountrySubdivision::Liberia_county_River_Cess,
                CountrySubdivision::Liberia_county_River_Gee,
                CountrySubdivision::Liberia_county_Sinoe
            ],
            self::Libya => [
                CountrySubdivision::Libya_popularate_Al_Butnan,
                CountrySubdivision::Libya_popularate_Al_Jabal_al_Akhdar,
                CountrySubdivision::Libya_popularate_Al_Jabal_al_Gharbi,
                CountrySubdivision::Libya_popularate_Al_Jafarah,
                CountrySubdivision::Libya_popularate_Al_Jufrah,
                CountrySubdivision::Libya_popularate_Al_Kufrah,
                CountrySubdivision::Libya_popularate_Al_Marj,
                CountrySubdivision::Libya_popularate_Al_Marqab,
                CountrySubdivision::Libya_popularate_Al_Wahat,
                CountrySubdivision::Libya_popularate_An_Nuqat_al_Khams,
                CountrySubdivision::Libya_popularate_Az_Zawiyah,
                CountrySubdivision::Libya_popularate_Banghazi,
                CountrySubdivision::Libya_popularate_Darnah,
                CountrySubdivision::Libya_popularate_Ghat,
                CountrySubdivision::Libya_popularate_Misratah,
                CountrySubdivision::Libya_popularate_Murzuq,
                CountrySubdivision::Libya_popularate_Nalut,
                CountrySubdivision::Libya_popularate_Sabha,
                CountrySubdivision::Libya_popularate_Surt,
                CountrySubdivision::Libya_popularate_Tarabulus,
                CountrySubdivision::Libya_popularate_Wadi_al_Hayat,
                CountrySubdivision::Libya_popularate_Wadi_ash_Shati
            ],
            self::Liechtenstein => [
                CountrySubdivision::Liechtenstein_commune_Balzers,
                CountrySubdivision::Liechtenstein_commune_Eschen,
                CountrySubdivision::Liechtenstein_commune_Gamprin,
                CountrySubdivision::Liechtenstein_commune_Mauren,
                CountrySubdivision::Liechtenstein_commune_Planken,
                CountrySubdivision::Liechtenstein_commune_Ruggell,
                CountrySubdivision::Liechtenstein_commune_Schaan,
                CountrySubdivision::Liechtenstein_commune_Schellenberg,
                CountrySubdivision::Liechtenstein_commune_Triesen,
                CountrySubdivision::Liechtenstein_commune_Triesenberg,
                CountrySubdivision::Liechtenstein_commune_Vaduz
            ],
            self::Lithuania => [
                CountrySubdivision::Lithuania_city_municipality_Alytaus_miestas,
                CountrySubdivision::Lithuania_city_municipality_Kauno_miestas,
                CountrySubdivision::Lithuania_city_municipality_Klaipedos_miestas,
                CountrySubdivision::Lithuania_city_municipality_Palangos_miestas,
                CountrySubdivision::Lithuania_city_municipality_Panevezio_miestas,
                CountrySubdivision::Lithuania_city_municipality_Siauliu_miestas,
                CountrySubdivision::Lithuania_city_municipality_Vilniaus_miestas,
                CountrySubdivision::Lithuania_county_Alytaus_apskritis,
                CountrySubdivision::Lithuania_county_Kauno_apskritis,
                CountrySubdivision::Lithuania_county_Klaipedos_apskritis,
                CountrySubdivision::Lithuania_county_Marijampoles_apskritis,
                CountrySubdivision::Lithuania_county_Panevezio_apskritis,
                CountrySubdivision::Lithuania_county_Siauliu_apskritis,
                CountrySubdivision::Lithuania_county_Taurages_apskritis,
                CountrySubdivision::Lithuania_county_Telsiu_apskritis,
                CountrySubdivision::Lithuania_county_Utenos_apskritis,
                CountrySubdivision::Lithuania_county_Vilniaus_apskritis,
                CountrySubdivision::Lithuania_district_municipality_Akmene,
                CountrySubdivision::Lithuania_district_municipality_Alytus,
                CountrySubdivision::Lithuania_district_municipality_Anyksciai,
                CountrySubdivision::Lithuania_district_municipality_Birzai,
                CountrySubdivision::Lithuania_district_municipality_Ignalina,
                CountrySubdivision::Lithuania_district_municipality_Jonava,
                CountrySubdivision::Lithuania_district_municipality_Joniskis,
                CountrySubdivision::Lithuania_district_municipality_Jurbarkas,
                CountrySubdivision::Lithuania_district_municipality_Kaisiadorys,
                CountrySubdivision::Lithuania_district_municipality_Kaunas,
                CountrySubdivision::Lithuania_district_municipality_Kedainiai,
                CountrySubdivision::Lithuania_district_municipality_Kelme,
                CountrySubdivision::Lithuania_district_municipality_Klaipeda,
                CountrySubdivision::Lithuania_district_municipality_Kretinga,
                CountrySubdivision::Lithuania_district_municipality_Kupiskis,
                CountrySubdivision::Lithuania_district_municipality_Lazdijai,
                CountrySubdivision::Lithuania_district_municipality_Marijampole,
                CountrySubdivision::Lithuania_district_municipality_Mazeikiai,
                CountrySubdivision::Lithuania_district_municipality_Moletai,
                CountrySubdivision::Lithuania_district_municipality_Pakruojis,
                CountrySubdivision::Lithuania_district_municipality_Panevezys,
                CountrySubdivision::Lithuania_district_municipality_Pasvalys,
                CountrySubdivision::Lithuania_district_municipality_Plunge,
                CountrySubdivision::Lithuania_district_municipality_Prienai,
                CountrySubdivision::Lithuania_district_municipality_Radviliskis,
                CountrySubdivision::Lithuania_district_municipality_Raseiniai,
                CountrySubdivision::Lithuania_district_municipality_Rokiskis,
                CountrySubdivision::Lithuania_district_municipality_Sakiai,
                CountrySubdivision::Lithuania_district_municipality_Salcininkai,
                CountrySubdivision::Lithuania_district_municipality_Siauliai,
                CountrySubdivision::Lithuania_district_municipality_Silale,
                CountrySubdivision::Lithuania_district_municipality_Silute,
                CountrySubdivision::Lithuania_district_municipality_Sirvintos,
                CountrySubdivision::Lithuania_district_municipality_Skuodas,
                CountrySubdivision::Lithuania_district_municipality_Svencionys,
                CountrySubdivision::Lithuania_district_municipality_Taurage,
                CountrySubdivision::Lithuania_district_municipality_Telsiai,
                CountrySubdivision::Lithuania_district_municipality_Trakai,
                CountrySubdivision::Lithuania_district_municipality_Ukmerge,
                CountrySubdivision::Lithuania_district_municipality_Utena,
                CountrySubdivision::Lithuania_district_municipality_Varena,
                CountrySubdivision::Lithuania_district_municipality_Vilkaviskis,
                CountrySubdivision::Lithuania_district_municipality_Vilnius,
                CountrySubdivision::Lithuania_district_municipality_Zarasai,
                CountrySubdivision::Lithuania_municipality_Birstonas,
                CountrySubdivision::Lithuania_municipality_Druskininkai,
                CountrySubdivision::Lithuania_municipality_Elektrenai,
                CountrySubdivision::Lithuania_municipality_Kalvarija,
                CountrySubdivision::Lithuania_municipality_Kazlu_Rudos,
                CountrySubdivision::Lithuania_municipality_Neringa,
                CountrySubdivision::Lithuania_municipality_Pagegiai,
                CountrySubdivision::Lithuania_municipality_Rietavas,
                CountrySubdivision::Lithuania_municipality_Visaginas
            ],
            self::Luxembourg => [
                CountrySubdivision::Luxembourg_canton_Capellen,
                CountrySubdivision::Luxembourg_canton_Clerf,
                CountrySubdivision::Luxembourg_canton_Diekirch,
                CountrySubdivision::Luxembourg_canton_Echternach,
                CountrySubdivision::Luxembourg_canton_Esch_an_der_Alzette,
                CountrySubdivision::Luxembourg_canton_Grevenmacher,
                CountrySubdivision::Luxembourg_canton_Luxembourg,
                CountrySubdivision::Luxembourg_canton_Mersch,
                CountrySubdivision::Luxembourg_canton_Redange,
                CountrySubdivision::Luxembourg_canton_Remich,
                CountrySubdivision::Luxembourg_canton_Veianen,
                CountrySubdivision::Luxembourg_canton_Wiltz
            ],
            self::Madagascar => [
                CountrySubdivision::Madagascar_province_Antananarivo,
                CountrySubdivision::Madagascar_province_Antsiranana,
                CountrySubdivision::Madagascar_province_Fianarantsoa,
                CountrySubdivision::Madagascar_province_Mahajanga,
                CountrySubdivision::Madagascar_province_Toamasina,
                CountrySubdivision::Madagascar_province_Toliara
            ],
            self::Malawi => [
                CountrySubdivision::Malawi_district_Balaka,
                CountrySubdivision::Malawi_district_Blantyre,
                CountrySubdivision::Malawi_district_Chikwawa,
                CountrySubdivision::Malawi_district_Chiradzulu,
                CountrySubdivision::Malawi_district_Chitipa,
                CountrySubdivision::Malawi_district_Dedza,
                CountrySubdivision::Malawi_district_Dowa,
                CountrySubdivision::Malawi_district_Karonga,
                CountrySubdivision::Malawi_district_Kasungu,
                CountrySubdivision::Malawi_district_Likoma,
                CountrySubdivision::Malawi_district_Lilongwe,
                CountrySubdivision::Malawi_district_Machinga,
                CountrySubdivision::Malawi_district_Mangochi,
                CountrySubdivision::Malawi_district_Mchinji,
                CountrySubdivision::Malawi_district_Mulanje,
                CountrySubdivision::Malawi_district_Mwanza,
                CountrySubdivision::Malawi_district_Mzimba,
                CountrySubdivision::Malawi_district_Neno,
                CountrySubdivision::Malawi_district_Nkhata_Bay,
                CountrySubdivision::Malawi_district_Nkhotakota,
                CountrySubdivision::Malawi_district_Nsanje,
                CountrySubdivision::Malawi_district_Ntcheu,
                CountrySubdivision::Malawi_district_Ntchisi,
                CountrySubdivision::Malawi_district_Phalombe,
                CountrySubdivision::Malawi_district_Rumphi,
                CountrySubdivision::Malawi_district_Salima,
                CountrySubdivision::Malawi_district_Thyolo,
                CountrySubdivision::Malawi_district_Zomba,
                CountrySubdivision::Malawi_region_Central_Region,
                CountrySubdivision::Malawi_region_Chakumpoto,
                CountrySubdivision::Malawi_region_Chakumwera
            ],
            self::Malaysia => [
                CountrySubdivision::Malaysia_federal_territory_Wilayah_Persekutuan_Kuala_Lumpur,
                CountrySubdivision::Malaysia_federal_territory_Wilayah_Persekutuan_Labuan,
                CountrySubdivision::Malaysia_federal_territory_Wilayah_Persekutuan_Putrajaya,
                CountrySubdivision::Malaysia_state_Johor,
                CountrySubdivision::Malaysia_state_Kedah,
                CountrySubdivision::Malaysia_state_Kelantan,
                CountrySubdivision::Malaysia_state_Melaka,
                CountrySubdivision::Malaysia_state_Negeri_Sembilan,
                CountrySubdivision::Malaysia_state_Pahang,
                CountrySubdivision::Malaysia_state_Perak,
                CountrySubdivision::Malaysia_state_Perlis,
                CountrySubdivision::Malaysia_state_Pulau_Pinang,
                CountrySubdivision::Malaysia_state_Sabah,
                CountrySubdivision::Malaysia_state_Sarawak,
                CountrySubdivision::Malaysia_state_Selangor,
                CountrySubdivision::Malaysia_state_Terengganu
            ],
            self::Maldives => [
                CountrySubdivision::Maldives_administrative_atoll_Ariatholhu_Dhekunuburi,
                CountrySubdivision::Maldives_administrative_atoll_Ariatholhu_Uthuruburi,
                CountrySubdivision::Maldives_administrative_atoll_Faadhippolhu,
                CountrySubdivision::Maldives_administrative_atoll_Felidheatholhu,
                CountrySubdivision::Maldives_administrative_atoll_Fuvammulah,
                CountrySubdivision::Maldives_administrative_atoll_Hahdhunmathi,
                CountrySubdivision::Maldives_administrative_atoll_Huvadhuatholhu_Dhekunuburi,
                CountrySubdivision::Maldives_administrative_atoll_Huvadhuatholhu_Uthuruburi,
                CountrySubdivision::Maldives_administrative_atoll_Kolhumadulu,
                CountrySubdivision::Maldives_administrative_atoll_Maaleatholhu,
                CountrySubdivision::Maldives_administrative_atoll_Maalhosmadulu_Dhekunuburi,
                CountrySubdivision::Maldives_administrative_atoll_Maalhosmadulu_Uthuruburi,
                CountrySubdivision::Maldives_administrative_atoll_Miladhunmadulu_Dhekunuburi,
                CountrySubdivision::Maldives_administrative_atoll_Miladhunmadulu_Uthuruburi,
                CountrySubdivision::Maldives_administrative_atoll_Mulakatholhu,
                CountrySubdivision::Maldives_administrative_atoll_Nilandheatholhu_Dhekunuburi,
                CountrySubdivision::Maldives_administrative_atoll_Nilandheatholhu_Uthuruburi,
                CountrySubdivision::Maldives_administrative_atoll_North_Thiladhunmathi,
                CountrySubdivision::Maldives_administrative_atoll_South_Thiladhunmathi,
                CountrySubdivision::Maldives_city_Addu,
                CountrySubdivision::Maldives_city_Maale
            ],
            self::Mali => [
                CountrySubdivision::Mali_district_Bamako,
                CountrySubdivision::Mali_region_Gao,
                CountrySubdivision::Mali_region_Kayes,
                CountrySubdivision::Mali_region_Kidal,
                CountrySubdivision::Mali_region_Koulikoro,
                CountrySubdivision::Mali_region_Menaka,
                CountrySubdivision::Mali_region_Mopti,
                CountrySubdivision::Mali_region_Segou,
                CountrySubdivision::Mali_region_Sikasso,
                CountrySubdivision::Mali_region_Taoudenit,
                CountrySubdivision::Mali_region_Tombouctou
            ],
            self::Malta => [
                CountrySubdivision::Malta_local_council_Attard,
                CountrySubdivision::Malta_local_council_Balzan,
                CountrySubdivision::Malta_local_council_Birgu,
                CountrySubdivision::Malta_local_council_Birkirkara,
                CountrySubdivision::Malta_local_council_Birzebbuga,
                CountrySubdivision::Malta_local_council_Bormla,
                CountrySubdivision::Malta_local_council_Dingli,
                CountrySubdivision::Malta_local_council_Fgura,
                CountrySubdivision::Malta_local_council_Floriana,
                CountrySubdivision::Malta_local_council_Fontana,
                CountrySubdivision::Malta_local_council_Ghajnsielem,
                CountrySubdivision::Malta_local_council_Gharb,
                CountrySubdivision::Malta_local_council_Gharghur,
                CountrySubdivision::Malta_local_council_Ghasri,
                CountrySubdivision::Malta_local_council_Ghaxaq,
                CountrySubdivision::Malta_local_council_Gudja,
                CountrySubdivision::Malta_local_council_Gzira,
                CountrySubdivision::Malta_local_council_Hamrun,
                CountrySubdivision::Malta_local_council_Iklin,
                CountrySubdivision::Malta_local_council_Isla,
                CountrySubdivision::Malta_local_council_Kalkara,
                CountrySubdivision::Malta_local_council_Kercem,
                CountrySubdivision::Malta_local_council_Kirkop,
                CountrySubdivision::Malta_local_council_Lija,
                CountrySubdivision::Malta_local_council_Luqa,
                CountrySubdivision::Malta_local_council_Marsa,
                CountrySubdivision::Malta_local_council_Marsaskala,
                CountrySubdivision::Malta_local_council_Marsaxlokk,
                CountrySubdivision::Malta_local_council_Mdina,
                CountrySubdivision::Malta_local_council_Mellieha,
                CountrySubdivision::Malta_local_council_Mgarr,
                CountrySubdivision::Malta_local_council_Mosta,
                CountrySubdivision::Malta_local_council_Mqabba,
                CountrySubdivision::Malta_local_council_Msida,
                CountrySubdivision::Malta_local_council_Mtarfa,
                CountrySubdivision::Malta_local_council_Munxar,
                CountrySubdivision::Malta_local_council_Nadur,
                CountrySubdivision::Malta_local_council_Naxxar,
                CountrySubdivision::Malta_local_council_Paola,
                CountrySubdivision::Malta_local_council_Pembroke,
                CountrySubdivision::Malta_local_council_Pieta,
                CountrySubdivision::Malta_local_council_Qala,
                CountrySubdivision::Malta_local_council_Qormi,
                CountrySubdivision::Malta_local_council_Qrendi,
                CountrySubdivision::Malta_local_council_Rabat_Gozo,
                CountrySubdivision::Malta_local_council_Rabat_Malta,
                CountrySubdivision::Malta_local_council_Safi,
                CountrySubdivision::Malta_local_council_Saint_John,
                CountrySubdivision::Malta_local_council_Saint_Julian_s,
                CountrySubdivision::Malta_local_council_Saint_Lawrence,
                CountrySubdivision::Malta_local_council_Saint_Lucia_s,
                CountrySubdivision::Malta_local_council_Saint_Paul_s_Bay,
                CountrySubdivision::Malta_local_council_Sannat,
                CountrySubdivision::Malta_local_council_Santa_Venera,
                CountrySubdivision::Malta_local_council_Siggiewi,
                CountrySubdivision::Malta_local_council_Sliema,
                CountrySubdivision::Malta_local_council_Swieqi,
                CountrySubdivision::Malta_local_council_Ta_Xbiex,
                CountrySubdivision::Malta_local_council_Tarxien,
                CountrySubdivision::Malta_local_council_Valletta,
                CountrySubdivision::Malta_local_council_Xaghra,
                CountrySubdivision::Malta_local_council_Xewkija,
                CountrySubdivision::Malta_local_council_Xghajra,
                CountrySubdivision::Malta_local_council_Zabbar,
                CountrySubdivision::Malta_local_council_Zebbug_Gozo,
                CountrySubdivision::Malta_local_council_Zebbug_Malta,
                CountrySubdivision::Malta_local_council_Zejtun,
                CountrySubdivision::Malta_local_council_Zurrieq
            ],
            self::Marshall_Islands => [
                CountrySubdivision::Marshall_Islands_chain_of_islands_Ralik_chain,
                CountrySubdivision::Marshall_Islands_chain_of_islands_Ratak_chain,
                CountrySubdivision::Marshall_Islands_municipality_Aelok,
                CountrySubdivision::Marshall_Islands_municipality_Aelonlaplap,
                CountrySubdivision::Marshall_Islands_municipality_Arno,
                CountrySubdivision::Marshall_Islands_municipality_Aur,
                CountrySubdivision::Marshall_Islands_municipality_Bikini_Kili,
                CountrySubdivision::Marshall_Islands_municipality_Ebon,
                CountrySubdivision::Marshall_Islands_municipality_Ellep,
                CountrySubdivision::Marshall_Islands_municipality_Enewetak_Ujelang,
                CountrySubdivision::Marshall_Islands_municipality_Jabat,
                CountrySubdivision::Marshall_Islands_municipality_Jaluit,
                CountrySubdivision::Marshall_Islands_municipality_Kuwajleen,
                CountrySubdivision::Marshall_Islands_municipality_Lae,
                CountrySubdivision::Marshall_Islands_municipality_Likiep,
                CountrySubdivision::Marshall_Islands_municipality_Majuro,
                CountrySubdivision::Marshall_Islands_municipality_Maloelap,
                CountrySubdivision::Marshall_Islands_municipality_Mejit,
                CountrySubdivision::Marshall_Islands_municipality_Mile,
                CountrySubdivision::Marshall_Islands_municipality_Namdrik,
                CountrySubdivision::Marshall_Islands_municipality_Namu,
                CountrySubdivision::Marshall_Islands_municipality_Rongelap,
                CountrySubdivision::Marshall_Islands_municipality_Ujae,
                CountrySubdivision::Marshall_Islands_municipality_Utrik,
                CountrySubdivision::Marshall_Islands_municipality_Wotho,
                CountrySubdivision::Marshall_Islands_municipality_Wotje
            ],
            self::Mauritania => [
                CountrySubdivision::Mauritania_region_Adrar,
                CountrySubdivision::Mauritania_region_Assaba,
                CountrySubdivision::Mauritania_region_Brakna,
                CountrySubdivision::Mauritania_region_Dakhlet_Nouadhibou,
                CountrySubdivision::Mauritania_region_Gorgol,
                CountrySubdivision::Mauritania_region_Guidimaka,
                CountrySubdivision::Mauritania_region_Hodh_ech_Chargui,
                CountrySubdivision::Mauritania_region_Hodh_el_Gharbi,
                CountrySubdivision::Mauritania_region_Inchiri,
                CountrySubdivision::Mauritania_region_Nouakchott_Nord,
                CountrySubdivision::Mauritania_region_Nouakchott_Ouest,
                CountrySubdivision::Mauritania_region_Nouakchott_Sud,
                CountrySubdivision::Mauritania_region_Tagant,
                CountrySubdivision::Mauritania_region_Tiris_Zemmour,
                CountrySubdivision::Mauritania_region_Trarza
            ],
            self::Mauritius => [
                CountrySubdivision::Mauritius_dependency_Agalega_Islands,
                CountrySubdivision::Mauritius_dependency_Cargados_Carajos_Shoals,
                CountrySubdivision::Mauritius_dependency_Rodrigues_Island,
                CountrySubdivision::Mauritius_district_Black_River,
                CountrySubdivision::Mauritius_district_Flacq,
                CountrySubdivision::Mauritius_district_Grand_Port,
                CountrySubdivision::Mauritius_district_Moka,
                CountrySubdivision::Mauritius_district_Pamplemousses,
                CountrySubdivision::Mauritius_district_Plaines_Wilhems,
                CountrySubdivision::Mauritius_district_Port_Louis,
                CountrySubdivision::Mauritius_district_Riviere_du_Rempart,
                CountrySubdivision::Mauritius_district_Savanne
            ],
            self::Mexico => [
                CountrySubdivision::Mexico_federal_entity_Ciudad_de_Mexico,
                CountrySubdivision::Mexico_state_Aguascalientes,
                CountrySubdivision::Mexico_state_Baja_California,
                CountrySubdivision::Mexico_state_Baja_California_Sur,
                CountrySubdivision::Mexico_state_Campeche,
                CountrySubdivision::Mexico_state_Chiapas,
                CountrySubdivision::Mexico_state_Chihuahua,
                CountrySubdivision::Mexico_state_Coahuila_de_Zaragoza,
                CountrySubdivision::Mexico_state_Colima,
                CountrySubdivision::Mexico_state_Durango,
                CountrySubdivision::Mexico_state_Guanajuato,
                CountrySubdivision::Mexico_state_Guerrero,
                CountrySubdivision::Mexico_state_Hidalgo,
                CountrySubdivision::Mexico_state_Jalisco,
                CountrySubdivision::Mexico_state_Mexico,
                CountrySubdivision::Mexico_state_Michoacan_de_Ocampo,
                CountrySubdivision::Mexico_state_Morelos,
                CountrySubdivision::Mexico_state_Nayarit,
                CountrySubdivision::Mexico_state_Nuevo_Leon,
                CountrySubdivision::Mexico_state_Oaxaca,
                CountrySubdivision::Mexico_state_Puebla,
                CountrySubdivision::Mexico_state_Queretaro,
                CountrySubdivision::Mexico_state_Quintana_Roo,
                CountrySubdivision::Mexico_state_San_Luis_Potosi,
                CountrySubdivision::Mexico_state_Sinaloa,
                CountrySubdivision::Mexico_state_Sonora,
                CountrySubdivision::Mexico_state_Tabasco,
                CountrySubdivision::Mexico_state_Tamaulipas,
                CountrySubdivision::Mexico_state_Tlaxcala,
                CountrySubdivision::Mexico_state_Veracruz_de_Ignacio_de_la_Llave,
                CountrySubdivision::Mexico_state_Yucatan,
                CountrySubdivision::Mexico_state_Zacatecas
            ],
            self::Micronesia => [
                CountrySubdivision::Micronesia_state_Chuuk,
                CountrySubdivision::Micronesia_state_Kosrae,
                CountrySubdivision::Micronesia_state_Pohnpei,
                CountrySubdivision::Micronesia_state_Yap
            ],
            self::Moldova => [
                CountrySubdivision::Moldova_autonomous_territorial_unit_Gagauzia_Unitatea_teritoriala_autonoma_UTAG,
                CountrySubdivision::Moldova_city_Balti,
                CountrySubdivision::Moldova_city_Bender_Tighina,
                CountrySubdivision::Moldova_city_Chisinau,
                CountrySubdivision::Moldova_district_Anenii_Noi,
                CountrySubdivision::Moldova_district_Basarabeasca,
                CountrySubdivision::Moldova_district_Briceni,
                CountrySubdivision::Moldova_district_Cahul,
                CountrySubdivision::Moldova_district_Calarasi,
                CountrySubdivision::Moldova_district_Cantemir,
                CountrySubdivision::Moldova_district_Causeni,
                CountrySubdivision::Moldova_district_Cimislia,
                CountrySubdivision::Moldova_district_Criuleni,
                CountrySubdivision::Moldova_district_Donduseni,
                CountrySubdivision::Moldova_district_Drochia,
                CountrySubdivision::Moldova_district_Dubasari,
                CountrySubdivision::Moldova_district_Edinet,
                CountrySubdivision::Moldova_district_Falesti,
                CountrySubdivision::Moldova_district_Floresti,
                CountrySubdivision::Moldova_district_Glodeni,
                CountrySubdivision::Moldova_district_Hincesti,
                CountrySubdivision::Moldova_district_Ialoveni,
                CountrySubdivision::Moldova_district_Leova,
                CountrySubdivision::Moldova_district_Nisporeni,
                CountrySubdivision::Moldova_district_Ocnita,
                CountrySubdivision::Moldova_district_Orhei,
                CountrySubdivision::Moldova_district_Rezina,
                CountrySubdivision::Moldova_district_Riscani,
                CountrySubdivision::Moldova_district_Singerei,
                CountrySubdivision::Moldova_district_Soldanesti,
                CountrySubdivision::Moldova_district_Soroca,
                CountrySubdivision::Moldova_district_Stefan_Voda,
                CountrySubdivision::Moldova_district_Straseni,
                CountrySubdivision::Moldova_district_Taraclia,
                CountrySubdivision::Moldova_district_Telenesti,
                CountrySubdivision::Moldova_district_Ungheni,
                CountrySubdivision::Moldova_territorial_unit_Stinga_Nistrului_unitatea_teritoriala_din
            ],
            self::Monaco => [
                CountrySubdivision::Monaco_quarter_Fontvieille,
                CountrySubdivision::Monaco_quarter_Jardin_Exotique,
                CountrySubdivision::Monaco_quarter_La_Colle,
                CountrySubdivision::Monaco_quarter_La_Condamine,
                CountrySubdivision::Monaco_quarter_La_Gare,
                CountrySubdivision::Monaco_quarter_La_Source,
                CountrySubdivision::Monaco_quarter_Larvotto,
                CountrySubdivision::Monaco_quarter_Malbousquet,
                CountrySubdivision::Monaco_quarter_Monaco_Ville,
                CountrySubdivision::Monaco_quarter_Moneghetti,
                CountrySubdivision::Monaco_quarter_Monte_Carlo,
                CountrySubdivision::Monaco_quarter_Moulins,
                CountrySubdivision::Monaco_quarter_Port_Hercule,
                CountrySubdivision::Monaco_quarter_Saint_Roman,
                CountrySubdivision::Monaco_quarter_Sainte_Devote,
                CountrySubdivision::Monaco_quarter_Spelugues,
                CountrySubdivision::Monaco_quarter_Vallon_de_la_Rousse
            ],
            self::Mongolia => [
                CountrySubdivision::Mongolia_capital_city_Ulaanbaatar,
                CountrySubdivision::Mongolia_province_Arhangay,
                CountrySubdivision::Mongolia_province_Bayan_Olgiy,
                CountrySubdivision::Mongolia_province_Bayanhongor,
                CountrySubdivision::Mongolia_province_Bulgan,
                CountrySubdivision::Mongolia_province_Darhan_uul,
                CountrySubdivision::Mongolia_province_Dornod,
                CountrySubdivision::Mongolia_province_Dornogovi,
                CountrySubdivision::Mongolia_province_Dundgovi,
                CountrySubdivision::Mongolia_province_Dzavhan,
                CountrySubdivision::Mongolia_province_Govi_Altay,
                CountrySubdivision::Mongolia_province_Govi_Sumber,
                CountrySubdivision::Mongolia_province_Hentiy,
                CountrySubdivision::Mongolia_province_Hovd,
                CountrySubdivision::Mongolia_province_Hovsgol,
                CountrySubdivision::Mongolia_province_Omnogovi,
                CountrySubdivision::Mongolia_province_Orhon,
                CountrySubdivision::Mongolia_province_Ovorhangay,
                CountrySubdivision::Mongolia_province_Selenge,
                CountrySubdivision::Mongolia_province_Suhbaatar,
                CountrySubdivision::Mongolia_province_Tov,
                CountrySubdivision::Mongolia_province_Uvs
            ],
            self::Montenegro => [
                CountrySubdivision::Montenegro_municipality_Andrijevica,
                CountrySubdivision::Montenegro_municipality_Bar,
                CountrySubdivision::Montenegro_municipality_Berane,
                CountrySubdivision::Montenegro_municipality_Bijelo_Polje,
                CountrySubdivision::Montenegro_municipality_Budva,
                CountrySubdivision::Montenegro_municipality_Cetinje,
                CountrySubdivision::Montenegro_municipality_Danilovgrad,
                CountrySubdivision::Montenegro_municipality_Gusinje,
                CountrySubdivision::Montenegro_municipality_Herceg_Novi,
                CountrySubdivision::Montenegro_municipality_Kolasin,
                CountrySubdivision::Montenegro_municipality_Kotor,
                CountrySubdivision::Montenegro_municipality_Mojkovac,
                CountrySubdivision::Montenegro_municipality_Niksic,
                CountrySubdivision::Montenegro_municipality_Petnjica,
                CountrySubdivision::Montenegro_municipality_Plav,
                CountrySubdivision::Montenegro_municipality_Pljevlja,
                CountrySubdivision::Montenegro_municipality_Pluzine,
                CountrySubdivision::Montenegro_municipality_Podgorica,
                CountrySubdivision::Montenegro_municipality_Rozaje,
                CountrySubdivision::Montenegro_municipality_Savnik,
                CountrySubdivision::Montenegro_municipality_Tivat,
                CountrySubdivision::Montenegro_municipality_Tuzi,
                CountrySubdivision::Montenegro_municipality_Ulcinj,
                CountrySubdivision::Montenegro_municipality_Zabljak,
                CountrySubdivision::Montenegro_municipality_Zeta
            ],
            self::Morocco => [
                CountrySubdivision::Morocco_prefecture_Agadir_Ida_Ou_Tanane,
                CountrySubdivision::Morocco_prefecture_Casablanca,
                CountrySubdivision::Morocco_prefecture_Fes,
                CountrySubdivision::Morocco_prefecture_Inezgane_Ait_Melloul,
                CountrySubdivision::Morocco_prefecture_M_diq_Fnideq,
                CountrySubdivision::Morocco_prefecture_Marrakech,
                CountrySubdivision::Morocco_prefecture_Meknes,
                CountrySubdivision::Morocco_prefecture_Mohammadia,
                CountrySubdivision::Morocco_prefecture_Oujda_Angad,
                CountrySubdivision::Morocco_prefecture_Rabat,
                CountrySubdivision::Morocco_prefecture_Sale,
                CountrySubdivision::Morocco_prefecture_Skhirate_Temara,
                CountrySubdivision::Morocco_prefecture_Tanger_Assilah,
                CountrySubdivision::Morocco_province_Al_Haouz,
                CountrySubdivision::Morocco_province_Al_Hoceima,
                CountrySubdivision::Morocco_province_Aousserd_EH,
                CountrySubdivision::Morocco_province_Assa_Zag_EH_partial,
                CountrySubdivision::Morocco_province_Azilal,
                CountrySubdivision::Morocco_province_Beni_Mellal,
                CountrySubdivision::Morocco_province_Benslimane,
                CountrySubdivision::Morocco_province_Berkane,
                CountrySubdivision::Morocco_province_Berrechid,
                CountrySubdivision::Morocco_province_Boujdour_EH,
                CountrySubdivision::Morocco_province_Boulemane,
                CountrySubdivision::Morocco_province_Chefchaouen,
                CountrySubdivision::Morocco_province_Chichaoua,
                CountrySubdivision::Morocco_province_Chtouka_Ait_Baha,
                CountrySubdivision::Morocco_province_Driouch,
                CountrySubdivision::Morocco_province_El_Hajeb,
                CountrySubdivision::Morocco_province_El_Jadida,
                CountrySubdivision::Morocco_province_El_Kelaa_des_Sraghna,
                CountrySubdivision::Morocco_province_Errachidia,
                CountrySubdivision::Morocco_province_Es_Semara_EH_partial,
                CountrySubdivision::Morocco_province_Essaouira,
                CountrySubdivision::Morocco_province_Fahs_Anjra,
                CountrySubdivision::Morocco_province_Figuig,
                CountrySubdivision::Morocco_province_Fquih_Ben_Salah,
                CountrySubdivision::Morocco_province_Guelmim,
                CountrySubdivision::Morocco_province_Guercif,
                CountrySubdivision::Morocco_province_Ifrane,
                CountrySubdivision::Morocco_province_Jerada,
                CountrySubdivision::Morocco_province_Kenitra,
                CountrySubdivision::Morocco_province_Khemisset,
                CountrySubdivision::Morocco_province_Khenifra,
                CountrySubdivision::Morocco_province_Khouribga,
                CountrySubdivision::Morocco_province_Laayoune_EH,
                CountrySubdivision::Morocco_province_Larache,
                CountrySubdivision::Morocco_province_Mediouna,
                CountrySubdivision::Morocco_province_Midelt,
                CountrySubdivision::Morocco_province_Moulay_Yacoub,
                CountrySubdivision::Morocco_province_Nador,
                CountrySubdivision::Morocco_province_Nouaceur,
                CountrySubdivision::Morocco_province_Ouarzazate,
                CountrySubdivision::Morocco_province_Oued_Ed_Dahab_EH,
                CountrySubdivision::Morocco_province_Ouezzane,
                CountrySubdivision::Morocco_province_Rehamna,
                CountrySubdivision::Morocco_province_Safi,
                CountrySubdivision::Morocco_province_Sefrou,
                CountrySubdivision::Morocco_province_Settat,
                CountrySubdivision::Morocco_province_Sidi_Bennour,
                CountrySubdivision::Morocco_province_Sidi_Ifni,
                CountrySubdivision::Morocco_province_Sidi_Kacem,
                CountrySubdivision::Morocco_province_Sidi_Slimane,
                CountrySubdivision::Morocco_province_Tan_Tan_EH_partial,
                CountrySubdivision::Morocco_province_Taounate,
                CountrySubdivision::Morocco_province_Taourirt,
                CountrySubdivision::Morocco_province_Tarfaya_EH_partial,
                CountrySubdivision::Morocco_province_Taroudannt,
                CountrySubdivision::Morocco_province_Tata,
                CountrySubdivision::Morocco_province_Taza,
                CountrySubdivision::Morocco_province_Tetouan,
                CountrySubdivision::Morocco_province_Tinghir,
                CountrySubdivision::Morocco_province_Tiznit,
                CountrySubdivision::Morocco_province_Youssoufia,
                CountrySubdivision::Morocco_province_Zagora,
                CountrySubdivision::Morocco_region_Beni_Mellal_Khenifra,
                CountrySubdivision::Morocco_region_Casablanca_Settat,
                CountrySubdivision::Morocco_region_Dakhla_Oued_Ed_Dahab_EH,
                CountrySubdivision::Morocco_region_Draa_Tafilalet,
                CountrySubdivision::Morocco_region_Fes_Meknes,
                CountrySubdivision::Morocco_region_Guelmim_Oued_Noun_EH_partial,
                CountrySubdivision::Morocco_region_L_Oriental,
                CountrySubdivision::Morocco_region_Laayoune_Sakia_El_Hamra_EH_partial,
                CountrySubdivision::Morocco_region_Marrakech_Safi,
                CountrySubdivision::Morocco_region_Rabat_Sale_Kenitra,
                CountrySubdivision::Morocco_region_Souss_Massa,
                CountrySubdivision::Morocco_region_Tanger_Tetouan_Al_Hoceima
            ],
            self::Mozambique => [
                CountrySubdivision::Mozambique_city_Maputo,
                CountrySubdivision::Mozambique_province_Cabo_Delgado,
                CountrySubdivision::Mozambique_province_Gaza,
                CountrySubdivision::Mozambique_province_Inhambane,
                CountrySubdivision::Mozambique_province_Manica,
                CountrySubdivision::Mozambique_province_Maputo,
                CountrySubdivision::Mozambique_province_Nampula,
                CountrySubdivision::Mozambique_province_Niassa,
                CountrySubdivision::Mozambique_province_Sofala,
                CountrySubdivision::Mozambique_province_Tete,
                CountrySubdivision::Mozambique_province_Zambezia
            ],
            self::Myanmar => [
                CountrySubdivision::Myanmar_Burma_region_Ayeyarwady,
                CountrySubdivision::Myanmar_Burma_region_Bago,
                CountrySubdivision::Myanmar_Burma_region_Magway,
                CountrySubdivision::Myanmar_Burma_region_Mandalay,
                CountrySubdivision::Myanmar_Burma_region_Sagaing,
                CountrySubdivision::Myanmar_Burma_region_Tanintharyi,
                CountrySubdivision::Myanmar_Burma_region_Yangon,
                CountrySubdivision::Myanmar_Burma_state_Chin,
                CountrySubdivision::Myanmar_Burma_state_Kachin,
                CountrySubdivision::Myanmar_Burma_state_Kayah,
                CountrySubdivision::Myanmar_Burma_state_Kayin,
                CountrySubdivision::Myanmar_Burma_state_Mon,
                CountrySubdivision::Myanmar_Burma_state_Rakhine,
                CountrySubdivision::Myanmar_Burma_state_Shan,
                CountrySubdivision::Myanmar_Burma_union_territory_Nay_Pyi_Taw
            ],
            self::Namibia => [
                CountrySubdivision::Namibia_region_Erongo,
                CountrySubdivision::Namibia_region_Hardap,
                CountrySubdivision::Namibia_region_Karas,
                CountrySubdivision::Namibia_region_Kavango_East,
                CountrySubdivision::Namibia_region_Kavango_West,
                CountrySubdivision::Namibia_region_Khomas,
                CountrySubdivision::Namibia_region_Kunene,
                CountrySubdivision::Namibia_region_Ohangwena,
                CountrySubdivision::Namibia_region_Omaheke,
                CountrySubdivision::Namibia_region_Omusati,
                CountrySubdivision::Namibia_region_Oshana,
                CountrySubdivision::Namibia_region_Oshikoto,
                CountrySubdivision::Namibia_region_Otjozondjupa,
                CountrySubdivision::Namibia_region_Zambezi
            ],
            self::Nauru => [
                CountrySubdivision::Nauru_district_Aiwo,
                CountrySubdivision::Nauru_district_Anabar,
                CountrySubdivision::Nauru_district_Anetan,
                CountrySubdivision::Nauru_district_Anibare,
                CountrySubdivision::Nauru_district_Baitsi,
                CountrySubdivision::Nauru_district_Boe,
                CountrySubdivision::Nauru_district_Buada,
                CountrySubdivision::Nauru_district_Denigomodu,
                CountrySubdivision::Nauru_district_Ewa,
                CountrySubdivision::Nauru_district_Ijuw,
                CountrySubdivision::Nauru_district_Meneng,
                CountrySubdivision::Nauru_district_Nibok,
                CountrySubdivision::Nauru_district_Uaboe,
                CountrySubdivision::Nauru_district_Yaren
            ],
            self::Nepal => [
                CountrySubdivision::Nepal_province_Bagmati,
                CountrySubdivision::Nepal_province_Gandaki,
                CountrySubdivision::Nepal_province_Karnali,
                CountrySubdivision::Nepal_province_Koshi,
                CountrySubdivision::Nepal_province_Lumbini,
                CountrySubdivision::Nepal_province_Madhesh,
                CountrySubdivision::Nepal_province_Sudurpashchim
            ],
            self::Netherlands => [
                CountrySubdivision::Netherlands_country_Aruba,
                CountrySubdivision::Netherlands_country_Curacao,
                CountrySubdivision::Netherlands_country_Sint_Maarten,
                CountrySubdivision::Netherlands_province_Drenthe,
                CountrySubdivision::Netherlands_province_Flevoland,
                CountrySubdivision::Netherlands_province_Fryslan,
                CountrySubdivision::Netherlands_province_Gelderland,
                CountrySubdivision::Netherlands_province_Groningen,
                CountrySubdivision::Netherlands_province_Limburg,
                CountrySubdivision::Netherlands_province_Noord_Brabant,
                CountrySubdivision::Netherlands_province_Noord_Holland,
                CountrySubdivision::Netherlands_province_Overijssel,
                CountrySubdivision::Netherlands_province_Utrecht,
                CountrySubdivision::Netherlands_province_Zeeland,
                CountrySubdivision::Netherlands_province_Zuid_Holland,
                CountrySubdivision::Netherlands_special_municipality_Bonaire,
                CountrySubdivision::Netherlands_special_municipality_Saba,
                CountrySubdivision::Netherlands_special_municipality_Sint_Eustatius
            ],
            self::New_Zealand => [
                CountrySubdivision::New_Zealand_region_Auckland,
                CountrySubdivision::New_Zealand_region_Bay_of_Plenty,
                CountrySubdivision::New_Zealand_region_Canterbury,
                CountrySubdivision::New_Zealand_region_Gisborne,
                CountrySubdivision::New_Zealand_region_Greater_Wellington,
                CountrySubdivision::New_Zealand_region_Hawke_s_Bay,
                CountrySubdivision::New_Zealand_region_Manawatu_Whanganui,
                CountrySubdivision::New_Zealand_region_Marlborough,
                CountrySubdivision::New_Zealand_region_Nelson,
                CountrySubdivision::New_Zealand_region_Northland,
                CountrySubdivision::New_Zealand_region_Otago,
                CountrySubdivision::New_Zealand_region_Southland,
                CountrySubdivision::New_Zealand_region_Taranaki,
                CountrySubdivision::New_Zealand_region_Tasman,
                CountrySubdivision::New_Zealand_region_Te_Tai_o_Poutini,
                CountrySubdivision::New_Zealand_region_Waikato,
                CountrySubdivision::New_Zealand_special_island_authority_Chatham_Islands_Territory
            ],
            self::Nicaragua => [
                CountrySubdivision::Nicaragua_autonomous_region_Costa_Caribe_Norte,
                CountrySubdivision::Nicaragua_autonomous_region_Costa_Caribe_Sur,
                CountrySubdivision::Nicaragua_department_Boaco,
                CountrySubdivision::Nicaragua_department_Carazo,
                CountrySubdivision::Nicaragua_department_Chinandega,
                CountrySubdivision::Nicaragua_department_Chontales,
                CountrySubdivision::Nicaragua_department_Esteli,
                CountrySubdivision::Nicaragua_department_Granada,
                CountrySubdivision::Nicaragua_department_Jinotega,
                CountrySubdivision::Nicaragua_department_Leon,
                CountrySubdivision::Nicaragua_department_Madriz,
                CountrySubdivision::Nicaragua_department_Managua,
                CountrySubdivision::Nicaragua_department_Masaya,
                CountrySubdivision::Nicaragua_department_Matagalpa,
                CountrySubdivision::Nicaragua_department_Nueva_Segovia,
                CountrySubdivision::Nicaragua_department_Rio_San_Juan,
                CountrySubdivision::Nicaragua_department_Rivas
            ],
            self::Niger => [
                CountrySubdivision::Niger_region_Agadez,
                CountrySubdivision::Niger_region_Diffa,
                CountrySubdivision::Niger_region_Dosso,
                CountrySubdivision::Niger_region_Maradi,
                CountrySubdivision::Niger_region_Tahoua,
                CountrySubdivision::Niger_region_Tillaberi,
                CountrySubdivision::Niger_region_Zinder,
                CountrySubdivision::Niger_urban_community_Niamey
            ],
            self::Nigeria => [
                CountrySubdivision::Nigeria_capital_territory_Abuja_Federal_Capital_Territory,
                CountrySubdivision::Nigeria_state_Abia,
                CountrySubdivision::Nigeria_state_Adamawa,
                CountrySubdivision::Nigeria_state_Akwa_Ibom,
                CountrySubdivision::Nigeria_state_Anambra,
                CountrySubdivision::Nigeria_state_Bauchi,
                CountrySubdivision::Nigeria_state_Bayelsa,
                CountrySubdivision::Nigeria_state_Benue,
                CountrySubdivision::Nigeria_state_Borno,
                CountrySubdivision::Nigeria_state_Cross_River,
                CountrySubdivision::Nigeria_state_Delta,
                CountrySubdivision::Nigeria_state_Ebonyi,
                CountrySubdivision::Nigeria_state_Edo,
                CountrySubdivision::Nigeria_state_Ekiti,
                CountrySubdivision::Nigeria_state_Enugu,
                CountrySubdivision::Nigeria_state_Gombe,
                CountrySubdivision::Nigeria_state_Imo,
                CountrySubdivision::Nigeria_state_Jigawa,
                CountrySubdivision::Nigeria_state_Kaduna,
                CountrySubdivision::Nigeria_state_Kano,
                CountrySubdivision::Nigeria_state_Katsina,
                CountrySubdivision::Nigeria_state_Kebbi,
                CountrySubdivision::Nigeria_state_Kogi,
                CountrySubdivision::Nigeria_state_Kwara,
                CountrySubdivision::Nigeria_state_Lagos,
                CountrySubdivision::Nigeria_state_Nasarawa,
                CountrySubdivision::Nigeria_state_Niger,
                CountrySubdivision::Nigeria_state_Ogun,
                CountrySubdivision::Nigeria_state_Ondo,
                CountrySubdivision::Nigeria_state_Osun,
                CountrySubdivision::Nigeria_state_Oyo,
                CountrySubdivision::Nigeria_state_Plateau,
                CountrySubdivision::Nigeria_state_Rivers,
                CountrySubdivision::Nigeria_state_Sokoto,
                CountrySubdivision::Nigeria_state_Taraba,
                CountrySubdivision::Nigeria_state_Yobe,
                CountrySubdivision::Nigeria_state_Zamfara
            ],
            self::North_Macedonia => [
                CountrySubdivision::North_Macedonia_municipality_Aerodrom,
                CountrySubdivision::North_Macedonia_municipality_Aracinovo,
                CountrySubdivision::North_Macedonia_municipality_Berovo,
                CountrySubdivision::North_Macedonia_municipality_Bitola,
                CountrySubdivision::North_Macedonia_municipality_Bogdanci,
                CountrySubdivision::North_Macedonia_municipality_Bogovinje,
                CountrySubdivision::North_Macedonia_municipality_Bosilovo,
                CountrySubdivision::North_Macedonia_municipality_Brvenica,
                CountrySubdivision::North_Macedonia_municipality_Butel,
                CountrySubdivision::North_Macedonia_municipality_Cair,
                CountrySubdivision::North_Macedonia_municipality_Caska,
                CountrySubdivision::North_Macedonia_municipality_Centar,
                CountrySubdivision::North_Macedonia_municipality_Centar_Zupa,
                CountrySubdivision::North_Macedonia_municipality_Cesinovo_Oblesevo,
                CountrySubdivision::North_Macedonia_municipality_Cucer_Sandevo,
                CountrySubdivision::North_Macedonia_municipality_Debar,
                CountrySubdivision::North_Macedonia_municipality_Debrca,
                CountrySubdivision::North_Macedonia_municipality_Delcevo,
                CountrySubdivision::North_Macedonia_municipality_Demir_Hisar,
                CountrySubdivision::North_Macedonia_municipality_Demir_Kapija,
                CountrySubdivision::North_Macedonia_municipality_Dojran,
                CountrySubdivision::North_Macedonia_municipality_Dolneni,
                CountrySubdivision::North_Macedonia_municipality_Gazi_Baba,
                CountrySubdivision::North_Macedonia_municipality_Gevgelija,
                CountrySubdivision::North_Macedonia_municipality_Gjorce_Petrov,
                CountrySubdivision::North_Macedonia_municipality_Gostivar,
                CountrySubdivision::North_Macedonia_municipality_Gradsko,
                CountrySubdivision::North_Macedonia_municipality_Ilinden,
                CountrySubdivision::North_Macedonia_municipality_Jegunovce,
                CountrySubdivision::North_Macedonia_municipality_Karbinci,
                CountrySubdivision::North_Macedonia_municipality_Karpos,
                CountrySubdivision::North_Macedonia_municipality_Kavadarci,
                CountrySubdivision::North_Macedonia_municipality_Kicevo,
                CountrySubdivision::North_Macedonia_municipality_Kisela_Voda,
                CountrySubdivision::North_Macedonia_municipality_Kocani,
                CountrySubdivision::North_Macedonia_municipality_Konce,
                CountrySubdivision::North_Macedonia_municipality_Kratovo,
                CountrySubdivision::North_Macedonia_municipality_Kriva_Palanka,
                CountrySubdivision::North_Macedonia_municipality_Krivogastani,
                CountrySubdivision::North_Macedonia_municipality_Krusevo,
                CountrySubdivision::North_Macedonia_municipality_Kumanovo,
                CountrySubdivision::North_Macedonia_municipality_Lipkovo,
                CountrySubdivision::North_Macedonia_municipality_Lozovo,
                CountrySubdivision::North_Macedonia_municipality_Makedonska_Kamenica,
                CountrySubdivision::North_Macedonia_municipality_Makedonski_Brod,
                CountrySubdivision::North_Macedonia_municipality_Mavrovo_i_Rostuse,
                CountrySubdivision::North_Macedonia_municipality_Mogila,
                CountrySubdivision::North_Macedonia_municipality_Negotino,
                CountrySubdivision::North_Macedonia_municipality_Novaci,
                CountrySubdivision::North_Macedonia_municipality_Novo_Selo,
                CountrySubdivision::North_Macedonia_municipality_Ohrid,
                CountrySubdivision::North_Macedonia_municipality_Pehcevo,
                CountrySubdivision::North_Macedonia_municipality_Petrovec,
                CountrySubdivision::North_Macedonia_municipality_Plasnica,
                CountrySubdivision::North_Macedonia_municipality_Prilep,
                CountrySubdivision::North_Macedonia_municipality_Probistip,
                CountrySubdivision::North_Macedonia_municipality_Radovis,
                CountrySubdivision::North_Macedonia_municipality_Rankovce,
                CountrySubdivision::North_Macedonia_municipality_Resen,
                CountrySubdivision::North_Macedonia_municipality_Rosoman,
                CountrySubdivision::North_Macedonia_municipality_Saraj,
                CountrySubdivision::North_Macedonia_municipality_Sopiste,
                CountrySubdivision::North_Macedonia_municipality_Staro_Nagoricane,
                CountrySubdivision::North_Macedonia_municipality_Stip,
                CountrySubdivision::North_Macedonia_municipality_Struga,
                CountrySubdivision::North_Macedonia_municipality_Strumica,
                CountrySubdivision::North_Macedonia_municipality_Studenicani,
                CountrySubdivision::North_Macedonia_municipality_Suto_Orizari,
                CountrySubdivision::North_Macedonia_municipality_Sveti_Nikole,
                CountrySubdivision::North_Macedonia_municipality_Tearce,
                CountrySubdivision::North_Macedonia_municipality_Tetovo,
                CountrySubdivision::North_Macedonia_municipality_Valandovo,
                CountrySubdivision::North_Macedonia_municipality_Vasilevo,
                CountrySubdivision::North_Macedonia_municipality_Veles,
                CountrySubdivision::North_Macedonia_municipality_Vevcani,
                CountrySubdivision::North_Macedonia_municipality_Vinica,
                CountrySubdivision::North_Macedonia_municipality_Vrapciste,
                CountrySubdivision::North_Macedonia_municipality_Zelenikovo,
                CountrySubdivision::North_Macedonia_municipality_Zelino,
                CountrySubdivision::North_Macedonia_municipality_Zrnovci
            ],
            self::Norway => [
                CountrySubdivision::Norway_arctic_region_Jan_Mayen_Arctic_Region,
                CountrySubdivision::Norway_arctic_region_Svalbard_Arctic_Region,
                CountrySubdivision::Norway_county_Agder,
                CountrySubdivision::Norway_county_Innlandet,
                CountrySubdivision::Norway_county_More_og_Romsdal,
                CountrySubdivision::Norway_county_Nordland,
                CountrySubdivision::Norway_county_Oslo,
                CountrySubdivision::Norway_county_Rogaland,
                CountrySubdivision::Norway_county_Romssa_ja_Finnmarkku,
                CountrySubdivision::Norway_county_Troondelage,
                CountrySubdivision::Norway_county_Vestfold_og_Telemark,
                CountrySubdivision::Norway_county_Vestland,
                CountrySubdivision::Norway_county_Viken
            ],
            self::Oman => [
                CountrySubdivision::Oman_governorate_Ad_Dakhiliyah,
                CountrySubdivision::Oman_governorate_Al_Buraymi,
                CountrySubdivision::Oman_governorate_Al_Wusta,
                CountrySubdivision::Oman_governorate_Az_Zahirah,
                CountrySubdivision::Oman_governorate_Janub_al_Batinah,
                CountrySubdivision::Oman_governorate_Janub_ash_Sharqiyah,
                CountrySubdivision::Oman_governorate_Masqat,
                CountrySubdivision::Oman_governorate_Musandam,
                CountrySubdivision::Oman_governorate_Shamal_al_Batinah,
                CountrySubdivision::Oman_governorate_Shamal_ash_Sharqiyah,
                CountrySubdivision::Oman_governorate_Zufar
            ],
            self::Pakistan => [
                CountrySubdivision::Pakistan_Pakistan_administered_area_Azad_Jammu_and_Kashmir,
                CountrySubdivision::Pakistan_Pakistan_administered_area_Gilgit_Baltistan,
                CountrySubdivision::Pakistan_federal_capital_territory_Islamabad,
                CountrySubdivision::Pakistan_province_Balochistan,
                CountrySubdivision::Pakistan_province_Khaibar_Pakhtunkhwa,
                CountrySubdivision::Pakistan_province_Panjab,
                CountrySubdivision::Pakistan_province_Sindh
            ],
            self::Palau => [
                CountrySubdivision::Palau_state_Aimeliik,
                CountrySubdivision::Palau_state_Airai,
                CountrySubdivision::Palau_state_Angaur,
                CountrySubdivision::Palau_state_Hatohobei,
                CountrySubdivision::Palau_state_Kayangel,
                CountrySubdivision::Palau_state_Koror,
                CountrySubdivision::Palau_state_Melekeok,
                CountrySubdivision::Palau_state_Ngaraard,
                CountrySubdivision::Palau_state_Ngarchelong,
                CountrySubdivision::Palau_state_Ngardmau,
                CountrySubdivision::Palau_state_Ngatpang,
                CountrySubdivision::Palau_state_Ngchesar,
                CountrySubdivision::Palau_state_Ngeremlengui,
                CountrySubdivision::Palau_state_Ngiwal,
                CountrySubdivision::Palau_state_Peleliu,
                CountrySubdivision::Palau_state_Sonsorol
            ],
            self::Palestine => [
                CountrySubdivision::Palestinian_Territories_governorate_Al_Khalil,
                CountrySubdivision::Palestinian_Territories_governorate_Al_Quds,
                CountrySubdivision::Palestinian_Territories_governorate_Ariha_wal_Aghwar,
                CountrySubdivision::Palestinian_Territories_governorate_Bayt_Lahm,
                CountrySubdivision::Palestinian_Territories_governorate_Dayr_al_Balah,
                CountrySubdivision::Palestinian_Territories_governorate_Gaza,
                CountrySubdivision::Palestinian_Territories_governorate_Janin,
                CountrySubdivision::Palestinian_Territories_governorate_Khan_Yunis,
                CountrySubdivision::Palestinian_Territories_governorate_Nablus,
                CountrySubdivision::Palestinian_Territories_governorate_North_Gaza,
                CountrySubdivision::Palestinian_Territories_governorate_Qalqilya,
                CountrySubdivision::Palestinian_Territories_governorate_Rafah,
                CountrySubdivision::Palestinian_Territories_governorate_Ramallah,
                CountrySubdivision::Palestinian_Territories_governorate_Salfit,
                CountrySubdivision::Palestinian_Territories_governorate_Tubas,
                CountrySubdivision::Palestinian_Territories_governorate_Tulkarm
            ],
            self::Panama => [
                CountrySubdivision::Panama_indigenous_region_Embera,
                CountrySubdivision::Panama_indigenous_region_Guna_Yala,
                CountrySubdivision::Panama_indigenous_region_Naso_Tjer_Di,
                CountrySubdivision::Panama_indigenous_region_Ngabe_Bugle,
                CountrySubdivision::Panama_province_Bocas_del_Toro,
                CountrySubdivision::Panama_province_Chiriqui,
                CountrySubdivision::Panama_province_Cocle,
                CountrySubdivision::Panama_province_Colon,
                CountrySubdivision::Panama_province_Darien,
                CountrySubdivision::Panama_province_Herrera,
                CountrySubdivision::Panama_province_Los_Santos,
                CountrySubdivision::Panama_province_Panama,
                CountrySubdivision::Panama_province_Panama_Oeste,
                CountrySubdivision::Panama_province_Veraguas
            ],
            self::Papua_New_Guinea => [
                CountrySubdivision::Papua_New_Guinea_autonomous_region_Bougainville,
                CountrySubdivision::Papua_New_Guinea_district_National_Capital_District_Port_Moresby,
                CountrySubdivision::Papua_New_Guinea_province_Central,
                CountrySubdivision::Papua_New_Guinea_province_Chimbu,
                CountrySubdivision::Papua_New_Guinea_province_East_New_Britain,
                CountrySubdivision::Papua_New_Guinea_province_East_Sepik,
                CountrySubdivision::Papua_New_Guinea_province_Eastern_Highlands,
                CountrySubdivision::Papua_New_Guinea_province_Enga,
                CountrySubdivision::Papua_New_Guinea_province_Gulf,
                CountrySubdivision::Papua_New_Guinea_province_Hela,
                CountrySubdivision::Papua_New_Guinea_province_Jiwaka,
                CountrySubdivision::Papua_New_Guinea_province_Madang,
                CountrySubdivision::Papua_New_Guinea_province_Manus,
                CountrySubdivision::Papua_New_Guinea_province_Milne_Bay,
                CountrySubdivision::Papua_New_Guinea_province_Morobe,
                CountrySubdivision::Papua_New_Guinea_province_New_Ireland,
                CountrySubdivision::Papua_New_Guinea_province_Northern,
                CountrySubdivision::Papua_New_Guinea_province_Southern_Highlands,
                CountrySubdivision::Papua_New_Guinea_province_West_New_Britain,
                CountrySubdivision::Papua_New_Guinea_province_West_Sepik,
                CountrySubdivision::Papua_New_Guinea_province_Western,
                CountrySubdivision::Papua_New_Guinea_province_Western_Highlands
            ],
            self::Paraguay => [
                CountrySubdivision::Paraguay_capital_Asuncion,
                CountrySubdivision::Paraguay_department_Alto_Paraguay,
                CountrySubdivision::Paraguay_department_Alto_Parana,
                CountrySubdivision::Paraguay_department_Amambay,
                CountrySubdivision::Paraguay_department_Boqueron,
                CountrySubdivision::Paraguay_department_Caaguazu,
                CountrySubdivision::Paraguay_department_Caazapa,
                CountrySubdivision::Paraguay_department_Canindeyu,
                CountrySubdivision::Paraguay_department_Central,
                CountrySubdivision::Paraguay_department_Concepcion,
                CountrySubdivision::Paraguay_department_Cordillera,
                CountrySubdivision::Paraguay_department_Guaira,
                CountrySubdivision::Paraguay_department_Itapua,
                CountrySubdivision::Paraguay_department_Misiones,
                CountrySubdivision::Paraguay_department_Neembucu,
                CountrySubdivision::Paraguay_department_Paraguari,
                CountrySubdivision::Paraguay_department_Presidente_Hayes,
                CountrySubdivision::Paraguay_department_San_Pedro
            ],
            self::Peru => [
                CountrySubdivision::Peru_municipality_Lima_hatun_llaqta,
                CountrySubdivision::Peru_region_Amarumayu,
                CountrySubdivision::Peru_region_Ancash,
                CountrySubdivision::Peru_region_Apurimaq,
                CountrySubdivision::Peru_region_Arequipa,
                CountrySubdivision::Peru_region_Ayacucho,
                CountrySubdivision::Peru_region_Cajamarca,
                CountrySubdivision::Peru_region_Cusco,
                CountrySubdivision::Peru_region_El_Callao,
                CountrySubdivision::Peru_region_Huancavelica,
                CountrySubdivision::Peru_region_Huanuco,
                CountrySubdivision::Peru_region_Hunin,
                CountrySubdivision::Peru_region_Ica,
                CountrySubdivision::Peru_region_La_Libertad,
                CountrySubdivision::Peru_region_Lambayeque,
                CountrySubdivision::Peru_region_Lima,
                CountrySubdivision::Peru_region_Loreto,
                CountrySubdivision::Peru_region_Madre_de_Dios,
                CountrySubdivision::Peru_region_Moquegua,
                CountrySubdivision::Peru_region_Pasco,
                CountrySubdivision::Peru_region_Piura,
                CountrySubdivision::Peru_region_Puno,
                CountrySubdivision::Peru_region_San_Martin,
                CountrySubdivision::Peru_region_Tacna,
                CountrySubdivision::Peru_region_Tumbes,
                CountrySubdivision::Peru_region_Ucayali
            ],
            self::Philippines => [
                CountrySubdivision::Philippines_province_Abra,
                CountrySubdivision::Philippines_province_Agusan_del_Norte,
                CountrySubdivision::Philippines_province_Agusan_del_Sur,
                CountrySubdivision::Philippines_province_Aklan,
                CountrySubdivision::Philippines_province_Albay,
                CountrySubdivision::Philippines_province_Antike,
                CountrySubdivision::Philippines_province_Apayao,
                CountrySubdivision::Philippines_province_Aurora,
                CountrySubdivision::Philippines_province_Basilan,
                CountrySubdivision::Philippines_province_Bataan,
                CountrySubdivision::Philippines_province_Batanes,
                CountrySubdivision::Philippines_province_Batangas,
                CountrySubdivision::Philippines_province_Benget,
                CountrySubdivision::Philippines_province_Biliran,
                CountrySubdivision::Philippines_province_Bohol,
                CountrySubdivision::Philippines_province_Bukidnon,
                CountrySubdivision::Philippines_province_Bulacan,
                CountrySubdivision::Philippines_province_Cagayan,
                CountrySubdivision::Philippines_province_Camarines_Norte,
                CountrySubdivision::Philippines_province_Camarines_Sur,
                CountrySubdivision::Philippines_province_Camiguin,
                CountrySubdivision::Philippines_province_Capiz,
                CountrySubdivision::Philippines_province_Catanduanes,
                CountrySubdivision::Philippines_province_Cavite,
                CountrySubdivision::Philippines_province_Cebu,
                CountrySubdivision::Philippines_province_Cotabato,
                CountrySubdivision::Philippines_province_Davao_Occidental,
                CountrySubdivision::Philippines_province_Davao_Oriental,
                CountrySubdivision::Philippines_province_Davao_de_Oro,
                CountrySubdivision::Philippines_province_Davao_del_Norte,
                CountrySubdivision::Philippines_province_Davao_del_Sur,
                CountrySubdivision::Philippines_province_Dinagat_Islands,
                CountrySubdivision::Philippines_province_Eastern_Samar,
                CountrySubdivision::Philippines_province_Gimaras,
                CountrySubdivision::Philippines_province_Hilagang_Iloko,
                CountrySubdivision::Philippines_province_Hilagang_Lanaw,
                CountrySubdivision::Philippines_province_Hilagang_Magindanaw,
                CountrySubdivision::Philippines_province_Hilagang_Samar,
                CountrySubdivision::Philippines_province_Hilagang_Sambuwangga,
                CountrySubdivision::Philippines_province_Hilagang_Surigaw,
                CountrySubdivision::Philippines_province_Ifugao,
                CountrySubdivision::Philippines_province_Ilocos_Sur,
                CountrySubdivision::Philippines_province_Iloilo,
                CountrySubdivision::Philippines_province_Isabela,
                CountrySubdivision::Philippines_province_Kalinga,
                CountrySubdivision::Philippines_province_Kanlurang_Mindoro,
                CountrySubdivision::Philippines_province_Kanlurang_Misamis,
                CountrySubdivision::Philippines_province_Kanlurang_Negros,
                CountrySubdivision::Philippines_province_Katimogang_Leyte,
                CountrySubdivision::Philippines_province_Keson,
                CountrySubdivision::Philippines_province_Kirino,
                CountrySubdivision::Philippines_province_La_Union,
                CountrySubdivision::Philippines_province_Laguna,
                CountrySubdivision::Philippines_province_Lalawigang_Bulubundukin,
                CountrySubdivision::Philippines_province_Lanao_del_Sur,
                CountrySubdivision::Philippines_province_Leyte,
                CountrySubdivision::Philippines_province_Maguindanao_del_Sur,
                CountrySubdivision::Philippines_province_Marinduke,
                CountrySubdivision::Philippines_province_Masbate,
                CountrySubdivision::Philippines_province_Mindoro_Oriental,
                CountrySubdivision::Philippines_province_Misamis_Oriental,
                CountrySubdivision::Philippines_province_Negros_Oriental,
                CountrySubdivision::Philippines_province_Nueva_Ecija,
                CountrySubdivision::Philippines_province_Nueva_Vizcaya,
                CountrySubdivision::Philippines_province_Palawan,
                CountrySubdivision::Philippines_province_Pampanga,
                CountrySubdivision::Philippines_province_Pangasinan,
                CountrySubdivision::Philippines_province_Risal,
                CountrySubdivision::Philippines_province_Romblon,
                CountrySubdivision::Philippines_province_Samar,
                CountrySubdivision::Philippines_province_Sambales,
                CountrySubdivision::Philippines_province_Sambuwangga_Sibugay,
                CountrySubdivision::Philippines_province_Sarangani,
                CountrySubdivision::Philippines_province_Sikihor,
                CountrySubdivision::Philippines_province_Sorsogon,
                CountrySubdivision::Philippines_province_South_Cotabato,
                CountrySubdivision::Philippines_province_Sultan_Kudarat,
                CountrySubdivision::Philippines_province_Sulu,
                CountrySubdivision::Philippines_province_Surigao_del_Sur,
                CountrySubdivision::Philippines_province_Tarlac,
                CountrySubdivision::Philippines_province_Tawi_Tawi,
                CountrySubdivision::Philippines_province_Timog_Sambuwangga,
                CountrySubdivision::Philippines_region_Autonomous_Region_in_Muslim_Mindanao_ARMM,
                CountrySubdivision::Philippines_region_Bicol_Region_V,
                CountrySubdivision::Philippines_region_Cagayan_Valley_Region_II,
                CountrySubdivision::Philippines_region_Calabarzon_Region_IV_A,
                CountrySubdivision::Philippines_region_Caraga_Region_XIII,
                CountrySubdivision::Philippines_region_Central_Luzon_Region_III,
                CountrySubdivision::Philippines_region_Central_Visayas_Region_VII,
                CountrySubdivision::Philippines_region_Cordillera_Administrative_Region_CAR,
                CountrySubdivision::Philippines_region_Davao_Region_XI,
                CountrySubdivision::Philippines_region_Eastern_Visayas_Region_VIII,
                CountrySubdivision::Philippines_region_Ilocos_Region_I,
                CountrySubdivision::Philippines_region_Mimaropa_Region_IV_B,
                CountrySubdivision::Philippines_region_National_Capital_Region,
                CountrySubdivision::Philippines_region_Northern_Mindanao_Region_X,
                CountrySubdivision::Philippines_region_Rehiyon_ng_Kanlurang_Bisaya,
                CountrySubdivision::Philippines_region_Rehiyon_ng_Soccsksargen,
                CountrySubdivision::Philippines_region_Rehiyon_ng_Tangway_ng_Sambuwangga
            ],
            self::Poland => [
                CountrySubdivision::Poland_voivodship_Dolnoslaskie,
                CountrySubdivision::Poland_voivodship_Kujawsko_Pomorskie,
                CountrySubdivision::Poland_voivodship_Lodzkie,
                CountrySubdivision::Poland_voivodship_Lubelskie,
                CountrySubdivision::Poland_voivodship_Lubuskie,
                CountrySubdivision::Poland_voivodship_Malopolskie,
                CountrySubdivision::Poland_voivodship_Mazowieckie,
                CountrySubdivision::Poland_voivodship_Opolskie,
                CountrySubdivision::Poland_voivodship_Podkarpackie,
                CountrySubdivision::Poland_voivodship_Podlaskie,
                CountrySubdivision::Poland_voivodship_Pomorskie,
                CountrySubdivision::Poland_voivodship_Slaskie,
                CountrySubdivision::Poland_voivodship_Swietokrzyskie,
                CountrySubdivision::Poland_voivodship_Warminsko_Mazurskie,
                CountrySubdivision::Poland_voivodship_Wielkopolskie,
                CountrySubdivision::Poland_voivodship_Zachodniopomorskie
            ],
            self::Portugal => [
                CountrySubdivision::Portugal_autonomous_region_Regiao_Autonoma_da_Madeira,
                CountrySubdivision::Portugal_autonomous_region_Regiao_Autonoma_dos_Acores,
                CountrySubdivision::Portugal_district_Aveiro,
                CountrySubdivision::Portugal_district_Beja,
                CountrySubdivision::Portugal_district_Braga,
                CountrySubdivision::Portugal_district_Braganca,
                CountrySubdivision::Portugal_district_Castelo_Branco,
                CountrySubdivision::Portugal_district_Coimbra,
                CountrySubdivision::Portugal_district_Evora,
                CountrySubdivision::Portugal_district_Faro,
                CountrySubdivision::Portugal_district_Guarda,
                CountrySubdivision::Portugal_district_Leiria,
                CountrySubdivision::Portugal_district_Lisboa,
                CountrySubdivision::Portugal_district_Portalegre,
                CountrySubdivision::Portugal_district_Porto,
                CountrySubdivision::Portugal_district_Santarem,
                CountrySubdivision::Portugal_district_Setubal,
                CountrySubdivision::Portugal_district_Viana_do_Castelo,
                CountrySubdivision::Portugal_district_Vila_Real,
                CountrySubdivision::Portugal_district_Viseu
            ],
            self::Qatar => [
                CountrySubdivision::Qatar_municipality_Ad_Dawhah,
                CountrySubdivision::Qatar_municipality_Al_Khawr_wa_adh_Dhakhirah,
                CountrySubdivision::Qatar_municipality_Al_Wakrah,
                CountrySubdivision::Qatar_municipality_Ar_Rayyan,
                CountrySubdivision::Qatar_municipality_Ash_Shamal,
                CountrySubdivision::Qatar_municipality_Ash_Shihaniyah,
                CountrySubdivision::Qatar_municipality_Az_Za_ayin,
                CountrySubdivision::Qatar_municipality_Umm_Salal
            ],
            self::Romania => [
                CountrySubdivision::Romania_department_Alba,
                CountrySubdivision::Romania_department_Arad,
                CountrySubdivision::Romania_department_Arges,
                CountrySubdivision::Romania_department_Bacau,
                CountrySubdivision::Romania_department_Bihor,
                CountrySubdivision::Romania_department_Bistrita_Nasaud,
                CountrySubdivision::Romania_department_Botosani,
                CountrySubdivision::Romania_department_Braila,
                CountrySubdivision::Romania_department_Brasov,
                CountrySubdivision::Romania_department_Buzau,
                CountrySubdivision::Romania_department_Calarasi,
                CountrySubdivision::Romania_department_Caras_Severin,
                CountrySubdivision::Romania_department_Cluj,
                CountrySubdivision::Romania_department_Constanta,
                CountrySubdivision::Romania_department_Covasna,
                CountrySubdivision::Romania_department_Dambovita,
                CountrySubdivision::Romania_department_Dolj,
                CountrySubdivision::Romania_department_Galati,
                CountrySubdivision::Romania_department_Giurgiu,
                CountrySubdivision::Romania_department_Gorj,
                CountrySubdivision::Romania_department_Harghita,
                CountrySubdivision::Romania_department_Hunedoara,
                CountrySubdivision::Romania_department_Ialomita,
                CountrySubdivision::Romania_department_Iasi,
                CountrySubdivision::Romania_department_Ilfov,
                CountrySubdivision::Romania_department_Maramures,
                CountrySubdivision::Romania_department_Mehedinti,
                CountrySubdivision::Romania_department_Mures,
                CountrySubdivision::Romania_department_Neamt,
                CountrySubdivision::Romania_department_Olt,
                CountrySubdivision::Romania_department_Prahova,
                CountrySubdivision::Romania_department_Salaj,
                CountrySubdivision::Romania_department_Satu_Mare,
                CountrySubdivision::Romania_department_Sibiu,
                CountrySubdivision::Romania_department_Suceava,
                CountrySubdivision::Romania_department_Teleorman,
                CountrySubdivision::Romania_department_Timis,
                CountrySubdivision::Romania_department_Tulcea,
                CountrySubdivision::Romania_department_Valcea,
                CountrySubdivision::Romania_department_Vaslui,
                CountrySubdivision::Romania_department_Vrancea,
                CountrySubdivision::Romania_municipality_Bucuresti
            ],
            self::Russian_Federation => [
                CountrySubdivision::Russia_administrative_region_Amurskaja_oblast,
                CountrySubdivision::Russia_administrative_region_Arhangel_skaja_oblast,
                CountrySubdivision::Russia_administrative_region_Astrahanskaja_oblast,
                CountrySubdivision::Russia_administrative_region_Belgorodskaja_oblast,
                CountrySubdivision::Russia_administrative_region_Brjanskaja_oblast,
                CountrySubdivision::Russia_administrative_region_Chelyabinskaya_oblast,
                CountrySubdivision::Russia_administrative_region_Irkutskaja_oblast,
                CountrySubdivision::Russia_administrative_region_Ivanovskaja_oblast,
                CountrySubdivision::Russia_administrative_region_Jaroslavskaja_oblast,
                CountrySubdivision::Russia_administrative_region_Kaliningradskaja_oblast,
                CountrySubdivision::Russia_administrative_region_Kaluzhskaya_oblast,
                CountrySubdivision::Russia_administrative_region_Kemerovskaja_oblast,
                CountrySubdivision::Russia_administrative_region_Kirovskaja_oblast,
                CountrySubdivision::Russia_administrative_region_Kostromskaja_oblast,
                CountrySubdivision::Russia_administrative_region_Kurganskaja_oblast,
                CountrySubdivision::Russia_administrative_region_Kurskaja_oblast,
                CountrySubdivision::Russia_administrative_region_Leningradskaja_oblast,
                CountrySubdivision::Russia_administrative_region_Lipeckaja_oblast,
                CountrySubdivision::Russia_administrative_region_Magadanskaja_oblast,
                CountrySubdivision::Russia_administrative_region_Moskovskaja_oblast,
                CountrySubdivision::Russia_administrative_region_Murmanskaja_oblast,
                CountrySubdivision::Russia_administrative_region_Nizhegorodskaya_oblast,
                CountrySubdivision::Russia_administrative_region_Novgorodskaja_oblast,
                CountrySubdivision::Russia_administrative_region_Novosibirskaja_oblast,
                CountrySubdivision::Russia_administrative_region_Omskaja_oblast,
                CountrySubdivision::Russia_administrative_region_Orenburgskaja_oblast,
                CountrySubdivision::Russia_administrative_region_Orlovskaja_oblast,
                CountrySubdivision::Russia_administrative_region_Penzenskaja_oblast,
                CountrySubdivision::Russia_administrative_region_Pskovskaja_oblast,
                CountrySubdivision::Russia_administrative_region_Rjazanskaja_oblast,
                CountrySubdivision::Russia_administrative_region_Rostovskaja_oblast,
                CountrySubdivision::Russia_administrative_region_Sahalinskaja_oblast,
                CountrySubdivision::Russia_administrative_region_Samarskaja_oblast,
                CountrySubdivision::Russia_administrative_region_Saratovskaja_oblast,
                CountrySubdivision::Russia_administrative_region_Smolenskaja_oblast,
                CountrySubdivision::Russia_administrative_region_Sverdlovskaja_oblast,
                CountrySubdivision::Russia_administrative_region_Tambovskaja_oblast,
                CountrySubdivision::Russia_administrative_region_Tjumenskaja_oblast,
                CountrySubdivision::Russia_administrative_region_Tomskaja_oblast,
                CountrySubdivision::Russia_administrative_region_Tul_skaja_oblast,
                CountrySubdivision::Russia_administrative_region_Tverskaja_oblast,
                CountrySubdivision::Russia_administrative_region_Ul_janovskaja_oblast,
                CountrySubdivision::Russia_administrative_region_Vladimirskaja_oblast,
                CountrySubdivision::Russia_administrative_region_Volgogradskaja_oblast,
                CountrySubdivision::Russia_administrative_region_Vologodskaja_oblast,
                CountrySubdivision::Russia_administrative_region_Voronezhskaya_oblast,
                CountrySubdivision::Russia_administrative_territory_Altajskij_kraj,
                CountrySubdivision::Russia_administrative_territory_Habarovskij_kraj,
                CountrySubdivision::Russia_administrative_territory_Kamchatskiy_kray,
                CountrySubdivision::Russia_administrative_territory_Krasnodarskij_kraj,
                CountrySubdivision::Russia_administrative_territory_Krasnojarskij_kraj,
                CountrySubdivision::Russia_administrative_territory_Permskij_kraj,
                CountrySubdivision::Russia_administrative_territory_Primorskij_kraj,
                CountrySubdivision::Russia_administrative_territory_Stavropol_skij_kraj,
                CountrySubdivision::Russia_administrative_territory_Zabajkal_skij_kraj,
                CountrySubdivision::Russia_autonomous_city_Moskva,
                CountrySubdivision::Russia_autonomous_city_Sankt_Peterburg,
                CountrySubdivision::Russia_autonomous_district_Chukotskiy_avtonomnyy_okrug,
                CountrySubdivision::Russia_autonomous_district_Hanty_Mansijskij_avtonomnyj_okrug,
                CountrySubdivision::Russia_autonomous_district_Jamalo_Neneckij_avtonomnyj_okrug,
                CountrySubdivision::Russia_autonomous_district_Neneckij_avtonomnyj_okrug,
                CountrySubdivision::Russia_autonomous_region_Evrejskaja_avtonomnaja_oblast,
                CountrySubdivision::Russia_republic_Adygeja_Respublika,
                CountrySubdivision::Russia_republic_Altaj_Respublika,
                CountrySubdivision::Russia_republic_Bashkortostan_Respublika,
                CountrySubdivision::Russia_republic_Burjatija_Respublika,
                CountrySubdivision::Russia_republic_Chechenskaya_Respublika,
                CountrySubdivision::Russia_republic_Chuvashskaya_Respublika,
                CountrySubdivision::Russia_republic_Dagestan_Respublika,
                CountrySubdivision::Russia_republic_Hakasija_Respublika,
                CountrySubdivision::Russia_republic_Ingushetiya_Respublika,
                CountrySubdivision::Russia_republic_Kabardino_Balkarskaja_Respublika,
                CountrySubdivision::Russia_republic_Kalmykija_Respublika,
                CountrySubdivision::Russia_republic_Karachayevo_Cherkesskaya_Respublika,
                CountrySubdivision::Russia_republic_Karelija_Respublika,
                CountrySubdivision::Russia_republic_Komi_Respublika,
                CountrySubdivision::Russia_republic_Marij_El_Respublika,
                CountrySubdivision::Russia_republic_Mordovija_Respublika,
                CountrySubdivision::Russia_republic_Saha_Respublika,
                CountrySubdivision::Russia_republic_Severnaja_Osetija_Respublika,
                CountrySubdivision::Russia_republic_Tatarstan_Respublika,
                CountrySubdivision::Russia_republic_Tyva_Respublika,
                CountrySubdivision::Russia_republic_Udmurtskaja_Respublika
            ],
            self::Rwanda => [
                CountrySubdivision::Rwanda_city_City_of_Kigali,
                CountrySubdivision::Rwanda_province_Amajyaruguru,
                CountrySubdivision::Rwanda_province_Amajyepfo,
                CountrySubdivision::Rwanda_province_Eastern,
                CountrySubdivision::Rwanda_province_Iburengerazuba
            ],
            self::Saint_Helena_Ascension_Tristan_da_Cunha => [
                CountrySubdivision::St_Helena_geographical_entity_Ascension,
                CountrySubdivision::St_Helena_geographical_entity_Saint_Helena,
                CountrySubdivision::St_Helena_geographical_entity_Tristan_da_Cunha
            ],
            self::Saint_Kitts_and_Nevis => [
                CountrySubdivision::St_Kitts_Nevis_parish_Christ_Church_Nichola_Town,
                CountrySubdivision::St_Kitts_Nevis_parish_Saint_Anne_Sandy_Point,
                CountrySubdivision::St_Kitts_Nevis_parish_Saint_George_Basseterre,
                CountrySubdivision::St_Kitts_Nevis_parish_Saint_George_Gingerland,
                CountrySubdivision::St_Kitts_Nevis_parish_Saint_James_Windward,
                CountrySubdivision::St_Kitts_Nevis_parish_Saint_John_Capisterre,
                CountrySubdivision::St_Kitts_Nevis_parish_Saint_John_Figtree,
                CountrySubdivision::St_Kitts_Nevis_parish_Saint_Mary_Cayon,
                CountrySubdivision::St_Kitts_Nevis_parish_Saint_Paul_Capisterre,
                CountrySubdivision::St_Kitts_Nevis_parish_Saint_Paul_Charlestown,
                CountrySubdivision::St_Kitts_Nevis_parish_Saint_Peter_Basseterre,
                CountrySubdivision::St_Kitts_Nevis_parish_Saint_Thomas_Lowland,
                CountrySubdivision::St_Kitts_Nevis_parish_Saint_Thomas_Middle_Island,
                CountrySubdivision::St_Kitts_Nevis_parish_Trinity_Palmetto_Point,
                CountrySubdivision::St_Kitts_Nevis_state_Nevis,
                CountrySubdivision::St_Kitts_Nevis_state_Saint_Kitts
            ],
            self::Saint_Lucia => [
                CountrySubdivision::St_Lucia_district_Anse_la_Raye,
                CountrySubdivision::St_Lucia_district_Canaries,
                CountrySubdivision::St_Lucia_district_Castries,
                CountrySubdivision::St_Lucia_district_Choiseul,
                CountrySubdivision::St_Lucia_district_Dennery,
                CountrySubdivision::St_Lucia_district_Gros_Islet,
                CountrySubdivision::St_Lucia_district_Laborie,
                CountrySubdivision::St_Lucia_district_Micoud,
                CountrySubdivision::St_Lucia_district_Soufriere,
                CountrySubdivision::St_Lucia_district_Vieux_Fort
            ],
            self::Saint_Vincent_and_the_Grenadines => [
                CountrySubdivision::St_Vincent_Grenadines_parish_Charlotte,
                CountrySubdivision::St_Vincent_Grenadines_parish_Grenadines,
                CountrySubdivision::St_Vincent_Grenadines_parish_Saint_Andrew,
                CountrySubdivision::St_Vincent_Grenadines_parish_Saint_David,
                CountrySubdivision::St_Vincent_Grenadines_parish_Saint_George,
                CountrySubdivision::St_Vincent_Grenadines_parish_Saint_Patrick
            ],
            self::Samoa => [
                CountrySubdivision::Samoa_district_A_ana,
                CountrySubdivision::Samoa_district_Aiga_i_le_Tai,
                CountrySubdivision::Samoa_district_Atua,
                CountrySubdivision::Samoa_district_Fa_asaleleaga,
                CountrySubdivision::Samoa_district_Gaga_emauga,
                CountrySubdivision::Samoa_district_Gagaifomauga,
                CountrySubdivision::Samoa_district_Palauli,
                CountrySubdivision::Samoa_district_Satupa_itea,
                CountrySubdivision::Samoa_district_Tuamasaga,
                CountrySubdivision::Samoa_district_Va_a_o_Fonoti,
                CountrySubdivision::Samoa_district_Vaisigano
            ],
            self::San_Marino => [
                CountrySubdivision::San_Marino_municipality_Acquaviva,
                CountrySubdivision::San_Marino_municipality_Borgo_Maggiore,
                CountrySubdivision::San_Marino_municipality_Chiesanuova,
                CountrySubdivision::San_Marino_municipality_Citta_di_San_Marino,
                CountrySubdivision::San_Marino_municipality_Domagnano,
                CountrySubdivision::San_Marino_municipality_Faetano,
                CountrySubdivision::San_Marino_municipality_Fiorentino,
                CountrySubdivision::San_Marino_municipality_Montegiardino,
                CountrySubdivision::San_Marino_municipality_Serravalle
            ],
            self::Sao_Tome_and_Principe => [
                CountrySubdivision::Sao_Tome_Principe_autonomous_region_Principe,
                CountrySubdivision::Sao_Tome_Principe_district_Agua_Grande,
                CountrySubdivision::Sao_Tome_Principe_district_Cantagalo,
                CountrySubdivision::Sao_Tome_Principe_district_Caue,
                CountrySubdivision::Sao_Tome_Principe_district_Lemba,
                CountrySubdivision::Sao_Tome_Principe_district_Lobata,
                CountrySubdivision::Sao_Tome_Principe_district_Me_Zochi
            ],
            self::Saudi_Arabia => [
                CountrySubdivision::Saudi_Arabia_region_Al_Bahah,
                CountrySubdivision::Saudi_Arabia_region_Al_Hudud_ash_Shamaliyah,
                CountrySubdivision::Saudi_Arabia_region_Al_Jawf,
                CountrySubdivision::Saudi_Arabia_region_Al_Madinah_al_Munawwarah,
                CountrySubdivision::Saudi_Arabia_region_Al_Qasim,
                CountrySubdivision::Saudi_Arabia_region_Ar_Riyad,
                CountrySubdivision::Saudi_Arabia_region_Ash_Sharqiyah,
                CountrySubdivision::Saudi_Arabia_region_Asir,
                CountrySubdivision::Saudi_Arabia_region_Ha_il,
                CountrySubdivision::Saudi_Arabia_region_Jazan,
                CountrySubdivision::Saudi_Arabia_region_Makkah_al_Mukarramah,
                CountrySubdivision::Saudi_Arabia_region_Najran,
                CountrySubdivision::Saudi_Arabia_region_Tabuk
            ],
            self::Senegal => [
                CountrySubdivision::Senegal_region_Dakar,
                CountrySubdivision::Senegal_region_Diourbel,
                CountrySubdivision::Senegal_region_Fatick,
                CountrySubdivision::Senegal_region_Kaffrine,
                CountrySubdivision::Senegal_region_Kaolack,
                CountrySubdivision::Senegal_region_Kedougou,
                CountrySubdivision::Senegal_region_Kolda,
                CountrySubdivision::Senegal_region_Louga,
                CountrySubdivision::Senegal_region_Matam,
                CountrySubdivision::Senegal_region_Saint_Louis,
                CountrySubdivision::Senegal_region_Sedhiou,
                CountrySubdivision::Senegal_region_Tambacounda,
                CountrySubdivision::Senegal_region_Thies,
                CountrySubdivision::Senegal_region_Ziguinchor
            ],
            self::Serbia => [
                CountrySubdivision::Serbia_autonomous_province_Kosovo_Metohija,
                CountrySubdivision::Serbia_autonomous_province_Vojvodina,
                CountrySubdivision::Serbia_city_Beograd,
                CountrySubdivision::Serbia_district_Borski_okrug,
                CountrySubdivision::Serbia_district_Branicevski_okrug,
                CountrySubdivision::Serbia_district_Jablanicki_okrug,
                CountrySubdivision::Serbia_district_Juznobacki_okrug,
                CountrySubdivision::Serbia_district_Juznobanatski_okrug,
                CountrySubdivision::Serbia_district_Kolubarski_okrug,
                CountrySubdivision::Serbia_district_Kosovski_okrug,
                CountrySubdivision::Serbia_district_Kosovsko_Mitrovacki_okrug,
                CountrySubdivision::Serbia_district_Kosovsko_Pomoravski_okrug,
                CountrySubdivision::Serbia_district_Macvanski_okrug,
                CountrySubdivision::Serbia_district_Moravicki_okrug,
                CountrySubdivision::Serbia_district_Nisavski_okrug,
                CountrySubdivision::Serbia_district_Pcinjski_okrug,
                CountrySubdivision::Serbia_district_Pecki_okrug,
                CountrySubdivision::Serbia_district_Pirotski_okrug,
                CountrySubdivision::Serbia_district_Podunavski_okrug,
                CountrySubdivision::Serbia_district_Pomoravski_okrug,
                CountrySubdivision::Serbia_district_Prizrenski_okrug,
                CountrySubdivision::Serbia_district_Rasinski_okrug,
                CountrySubdivision::Serbia_district_Raski_okrug,
                CountrySubdivision::Serbia_district_Severnobacki_okrug,
                CountrySubdivision::Serbia_district_Severnobanatski_okrug,
                CountrySubdivision::Serbia_district_Srednjebanatski_okrug,
                CountrySubdivision::Serbia_district_Sremski_okrug,
                CountrySubdivision::Serbia_district_Sumadijski_okrug,
                CountrySubdivision::Serbia_district_Toplicki_okrug,
                CountrySubdivision::Serbia_district_Zajecarski_okrug,
                CountrySubdivision::Serbia_district_Zapadnobacki_okrug,
                CountrySubdivision::Serbia_district_Zlatiborski_okrug
            ],
            self::Seychelles => [
                CountrySubdivision::Seychelles_district_Ans_Bwalo,
                CountrySubdivision::Seychelles_district_Ans_Etwal,
                CountrySubdivision::Seychelles_district_Ans_Royal,
                CountrySubdivision::Seychelles_district_Ans_o_Pen,
                CountrySubdivision::Seychelles_district_Au_Cap,
                CountrySubdivision::Seychelles_district_Baie_Lazare,
                CountrySubdivision::Seychelles_district_Baie_Sainte_Anne,
                CountrySubdivision::Seychelles_district_Beau_Vallon,
                CountrySubdivision::Seychelles_district_Bel_Air,
                CountrySubdivision::Seychelles_district_Bel_Ombre,
                CountrySubdivision::Seychelles_district_Cascade,
                CountrySubdivision::Seychelles_district_English_River,
                CountrySubdivision::Seychelles_district_Glacis,
                CountrySubdivision::Seychelles_district_Grand_Ans_Mae,
                CountrySubdivision::Seychelles_district_Grand_Ans_Pralen,
                CountrySubdivision::Seychelles_district_Ile_Perseverance_I,
                CountrySubdivision::Seychelles_district_Ile_Perseverance_II,
                CountrySubdivision::Seychelles_district_La_Digue,
                CountrySubdivision::Seychelles_district_Lemamel,
                CountrySubdivision::Seychelles_district_Mon_Bikston,
                CountrySubdivision::Seychelles_district_Mon_Fleri,
                CountrySubdivision::Seychelles_district_Plaisance,
                CountrySubdivision::Seychelles_district_Pointe_La_Rue,
                CountrySubdivision::Seychelles_district_Porglo,
                CountrySubdivision::Seychelles_district_Roche_Caiman,
                CountrySubdivision::Seychelles_district_Saint_Louis,
                CountrySubdivision::Seychelles_district_Takamaka
            ],
            self::Sierra_Leone => [
                CountrySubdivision::Sierra_Leone_area_Western_Area_Freetown,
                CountrySubdivision::Sierra_Leone_province_Eastern,
                CountrySubdivision::Sierra_Leone_province_North_Western,
                CountrySubdivision::Sierra_Leone_province_Northern,
                CountrySubdivision::Sierra_Leone_province_Southern
            ],
            self::Singapore => [
                CountrySubdivision::Singapore_district_Central_Singapore,
                CountrySubdivision::Singapore_district_North_East,
                CountrySubdivision::Singapore_district_North_West,
                CountrySubdivision::Singapore_district_South_East,
                CountrySubdivision::Singapore_district_South_West
            ],
            self::Slovakia => [
                CountrySubdivision::Slovakia_region_Banskobystricky_kraj,
                CountrySubdivision::Slovakia_region_Bratislavsky_kraj,
                CountrySubdivision::Slovakia_region_Kosicky_kraj,
                CountrySubdivision::Slovakia_region_Nitriansky_kraj,
                CountrySubdivision::Slovakia_region_Presovsky_kraj,
                CountrySubdivision::Slovakia_region_Trenciansky_kraj,
                CountrySubdivision::Slovakia_region_Trnavsky_kraj,
                CountrySubdivision::Slovakia_region_Zilinsky_kraj
            ],
            self::Slovenia => [
                CountrySubdivision::Slovenia_municipality_Ajdovscina,
                CountrySubdivision::Slovenia_municipality_Ankaran,
                CountrySubdivision::Slovenia_municipality_Apace,
                CountrySubdivision::Slovenia_municipality_Beltinci,
                CountrySubdivision::Slovenia_municipality_Benedikt,
                CountrySubdivision::Slovenia_municipality_Bistrica_ob_Sotli,
                CountrySubdivision::Slovenia_municipality_Bled,
                CountrySubdivision::Slovenia_municipality_Bloke,
                CountrySubdivision::Slovenia_municipality_Bohinj,
                CountrySubdivision::Slovenia_municipality_Borovnica,
                CountrySubdivision::Slovenia_municipality_Bovec,
                CountrySubdivision::Slovenia_municipality_Braslovce,
                CountrySubdivision::Slovenia_municipality_Brda,
                CountrySubdivision::Slovenia_municipality_Brezice,
                CountrySubdivision::Slovenia_municipality_Brezovica,
                CountrySubdivision::Slovenia_municipality_Cankova,
                CountrySubdivision::Slovenia_municipality_Cerklje_na_Gorenjskem,
                CountrySubdivision::Slovenia_municipality_Cerknica,
                CountrySubdivision::Slovenia_municipality_Cerkno,
                CountrySubdivision::Slovenia_municipality_Cerkvenjak,
                CountrySubdivision::Slovenia_municipality_Cirkulane,
                CountrySubdivision::Slovenia_municipality_Crensovci,
                CountrySubdivision::Slovenia_municipality_Crna_na_Koroskem,
                CountrySubdivision::Slovenia_municipality_Crnomelj,
                CountrySubdivision::Slovenia_municipality_Destrnik,
                CountrySubdivision::Slovenia_municipality_Divaca,
                CountrySubdivision::Slovenia_municipality_Dobje,
                CountrySubdivision::Slovenia_municipality_Dobrepolje,
                CountrySubdivision::Slovenia_municipality_Dobrna,
                CountrySubdivision::Slovenia_municipality_Dobrova_Polhov_Gradec,
                CountrySubdivision::Slovenia_municipality_Dobrovnik,
                CountrySubdivision::Slovenia_municipality_Dol_pri_Ljubljani,
                CountrySubdivision::Slovenia_municipality_Dolenjske_Toplice,
                CountrySubdivision::Slovenia_municipality_Domzale,
                CountrySubdivision::Slovenia_municipality_Dornava,
                CountrySubdivision::Slovenia_municipality_Dravograd,
                CountrySubdivision::Slovenia_municipality_Duplek,
                CountrySubdivision::Slovenia_municipality_Gorenja_vas_Poljane,
                CountrySubdivision::Slovenia_municipality_Gorisnica,
                CountrySubdivision::Slovenia_municipality_Gorje,
                CountrySubdivision::Slovenia_municipality_Gornja_Radgona,
                CountrySubdivision::Slovenia_municipality_Gornji_Grad,
                CountrySubdivision::Slovenia_municipality_Gornji_Petrovci,
                CountrySubdivision::Slovenia_municipality_Grad,
                CountrySubdivision::Slovenia_municipality_Grosuplje,
                CountrySubdivision::Slovenia_municipality_Hajdina,
                CountrySubdivision::Slovenia_municipality_Hoce_Slivnica,
                CountrySubdivision::Slovenia_municipality_Hodos,
                CountrySubdivision::Slovenia_municipality_Horjul,
                CountrySubdivision::Slovenia_municipality_Hrastnik,
                CountrySubdivision::Slovenia_municipality_Hrpelje_Kozina,
                CountrySubdivision::Slovenia_municipality_Idrija,
                CountrySubdivision::Slovenia_municipality_Ig,
                CountrySubdivision::Slovenia_municipality_Ilirska_Bistrica,
                CountrySubdivision::Slovenia_municipality_Ivancna_Gorica,
                CountrySubdivision::Slovenia_municipality_Izola,
                CountrySubdivision::Slovenia_municipality_Jesenice,
                CountrySubdivision::Slovenia_municipality_Jezersko,
                CountrySubdivision::Slovenia_municipality_Jursinci,
                CountrySubdivision::Slovenia_municipality_Kamnik,
                CountrySubdivision::Slovenia_municipality_Kanal_ob_Soci,
                CountrySubdivision::Slovenia_municipality_Kidricevo,
                CountrySubdivision::Slovenia_municipality_Kobarid,
                CountrySubdivision::Slovenia_municipality_Kobilje,
                CountrySubdivision::Slovenia_municipality_Kocevje,
                CountrySubdivision::Slovenia_municipality_Komen,
                CountrySubdivision::Slovenia_municipality_Komenda,
                CountrySubdivision::Slovenia_municipality_Kostanjevica_na_Krki,
                CountrySubdivision::Slovenia_municipality_Kostel,
                CountrySubdivision::Slovenia_municipality_Kozje,
                CountrySubdivision::Slovenia_municipality_Kranjska_Gora,
                CountrySubdivision::Slovenia_municipality_Krizevci,
                CountrySubdivision::Slovenia_municipality_Kungota,
                CountrySubdivision::Slovenia_municipality_Kuzma,
                CountrySubdivision::Slovenia_municipality_Lasko,
                CountrySubdivision::Slovenia_municipality_Lenart,
                CountrySubdivision::Slovenia_municipality_Lendava,
                CountrySubdivision::Slovenia_municipality_Litija,
                CountrySubdivision::Slovenia_municipality_Ljubno,
                CountrySubdivision::Slovenia_municipality_Ljutomer,
                CountrySubdivision::Slovenia_municipality_Log_Dragomer,
                CountrySubdivision::Slovenia_municipality_Logatec,
                CountrySubdivision::Slovenia_municipality_Loska_dolina,
                CountrySubdivision::Slovenia_municipality_Loski_Potok,
                CountrySubdivision::Slovenia_municipality_Lovrenc_na_Pohorju,
                CountrySubdivision::Slovenia_municipality_Luce,
                CountrySubdivision::Slovenia_municipality_Lukovica,
                CountrySubdivision::Slovenia_municipality_Majsperk,
                CountrySubdivision::Slovenia_municipality_Makole,
                CountrySubdivision::Slovenia_municipality_Markovci,
                CountrySubdivision::Slovenia_municipality_Medvode,
                CountrySubdivision::Slovenia_municipality_Menges,
                CountrySubdivision::Slovenia_municipality_Metlika,
                CountrySubdivision::Slovenia_municipality_Mezica,
                CountrySubdivision::Slovenia_municipality_Miklavz_na_Dravskem_polju,
                CountrySubdivision::Slovenia_municipality_Miren_Kostanjevica,
                CountrySubdivision::Slovenia_municipality_Mirna,
                CountrySubdivision::Slovenia_municipality_Mirna_Pec,
                CountrySubdivision::Slovenia_municipality_Mislinja,
                CountrySubdivision::Slovenia_municipality_Mokronog_Trebelno,
                CountrySubdivision::Slovenia_municipality_Moravce,
                CountrySubdivision::Slovenia_municipality_Moravske_Toplice,
                CountrySubdivision::Slovenia_municipality_Mozirje,
                CountrySubdivision::Slovenia_municipality_Muta,
                CountrySubdivision::Slovenia_municipality_Naklo,
                CountrySubdivision::Slovenia_municipality_Nazarje,
                CountrySubdivision::Slovenia_municipality_Odranci,
                CountrySubdivision::Slovenia_municipality_Oplotnica,
                CountrySubdivision::Slovenia_municipality_Ormoz,
                CountrySubdivision::Slovenia_municipality_Osilnica,
                CountrySubdivision::Slovenia_municipality_Pesnica,
                CountrySubdivision::Slovenia_municipality_Piran,
                CountrySubdivision::Slovenia_municipality_Pivka,
                CountrySubdivision::Slovenia_municipality_Podcetrtek,
                CountrySubdivision::Slovenia_municipality_Podlehnik,
                CountrySubdivision::Slovenia_municipality_Podvelka,
                CountrySubdivision::Slovenia_municipality_Poljcane,
                CountrySubdivision::Slovenia_municipality_Polzela,
                CountrySubdivision::Slovenia_municipality_Postojna,
                CountrySubdivision::Slovenia_municipality_Prebold,
                CountrySubdivision::Slovenia_municipality_Preddvor,
                CountrySubdivision::Slovenia_municipality_Prevalje,
                CountrySubdivision::Slovenia_municipality_Puconci,
                CountrySubdivision::Slovenia_municipality_Race_Fram,
                CountrySubdivision::Slovenia_municipality_Radece,
                CountrySubdivision::Slovenia_municipality_Radenci,
                CountrySubdivision::Slovenia_municipality_Radlje_ob_Dravi,
                CountrySubdivision::Slovenia_municipality_Radovljica,
                CountrySubdivision::Slovenia_municipality_Ravne_na_Koroskem,
                CountrySubdivision::Slovenia_municipality_Razkrizje,
                CountrySubdivision::Slovenia_municipality_Recica_ob_Savinji,
                CountrySubdivision::Slovenia_municipality_Rence_Vogrsko,
                CountrySubdivision::Slovenia_municipality_Ribnica,
                CountrySubdivision::Slovenia_municipality_Ribnica_na_Pohorju,
                CountrySubdivision::Slovenia_municipality_Rogaska_Slatina,
                CountrySubdivision::Slovenia_municipality_Rogasovci,
                CountrySubdivision::Slovenia_municipality_Rogatec,
                CountrySubdivision::Slovenia_municipality_Ruse,
                CountrySubdivision::Slovenia_municipality_Salovci,
                CountrySubdivision::Slovenia_municipality_Selnica_ob_Dravi,
                CountrySubdivision::Slovenia_municipality_Semic,
                CountrySubdivision::Slovenia_municipality_Sempeter_Vrtojba,
                CountrySubdivision::Slovenia_municipality_Sencur,
                CountrySubdivision::Slovenia_municipality_Sentilj,
                CountrySubdivision::Slovenia_municipality_Sentjernej,
                CountrySubdivision::Slovenia_municipality_Sentjur,
                CountrySubdivision::Slovenia_municipality_Sentrupert,
                CountrySubdivision::Slovenia_municipality_Sevnica,
                CountrySubdivision::Slovenia_municipality_Sezana,
                CountrySubdivision::Slovenia_municipality_Skocjan,
                CountrySubdivision::Slovenia_municipality_Skofja_Loka,
                CountrySubdivision::Slovenia_municipality_Skofljica,
                CountrySubdivision::Slovenia_municipality_Slovenska_Bistrica,
                CountrySubdivision::Slovenia_municipality_Slovenske_Konjice,
                CountrySubdivision::Slovenia_municipality_Smarje_pri_Jelsah,
                CountrySubdivision::Slovenia_municipality_Smarjeske_Toplice,
                CountrySubdivision::Slovenia_municipality_Smartno_ob_Paki,
                CountrySubdivision::Slovenia_municipality_Smartno_pri_Litiji,
                CountrySubdivision::Slovenia_municipality_Sodrazica,
                CountrySubdivision::Slovenia_municipality_Solcava,
                CountrySubdivision::Slovenia_municipality_Sostanj,
                CountrySubdivision::Slovenia_municipality_Sredisce_ob_Dravi,
                CountrySubdivision::Slovenia_municipality_Starse,
                CountrySubdivision::Slovenia_municipality_Store,
                CountrySubdivision::Slovenia_municipality_Straza,
                CountrySubdivision::Slovenia_municipality_Sveta_Ana,
                CountrySubdivision::Slovenia_municipality_Sveta_Trojica_v_Slovenskih_goricah,
                CountrySubdivision::Slovenia_municipality_Sveti_Andraz_v_Slovenskih_goricah,
                CountrySubdivision::Slovenia_municipality_Sveti_Jurij_ob_Scavnici,
                CountrySubdivision::Slovenia_municipality_Sveti_Jurij_v_Slovenskih_goricah,
                CountrySubdivision::Slovenia_municipality_Sveti_Tomaz,
                CountrySubdivision::Slovenia_municipality_Tabor,
                CountrySubdivision::Slovenia_municipality_Tisina,
                CountrySubdivision::Slovenia_municipality_Tolmin,
                CountrySubdivision::Slovenia_municipality_Trbovlje,
                CountrySubdivision::Slovenia_municipality_Trebnje,
                CountrySubdivision::Slovenia_municipality_Trnovska_Vas,
                CountrySubdivision::Slovenia_municipality_Trzic,
                CountrySubdivision::Slovenia_municipality_Trzin,
                CountrySubdivision::Slovenia_municipality_Turnisce,
                CountrySubdivision::Slovenia_municipality_Velika_Polana,
                CountrySubdivision::Slovenia_municipality_Velike_Lasce,
                CountrySubdivision::Slovenia_municipality_Verzej,
                CountrySubdivision::Slovenia_municipality_Videm,
                CountrySubdivision::Slovenia_municipality_Vipava,
                CountrySubdivision::Slovenia_municipality_Vitanje,
                CountrySubdivision::Slovenia_municipality_Vodice,
                CountrySubdivision::Slovenia_municipality_Vojnik,
                CountrySubdivision::Slovenia_municipality_Vransko,
                CountrySubdivision::Slovenia_municipality_Vrhnika,
                CountrySubdivision::Slovenia_municipality_Vuzenica,
                CountrySubdivision::Slovenia_municipality_Zagorje_ob_Savi,
                CountrySubdivision::Slovenia_municipality_Zalec,
                CountrySubdivision::Slovenia_municipality_Zavrc,
                CountrySubdivision::Slovenia_municipality_Zelezniki,
                CountrySubdivision::Slovenia_municipality_Zetale,
                CountrySubdivision::Slovenia_municipality_Ziri,
                CountrySubdivision::Slovenia_municipality_Zirovnica,
                CountrySubdivision::Slovenia_municipality_Zrece,
                CountrySubdivision::Slovenia_municipality_Zuzemberk,
                CountrySubdivision::Slovenia_urban_municipality_Celje,
                CountrySubdivision::Slovenia_urban_municipality_Koper,
                CountrySubdivision::Slovenia_urban_municipality_Kranj,
                CountrySubdivision::Slovenia_urban_municipality_Krsko,
                CountrySubdivision::Slovenia_urban_municipality_Ljubljana,
                CountrySubdivision::Slovenia_urban_municipality_Maribor,
                CountrySubdivision::Slovenia_urban_municipality_Murska_Sobota,
                CountrySubdivision::Slovenia_urban_municipality_Nova_Gorica,
                CountrySubdivision::Slovenia_urban_municipality_Novo_Mesto,
                CountrySubdivision::Slovenia_urban_municipality_Ptuj,
                CountrySubdivision::Slovenia_urban_municipality_Slovenj_Gradec,
                CountrySubdivision::Slovenia_urban_municipality_Velenje
            ],
            self::Solomon_Islands => [
                CountrySubdivision::Solomon_Islands_capital_territory_Capital_Territory_Honiara,
                CountrySubdivision::Solomon_Islands_province_Central,
                CountrySubdivision::Solomon_Islands_province_Choiseul,
                CountrySubdivision::Solomon_Islands_province_Guadalcanal,
                CountrySubdivision::Solomon_Islands_province_Isabel,
                CountrySubdivision::Solomon_Islands_province_Makira_Ulawa,
                CountrySubdivision::Solomon_Islands_province_Malaita,
                CountrySubdivision::Solomon_Islands_province_Rennell_and_Bellona,
                CountrySubdivision::Solomon_Islands_province_Temotu,
                CountrySubdivision::Solomon_Islands_province_Western
            ],
            self::Somalia => [
                CountrySubdivision::Somalia_region_Awdal,
                CountrySubdivision::Somalia_region_Bakool,
                CountrySubdivision::Somalia_region_Banaadir,
                CountrySubdivision::Somalia_region_Bari,
                CountrySubdivision::Somalia_region_Bay,
                CountrySubdivision::Somalia_region_Galguduud,
                CountrySubdivision::Somalia_region_Gedo,
                CountrySubdivision::Somalia_region_Hiiraan,
                CountrySubdivision::Somalia_region_Jubbada_Dhexe,
                CountrySubdivision::Somalia_region_Jubbada_Hoose,
                CountrySubdivision::Somalia_region_Mudug,
                CountrySubdivision::Somalia_region_Nugaal,
                CountrySubdivision::Somalia_region_Sanaag,
                CountrySubdivision::Somalia_region_Shabeellaha_Dhexe,
                CountrySubdivision::Somalia_region_Shabeellaha_Hoose,
                CountrySubdivision::Somalia_region_Sool,
                CountrySubdivision::Somalia_region_Togdheer,
                CountrySubdivision::Somalia_region_Woqooyi_Galbeed
            ],
            self::South_Africa => [
                CountrySubdivision::South_Africa_province_Bokone_Bophirima,
                CountrySubdivision::South_Africa_province_Eastern_Cape,
                CountrySubdivision::South_Africa_province_Foreisetata,
                CountrySubdivision::South_Africa_province_GaZulu_Natala,
                CountrySubdivision::South_Africa_province_Gauteng,
                CountrySubdivision::South_Africa_province_Kapa_Bodikela,
                CountrySubdivision::South_Africa_province_Kapa_Bokone,
                CountrySubdivision::South_Africa_province_Limpopo,
                CountrySubdivision::South_Africa_province_Mpumalanga
            ],
            self::South_Sudan => [
                CountrySubdivision::South_Sudan_state_Central_Equatoria,
                CountrySubdivision::South_Sudan_state_Eastern_Equatoria,
                CountrySubdivision::South_Sudan_state_Jonglei,
                CountrySubdivision::South_Sudan_state_Lakes,
                CountrySubdivision::South_Sudan_state_Northern_Bahr_el_Ghazal,
                CountrySubdivision::South_Sudan_state_Unity,
                CountrySubdivision::South_Sudan_state_Upper_Nile,
                CountrySubdivision::South_Sudan_state_Warrap,
                CountrySubdivision::South_Sudan_state_Western_Bahr_el_Ghazal,
                CountrySubdivision::South_Sudan_state_Western_Equatoria
            ],
            self::Spain => [
                CountrySubdivision::Spain_autonomous_city_in_North_Africa_Ceuta,
                CountrySubdivision::Spain_autonomous_city_in_North_Africa_Melilla,
                CountrySubdivision::Spain_autonomous_community_Andalucia,
                CountrySubdivision::Spain_autonomous_community_Aragon,
                CountrySubdivision::Spain_autonomous_community_Asturias_Principado_de,
                CountrySubdivision::Spain_autonomous_community_Canarias,
                CountrySubdivision::Spain_autonomous_community_Cantabria,
                CountrySubdivision::Spain_autonomous_community_Castilla_La_Mancha,
                CountrySubdivision::Spain_autonomous_community_Castilla_y_Leon,
                CountrySubdivision::Spain_autonomous_community_Catalunya_Cataluna,
                CountrySubdivision::Spain_autonomous_community_Euskal_Herria,
                CountrySubdivision::Spain_autonomous_community_Extremadura,
                CountrySubdivision::Spain_autonomous_community_Galicia_Galicia,
                CountrySubdivision::Spain_autonomous_community_Illes_Balears_Islas_Baleares,
                CountrySubdivision::Spain_autonomous_community_La_Rioja,
                CountrySubdivision::Spain_autonomous_community_Madrid_Comunidad_de,
                CountrySubdivision::Spain_autonomous_community_Murcia_Region_de,
                CountrySubdivision::Spain_autonomous_community_Nafarroako_Foru_Komunitatea,
                CountrySubdivision::Spain_autonomous_community_Valenciana_Comunidad,
                CountrySubdivision::Spain_province_A_Coruna_La_Coruna,
                CountrySubdivision::Spain_province_Alacant,
                CountrySubdivision::Spain_province_Albacete,
                CountrySubdivision::Spain_province_Almeria,
                CountrySubdivision::Spain_province_Araba,
                CountrySubdivision::Spain_province_Asturias,
                CountrySubdivision::Spain_province_Avila,
                CountrySubdivision::Spain_province_Badajoz,
                CountrySubdivision::Spain_province_Barcelona_Barcelona,
                CountrySubdivision::Spain_province_Bizkaia,
                CountrySubdivision::Spain_province_Burgos,
                CountrySubdivision::Spain_province_Caceres,
                CountrySubdivision::Spain_province_Cadiz,
                CountrySubdivision::Spain_province_Cantabria,
                CountrySubdivision::Spain_province_Castello,
                CountrySubdivision::Spain_province_Ciudad_Real,
                CountrySubdivision::Spain_province_Cordoba,
                CountrySubdivision::Spain_province_Cuenca,
                CountrySubdivision::Spain_province_Gipuzkoa,
                CountrySubdivision::Spain_province_Girona_Gerona,
                CountrySubdivision::Spain_province_Granada,
                CountrySubdivision::Spain_province_Guadalajara,
                CountrySubdivision::Spain_province_Huelva,
                CountrySubdivision::Spain_province_Huesca,
                CountrySubdivision::Spain_province_Illes_Balears_Islas_Baleares,
                CountrySubdivision::Spain_province_Jaen,
                CountrySubdivision::Spain_province_La_Rioja,
                CountrySubdivision::Spain_province_Las_Palmas,
                CountrySubdivision::Spain_province_Leon,
                CountrySubdivision::Spain_province_Lleida_Lerida,
                CountrySubdivision::Spain_province_Lugo_Lugo,
                CountrySubdivision::Spain_province_Madrid,
                CountrySubdivision::Spain_province_Malaga,
                CountrySubdivision::Spain_province_Murcia,
                CountrySubdivision::Spain_province_Nafarroa,
                CountrySubdivision::Spain_province_Ourense_Orense,
                CountrySubdivision::Spain_province_Palencia,
                CountrySubdivision::Spain_province_Pontevedra_Pontevedra,
                CountrySubdivision::Spain_province_Salamanca,
                CountrySubdivision::Spain_province_Santa_Cruz_de_Tenerife,
                CountrySubdivision::Spain_province_Segovia,
                CountrySubdivision::Spain_province_Sevilla,
                CountrySubdivision::Spain_province_Soria,
                CountrySubdivision::Spain_province_Tarragona_Tarragona,
                CountrySubdivision::Spain_province_Teruel,
                CountrySubdivision::Spain_province_Toledo,
                CountrySubdivision::Spain_province_Valencia,
                CountrySubdivision::Spain_province_Valladolid,
                CountrySubdivision::Spain_province_Zamora,
                CountrySubdivision::Spain_province_Zaragoza
            ],
            self::Sri_Lanka => [
                CountrySubdivision::Sri_Lanka_district_Ampantottai,
                CountrySubdivision::Sri_Lanka_district_Ampara,
                CountrySubdivision::Sri_Lanka_district_Anuradhapura,
                CountrySubdivision::Sri_Lanka_district_Badulla,
                CountrySubdivision::Sri_Lanka_district_Batticaloa,
                CountrySubdivision::Sri_Lanka_district_Colombo,
                CountrySubdivision::Sri_Lanka_district_Galle,
                CountrySubdivision::Sri_Lanka_district_Gampaha,
                CountrySubdivision::Sri_Lanka_district_Irattinapuri,
                CountrySubdivision::Sri_Lanka_district_Jaffna,
                CountrySubdivision::Sri_Lanka_district_Kalutara,
                CountrySubdivision::Sri_Lanka_district_Kandy,
                CountrySubdivision::Sri_Lanka_district_Kegalla,
                CountrySubdivision::Sri_Lanka_district_Kilinochchi,
                CountrySubdivision::Sri_Lanka_district_Kurunegala,
                CountrySubdivision::Sri_Lanka_district_Mannar,
                CountrySubdivision::Sri_Lanka_district_Matale,
                CountrySubdivision::Sri_Lanka_district_Matara,
                CountrySubdivision::Sri_Lanka_district_Monaragala,
                CountrySubdivision::Sri_Lanka_district_Mulativ,
                CountrySubdivision::Sri_Lanka_district_Nuvara_Eliya,
                CountrySubdivision::Sri_Lanka_district_Polonnaruwa,
                CountrySubdivision::Sri_Lanka_district_Puttalam,
                CountrySubdivision::Sri_Lanka_district_Tirukonamalai,
                CountrySubdivision::Sri_Lanka_district_Vavuniya,
                CountrySubdivision::Sri_Lanka_province_Basnahira_palata,
                CountrySubdivision::Sri_Lanka_province_Central_Province,
                CountrySubdivision::Sri_Lanka_province_Chappirakamuva_makanam,
                CountrySubdivision::Sri_Lanka_province_Dakunu_palata,
                CountrySubdivision::Sri_Lanka_province_Eastern_Province,
                CountrySubdivision::Sri_Lanka_province_North_Central_Province,
                CountrySubdivision::Sri_Lanka_province_North_Western_Province,
                CountrySubdivision::Sri_Lanka_province_Northern_Province,
                CountrySubdivision::Sri_Lanka_province_Uva_Province
            ],
            self::Sudan => [
                CountrySubdivision::Sudan_state_Al_Bahr_al_Ahmar,
                CountrySubdivision::Sudan_state_Al_Jazirah,
                CountrySubdivision::Sudan_state_Al_Khartum,
                CountrySubdivision::Sudan_state_Al_Qadarif,
                CountrySubdivision::Sudan_state_An_Nil_al_Abyad,
                CountrySubdivision::Sudan_state_An_Nil_al_Azraq,
                CountrySubdivision::Sudan_state_Ash_Shamaliyah,
                CountrySubdivision::Sudan_state_Central_Darfur,
                CountrySubdivision::Sudan_state_East_Darfur,
                CountrySubdivision::Sudan_state_Gharb_Darfur,
                CountrySubdivision::Sudan_state_Gharb_Kurdufan,
                CountrySubdivision::Sudan_state_Janub_Darfur,
                CountrySubdivision::Sudan_state_Janub_Kurdufan,
                CountrySubdivision::Sudan_state_Kassala,
                CountrySubdivision::Sudan_state_Nahr_an_Nil,
                CountrySubdivision::Sudan_state_North_Darfur,
                CountrySubdivision::Sudan_state_North_Kordofan,
                CountrySubdivision::Sudan_state_Sennar
            ],
            self::Suriname => [
                CountrySubdivision::Suriname_district_Brokopondo,
                CountrySubdivision::Suriname_district_Commewijne,
                CountrySubdivision::Suriname_district_Coronie,
                CountrySubdivision::Suriname_district_Marowijne,
                CountrySubdivision::Suriname_district_Nickerie,
                CountrySubdivision::Suriname_district_Para,
                CountrySubdivision::Suriname_district_Paramaribo,
                CountrySubdivision::Suriname_district_Saramacca,
                CountrySubdivision::Suriname_district_Sipaliwini,
                CountrySubdivision::Suriname_district_Wanica
            ],
            self::Sweden => [
                CountrySubdivision::Sweden_county_Blekinge_lan_SE_10,
                CountrySubdivision::Sweden_county_Dalarnas_lan_SE_20,
                CountrySubdivision::Sweden_county_Gavleborgs_lan_SE_21,
                CountrySubdivision::Sweden_county_Gotlands_lan_SE_09,
                CountrySubdivision::Sweden_county_Hallands_lan_SE_13,
                CountrySubdivision::Sweden_county_Jamtlands_lan_SE_23,
                CountrySubdivision::Sweden_county_Jonkopings_lan_SE_06,
                CountrySubdivision::Sweden_county_Kalmar_lan_SE_08,
                CountrySubdivision::Sweden_county_Kronobergs_lan_SE_07,
                CountrySubdivision::Sweden_county_Norrbottens_lan_SE_25,
                CountrySubdivision::Sweden_county_Orebro_lan_SE_18,
                CountrySubdivision::Sweden_county_Ostergotlands_lan_SE_05,
                CountrySubdivision::Sweden_county_Skane_lan_SE_12,
                CountrySubdivision::Sweden_county_Sodermanlands_lan_SE_04,
                CountrySubdivision::Sweden_county_Stockholms_lan_SE_01,
                CountrySubdivision::Sweden_county_Uppsala_lan_SE_03,
                CountrySubdivision::Sweden_county_Varmlands_lan_SE_17,
                CountrySubdivision::Sweden_county_Vasterbottens_lan_SE_24,
                CountrySubdivision::Sweden_county_Vasternorrlands_lan_SE_22,
                CountrySubdivision::Sweden_county_Vastmanlands_lan_SE_19,
                CountrySubdivision::Sweden_county_Vastra_Gotalands_lan_SE_14
            ],
            self::Switzerland => [
                CountrySubdivision::Switzerland_canton_Aargau,
                CountrySubdivision::Switzerland_canton_Appenzell_Ausserrhoden,
                CountrySubdivision::Switzerland_canton_Appenzell_Innerrhoden,
                CountrySubdivision::Switzerland_canton_Basel_Landschaft,
                CountrySubdivision::Switzerland_canton_Basel_Stadt,
                CountrySubdivision::Switzerland_canton_Bern,
                CountrySubdivision::Switzerland_canton_Freiburg,
                CountrySubdivision::Switzerland_canton_Geneve,
                CountrySubdivision::Switzerland_canton_Glarus,
                CountrySubdivision::Switzerland_canton_Graubunden,
                CountrySubdivision::Switzerland_canton_Jura,
                CountrySubdivision::Switzerland_canton_Luzern,
                CountrySubdivision::Switzerland_canton_Neuchatel,
                CountrySubdivision::Switzerland_canton_Nidwalden,
                CountrySubdivision::Switzerland_canton_Obwalden,
                CountrySubdivision::Switzerland_canton_Sankt_Gallen,
                CountrySubdivision::Switzerland_canton_Schaffhausen,
                CountrySubdivision::Switzerland_canton_Schwyz,
                CountrySubdivision::Switzerland_canton_Solothurn,
                CountrySubdivision::Switzerland_canton_Thurgau,
                CountrySubdivision::Switzerland_canton_Ticino,
                CountrySubdivision::Switzerland_canton_Uri,
                CountrySubdivision::Switzerland_canton_Valais,
                CountrySubdivision::Switzerland_canton_Vaud,
                CountrySubdivision::Switzerland_canton_Zug,
                CountrySubdivision::Switzerland_canton_Zurich
            ],
            self::Syrian_Arab_Republic => [
                CountrySubdivision::Syria_province_Al_Hasakah,
                CountrySubdivision::Syria_province_Al_Ladhiqiyah,
                CountrySubdivision::Syria_province_Al_Qunaytirah,
                CountrySubdivision::Syria_province_Ar_Raqqah,
                CountrySubdivision::Syria_province_As_Suwayda,
                CountrySubdivision::Syria_province_Dar_a,
                CountrySubdivision::Syria_province_Dayr_az_Zawr,
                CountrySubdivision::Syria_province_Dimashq,
                CountrySubdivision::Syria_province_Halab,
                CountrySubdivision::Syria_province_Hamah,
                CountrySubdivision::Syria_province_Hims,
                CountrySubdivision::Syria_province_Idlib,
                CountrySubdivision::Syria_province_Rif_Dimashq,
                CountrySubdivision::Syria_province_Tartus
            ],
            self::Taiwan_Province_of_China => [
                CountrySubdivision::Taiwan_city_Chiayi,
                CountrySubdivision::Taiwan_city_Hsinchu,
                CountrySubdivision::Taiwan_city_Keelung,
                CountrySubdivision::Taiwan_county_Changhua,
                CountrySubdivision::Taiwan_county_Chiayi,
                CountrySubdivision::Taiwan_county_Hsinchu,
                CountrySubdivision::Taiwan_county_Hualien,
                CountrySubdivision::Taiwan_county_Kinmen,
                CountrySubdivision::Taiwan_county_Lienchiang,
                CountrySubdivision::Taiwan_county_Miaoli,
                CountrySubdivision::Taiwan_county_Nantou,
                CountrySubdivision::Taiwan_county_Penghu,
                CountrySubdivision::Taiwan_county_Pingtung,
                CountrySubdivision::Taiwan_county_Taitung,
                CountrySubdivision::Taiwan_county_Yilan,
                CountrySubdivision::Taiwan_county_Yunlin,
                CountrySubdivision::Taiwan_special_municipality_Kaohsiung,
                CountrySubdivision::Taiwan_special_municipality_New_Taipei,
                CountrySubdivision::Taiwan_special_municipality_Taichung,
                CountrySubdivision::Taiwan_special_municipality_Tainan,
                CountrySubdivision::Taiwan_special_municipality_Taipei,
                CountrySubdivision::Taiwan_special_municipality_Taoyuan
            ],
            self::Tajikistan => [
                CountrySubdivision::Tajikistan_autonomous_region_Kuhistoni_Badakhshon,
                CountrySubdivision::Tajikistan_capital_territory_Dushanbe,
                CountrySubdivision::Tajikistan_districts_under_republic_administration_nohiyahoi_tobei_jumhuri,
                CountrySubdivision::Tajikistan_region_Khatlon,
                CountrySubdivision::Tajikistan_region_Sughd
            ],
            self::Tanzania => [
                CountrySubdivision::Tanzania_region_Arusha,
                CountrySubdivision::Tanzania_region_Coast,
                CountrySubdivision::Tanzania_region_Dar_es_Salaam,
                CountrySubdivision::Tanzania_region_Dodoma,
                CountrySubdivision::Tanzania_region_Geita,
                CountrySubdivision::Tanzania_region_Iringa,
                CountrySubdivision::Tanzania_region_Kagera,
                CountrySubdivision::Tanzania_region_Kaskazini_Pemba,
                CountrySubdivision::Tanzania_region_Kaskazini_Unguja,
                CountrySubdivision::Tanzania_region_Katavi,
                CountrySubdivision::Tanzania_region_Kigoma,
                CountrySubdivision::Tanzania_region_Kilimanjaro,
                CountrySubdivision::Tanzania_region_Kusini_Pemba,
                CountrySubdivision::Tanzania_region_Kusini_Unguja,
                CountrySubdivision::Tanzania_region_Lindi,
                CountrySubdivision::Tanzania_region_Manyara,
                CountrySubdivision::Tanzania_region_Mara,
                CountrySubdivision::Tanzania_region_Mbeya,
                CountrySubdivision::Tanzania_region_Mjini_Magharibi,
                CountrySubdivision::Tanzania_region_Morogoro,
                CountrySubdivision::Tanzania_region_Mtwara,
                CountrySubdivision::Tanzania_region_Mwanza,
                CountrySubdivision::Tanzania_region_Njombe,
                CountrySubdivision::Tanzania_region_Rukwa,
                CountrySubdivision::Tanzania_region_Ruvuma,
                CountrySubdivision::Tanzania_region_Shinyanga,
                CountrySubdivision::Tanzania_region_Simiyu,
                CountrySubdivision::Tanzania_region_Singida,
                CountrySubdivision::Tanzania_region_Songwe,
                CountrySubdivision::Tanzania_region_Tabora,
                CountrySubdivision::Tanzania_region_Tanga
            ],
            self::Thailand => [
                CountrySubdivision::Thailand_metropolitan_administration_Krung_Thep_Maha_Nakhon,
                CountrySubdivision::Thailand_province_Amnat_Charoen,
                CountrySubdivision::Thailand_province_Ang_Thong,
                CountrySubdivision::Thailand_province_Bueng_Kan,
                CountrySubdivision::Thailand_province_Buri_Ram,
                CountrySubdivision::Thailand_province_Chachoengsao,
                CountrySubdivision::Thailand_province_Chai_Nat,
                CountrySubdivision::Thailand_province_Chaiyaphum,
                CountrySubdivision::Thailand_province_Chanthaburi,
                CountrySubdivision::Thailand_province_Chiang_Mai,
                CountrySubdivision::Thailand_province_Chiang_Rai,
                CountrySubdivision::Thailand_province_Chon_Buri,
                CountrySubdivision::Thailand_province_Chumphon,
                CountrySubdivision::Thailand_province_Kalasin,
                CountrySubdivision::Thailand_province_Kamphaeng_Phet,
                CountrySubdivision::Thailand_province_Kanchanaburi,
                CountrySubdivision::Thailand_province_Khon_Kaen,
                CountrySubdivision::Thailand_province_Krabi,
                CountrySubdivision::Thailand_province_Lampang,
                CountrySubdivision::Thailand_province_Lamphun,
                CountrySubdivision::Thailand_province_Loei,
                CountrySubdivision::Thailand_province_Lop_Buri,
                CountrySubdivision::Thailand_province_Mae_Hong_Son,
                CountrySubdivision::Thailand_province_Maha_Sarakham,
                CountrySubdivision::Thailand_province_Mukdahan,
                CountrySubdivision::Thailand_province_Nakhon_Nayok,
                CountrySubdivision::Thailand_province_Nakhon_Pathom,
                CountrySubdivision::Thailand_province_Nakhon_Phanom,
                CountrySubdivision::Thailand_province_Nakhon_Ratchasima,
                CountrySubdivision::Thailand_province_Nakhon_Sawan,
                CountrySubdivision::Thailand_province_Nakhon_Si_Thammarat,
                CountrySubdivision::Thailand_province_Nan,
                CountrySubdivision::Thailand_province_Narathiwat,
                CountrySubdivision::Thailand_province_Nong_Bua_Lam_Phu,
                CountrySubdivision::Thailand_province_Nong_Khai,
                CountrySubdivision::Thailand_province_Nonthaburi,
                CountrySubdivision::Thailand_province_Pathum_Thani,
                CountrySubdivision::Thailand_province_Pattani,
                CountrySubdivision::Thailand_province_Phangnga,
                CountrySubdivision::Thailand_province_Phatthalung,
                CountrySubdivision::Thailand_province_Phayao,
                CountrySubdivision::Thailand_province_Phetchabun,
                CountrySubdivision::Thailand_province_Phetchaburi,
                CountrySubdivision::Thailand_province_Phichit,
                CountrySubdivision::Thailand_province_Phitsanulok,
                CountrySubdivision::Thailand_province_Phra_Nakhon_Si_Ayutthaya,
                CountrySubdivision::Thailand_province_Phrae,
                CountrySubdivision::Thailand_province_Phuket,
                CountrySubdivision::Thailand_province_Prachin_Buri,
                CountrySubdivision::Thailand_province_Prachuap_Khiri_Khan,
                CountrySubdivision::Thailand_province_Ranong,
                CountrySubdivision::Thailand_province_Ratchaburi,
                CountrySubdivision::Thailand_province_Rayong,
                CountrySubdivision::Thailand_province_Roi_Et,
                CountrySubdivision::Thailand_province_Sa_Kaeo,
                CountrySubdivision::Thailand_province_Sakon_Nakhon,
                CountrySubdivision::Thailand_province_Samut_Prakan,
                CountrySubdivision::Thailand_province_Samut_Sakhon,
                CountrySubdivision::Thailand_province_Samut_Songkhram,
                CountrySubdivision::Thailand_province_Saraburi,
                CountrySubdivision::Thailand_province_Satun,
                CountrySubdivision::Thailand_province_Si_Sa_Ket,
                CountrySubdivision::Thailand_province_Sing_Buri,
                CountrySubdivision::Thailand_province_Songkhla,
                CountrySubdivision::Thailand_province_Sukhothai,
                CountrySubdivision::Thailand_province_Suphan_Buri,
                CountrySubdivision::Thailand_province_Surat_Thani,
                CountrySubdivision::Thailand_province_Surin,
                CountrySubdivision::Thailand_province_Tak,
                CountrySubdivision::Thailand_province_Trang,
                CountrySubdivision::Thailand_province_Trat,
                CountrySubdivision::Thailand_province_Ubon_Ratchathani,
                CountrySubdivision::Thailand_province_Udon_Thani,
                CountrySubdivision::Thailand_province_Uthai_Thani,
                CountrySubdivision::Thailand_province_Uttaradit,
                CountrySubdivision::Thailand_province_Yala,
                CountrySubdivision::Thailand_province_Yasothon,
                CountrySubdivision::Thailand_special_administrative_city_Phatthaya
            ],
            self::Timor_Leste => [
                CountrySubdivision::Timor_Leste_municipality_Aileu,
                CountrySubdivision::Timor_Leste_municipality_Ainaro,
                CountrySubdivision::Timor_Leste_municipality_Baucau,
                CountrySubdivision::Timor_Leste_municipality_Bobonaro,
                CountrySubdivision::Timor_Leste_municipality_Cova_Lima,
                CountrySubdivision::Timor_Leste_municipality_Dili,
                CountrySubdivision::Timor_Leste_municipality_Ermera,
                CountrySubdivision::Timor_Leste_municipality_Lautein,
                CountrySubdivision::Timor_Leste_municipality_Likisa,
                CountrySubdivision::Timor_Leste_municipality_Manatuto,
                CountrySubdivision::Timor_Leste_municipality_Manufahi,
                CountrySubdivision::Timor_Leste_municipality_Vikeke,
                CountrySubdivision::Timor_Leste_special_administrative_region_Oekusi_Ambenu
            ],
            self::Togo => [
                CountrySubdivision::Togo_region_Centrale,
                CountrySubdivision::Togo_region_Kara,
                CountrySubdivision::Togo_region_Maritime_Region,
                CountrySubdivision::Togo_region_Plateaux,
                CountrySubdivision::Togo_region_Savanes
            ],
            self::Tonga => [
                CountrySubdivision::Tonga_division_Eua,
                CountrySubdivision::Tonga_division_Ha_apai,
                CountrySubdivision::Tonga_division_Niuas,
                CountrySubdivision::Tonga_division_Tongatapu,
                CountrySubdivision::Tonga_division_Vava_u
            ],
            self::Trinidad_and_Tobago => [
                CountrySubdivision::Trinidad_Tobago_borough_Arima,
                CountrySubdivision::Trinidad_Tobago_borough_Chaguanas,
                CountrySubdivision::Trinidad_Tobago_borough_Point_Fortin,
                CountrySubdivision::Trinidad_Tobago_city_Port_of_Spain,
                CountrySubdivision::Trinidad_Tobago_city_San_Fernando,
                CountrySubdivision::Trinidad_Tobago_region_Couva_Tabaquite_Talparo,
                CountrySubdivision::Trinidad_Tobago_region_Diego_Martin,
                CountrySubdivision::Trinidad_Tobago_region_Mayaro_Rio_Claro,
                CountrySubdivision::Trinidad_Tobago_region_Penal_Debe,
                CountrySubdivision::Trinidad_Tobago_region_Princes_Town,
                CountrySubdivision::Trinidad_Tobago_region_San_Juan_Laventille,
                CountrySubdivision::Trinidad_Tobago_region_Sangre_Grande,
                CountrySubdivision::Trinidad_Tobago_region_Siparia,
                CountrySubdivision::Trinidad_Tobago_region_Tunapuna_Piarco,
                CountrySubdivision::Trinidad_Tobago_ward_Tobago
            ],
            self::Tunisia => [
                CountrySubdivision::Tunisia_governorate_Beja,
                CountrySubdivision::Tunisia_governorate_Ben_Arous,
                CountrySubdivision::Tunisia_governorate_Bizerte,
                CountrySubdivision::Tunisia_governorate_Gabes,
                CountrySubdivision::Tunisia_governorate_Gafsa,
                CountrySubdivision::Tunisia_governorate_Jendouba,
                CountrySubdivision::Tunisia_governorate_Kairouan,
                CountrySubdivision::Tunisia_governorate_Kasserine,
                CountrySubdivision::Tunisia_governorate_Kebili,
                CountrySubdivision::Tunisia_governorate_L_Ariana,
                CountrySubdivision::Tunisia_governorate_La_Manouba,
                CountrySubdivision::Tunisia_governorate_Le_Kef,
                CountrySubdivision::Tunisia_governorate_Mahdia,
                CountrySubdivision::Tunisia_governorate_Medenine,
                CountrySubdivision::Tunisia_governorate_Monastir,
                CountrySubdivision::Tunisia_governorate_Nabeul,
                CountrySubdivision::Tunisia_governorate_Sfax,
                CountrySubdivision::Tunisia_governorate_Sidi_Bouzid,
                CountrySubdivision::Tunisia_governorate_Siliana,
                CountrySubdivision::Tunisia_governorate_Sousse,
                CountrySubdivision::Tunisia_governorate_Tataouine,
                CountrySubdivision::Tunisia_governorate_Tozeur,
                CountrySubdivision::Tunisia_governorate_Tunis,
                CountrySubdivision::Tunisia_governorate_Zaghouan
            ],
            self::Turkey => [
                CountrySubdivision::Turkiye_province_Adana,
                CountrySubdivision::Turkiye_province_Adiyaman,
                CountrySubdivision::Turkiye_province_Afyonkarahisar,
                CountrySubdivision::Turkiye_province_Agri,
                CountrySubdivision::Turkiye_province_Aksaray,
                CountrySubdivision::Turkiye_province_Amasya,
                CountrySubdivision::Turkiye_province_Ankara,
                CountrySubdivision::Turkiye_province_Antalya,
                CountrySubdivision::Turkiye_province_Ardahan,
                CountrySubdivision::Turkiye_province_Artvin,
                CountrySubdivision::Turkiye_province_Aydin,
                CountrySubdivision::Turkiye_province_Balikesir,
                CountrySubdivision::Turkiye_province_Bartin,
                CountrySubdivision::Turkiye_province_Batman,
                CountrySubdivision::Turkiye_province_Bayburt,
                CountrySubdivision::Turkiye_province_Bilecik,
                CountrySubdivision::Turkiye_province_Bingol,
                CountrySubdivision::Turkiye_province_Bitlis,
                CountrySubdivision::Turkiye_province_Bolu,
                CountrySubdivision::Turkiye_province_Burdur,
                CountrySubdivision::Turkiye_province_Bursa,
                CountrySubdivision::Turkiye_province_Canakkale,
                CountrySubdivision::Turkiye_province_Cankiri,
                CountrySubdivision::Turkiye_province_Corum,
                CountrySubdivision::Turkiye_province_Denizli,
                CountrySubdivision::Turkiye_province_Diyarbakir,
                CountrySubdivision::Turkiye_province_Duzce,
                CountrySubdivision::Turkiye_province_Edirne,
                CountrySubdivision::Turkiye_province_Elazig,
                CountrySubdivision::Turkiye_province_Erzincan,
                CountrySubdivision::Turkiye_province_Erzurum,
                CountrySubdivision::Turkiye_province_Eskisehir,
                CountrySubdivision::Turkiye_province_Gaziantep,
                CountrySubdivision::Turkiye_province_Giresun,
                CountrySubdivision::Turkiye_province_Gumushane,
                CountrySubdivision::Turkiye_province_Hakkari,
                CountrySubdivision::Turkiye_province_Hatay,
                CountrySubdivision::Turkiye_province_Igdir,
                CountrySubdivision::Turkiye_province_Isparta,
                CountrySubdivision::Turkiye_province_Istanbul,
                CountrySubdivision::Turkiye_province_Izmir,
                CountrySubdivision::Turkiye_province_Kahramanmaras,
                CountrySubdivision::Turkiye_province_Karabuk,
                CountrySubdivision::Turkiye_province_Karaman,
                CountrySubdivision::Turkiye_province_Kars,
                CountrySubdivision::Turkiye_province_Kastamonu,
                CountrySubdivision::Turkiye_province_Kayseri,
                CountrySubdivision::Turkiye_province_Kilis,
                CountrySubdivision::Turkiye_province_Kirikkale,
                CountrySubdivision::Turkiye_province_Kirklareli,
                CountrySubdivision::Turkiye_province_Kirsehir,
                CountrySubdivision::Turkiye_province_Kocaeli,
                CountrySubdivision::Turkiye_province_Konya,
                CountrySubdivision::Turkiye_province_Kutahya,
                CountrySubdivision::Turkiye_province_Malatya,
                CountrySubdivision::Turkiye_province_Manisa,
                CountrySubdivision::Turkiye_province_Mardin,
                CountrySubdivision::Turkiye_province_Mersin,
                CountrySubdivision::Turkiye_province_Mugla,
                CountrySubdivision::Turkiye_province_Mus,
                CountrySubdivision::Turkiye_province_Nevsehir,
                CountrySubdivision::Turkiye_province_Nigde,
                CountrySubdivision::Turkiye_province_Ordu,
                CountrySubdivision::Turkiye_province_Osmaniye,
                CountrySubdivision::Turkiye_province_Rize,
                CountrySubdivision::Turkiye_province_Sakarya,
                CountrySubdivision::Turkiye_province_Samsun,
                CountrySubdivision::Turkiye_province_Sanliurfa,
                CountrySubdivision::Turkiye_province_Siirt,
                CountrySubdivision::Turkiye_province_Sinop,
                CountrySubdivision::Turkiye_province_Sirnak,
                CountrySubdivision::Turkiye_province_Sivas,
                CountrySubdivision::Turkiye_province_Tekirdag,
                CountrySubdivision::Turkiye_province_Tokat,
                CountrySubdivision::Turkiye_province_Trabzon,
                CountrySubdivision::Turkiye_province_Tunceli,
                CountrySubdivision::Turkiye_province_Usak,
                CountrySubdivision::Turkiye_province_Van,
                CountrySubdivision::Turkiye_province_Yalova,
                CountrySubdivision::Turkiye_province_Yozgat,
                CountrySubdivision::Turkiye_province_Zonguldak
            ],
            self::Turkmenistan => [
                CountrySubdivision::Turkmenistan_city_Asgabat,
                CountrySubdivision::Turkmenistan_region_Ahal,
                CountrySubdivision::Turkmenistan_region_Balkan,
                CountrySubdivision::Turkmenistan_region_Dasoguz,
                CountrySubdivision::Turkmenistan_region_Lebap,
                CountrySubdivision::Turkmenistan_region_Mary
            ],
            self::Tuvalu => [
                CountrySubdivision::Tuvalu_island_council_Nanumaga,
                CountrySubdivision::Tuvalu_island_council_Nanumea,
                CountrySubdivision::Tuvalu_island_council_Niutao,
                CountrySubdivision::Tuvalu_island_council_Nui,
                CountrySubdivision::Tuvalu_island_council_Nukufetau,
                CountrySubdivision::Tuvalu_island_council_Nukulaelae,
                CountrySubdivision::Tuvalu_island_council_Vaitupu,
                CountrySubdivision::Tuvalu_town_council_Funafuti
            ],
            self::Uganda => [
                CountrySubdivision::Uganda_city_Kampala,
                CountrySubdivision::Uganda_district_Abim,
                CountrySubdivision::Uganda_district_Adjumani,
                CountrySubdivision::Uganda_district_Agago,
                CountrySubdivision::Uganda_district_Alebtong,
                CountrySubdivision::Uganda_district_Amolatar,
                CountrySubdivision::Uganda_district_Amudat,
                CountrySubdivision::Uganda_district_Amuria,
                CountrySubdivision::Uganda_district_Amuru,
                CountrySubdivision::Uganda_district_Apac,
                CountrySubdivision::Uganda_district_Arua,
                CountrySubdivision::Uganda_district_Budaka,
                CountrySubdivision::Uganda_district_Bududa,
                CountrySubdivision::Uganda_district_Bugiri,
                CountrySubdivision::Uganda_district_Bugweri,
                CountrySubdivision::Uganda_district_Buhweju,
                CountrySubdivision::Uganda_district_Buikwe,
                CountrySubdivision::Uganda_district_Bukedea,
                CountrySubdivision::Uganda_district_Bukomansibi,
                CountrySubdivision::Uganda_district_Bukwo,
                CountrySubdivision::Uganda_district_Bulambuli,
                CountrySubdivision::Uganda_district_Buliisa,
                CountrySubdivision::Uganda_district_Bundibugyo,
                CountrySubdivision::Uganda_district_Bunyangabu,
                CountrySubdivision::Uganda_district_Bushenyi,
                CountrySubdivision::Uganda_district_Busia,
                CountrySubdivision::Uganda_district_Butaleja,
                CountrySubdivision::Uganda_district_Butambala,
                CountrySubdivision::Uganda_district_Butebo,
                CountrySubdivision::Uganda_district_Buvuma,
                CountrySubdivision::Uganda_district_Buyende,
                CountrySubdivision::Uganda_district_Dokolo,
                CountrySubdivision::Uganda_district_Gomba,
                CountrySubdivision::Uganda_district_Gulu,
                CountrySubdivision::Uganda_district_Hoima,
                CountrySubdivision::Uganda_district_Ibanda,
                CountrySubdivision::Uganda_district_Iganga,
                CountrySubdivision::Uganda_district_Isingiro,
                CountrySubdivision::Uganda_district_Jinja,
                CountrySubdivision::Uganda_district_Kaabong,
                CountrySubdivision::Uganda_district_Kabale,
                CountrySubdivision::Uganda_district_Kabarole,
                CountrySubdivision::Uganda_district_Kaberamaido,
                CountrySubdivision::Uganda_district_Kagadi,
                CountrySubdivision::Uganda_district_Kakumiro,
                CountrySubdivision::Uganda_district_Kalaki,
                CountrySubdivision::Uganda_district_Kalangala,
                CountrySubdivision::Uganda_district_Kaliro,
                CountrySubdivision::Uganda_district_Kalungu,
                CountrySubdivision::Uganda_district_Kamuli,
                CountrySubdivision::Uganda_district_Kamwenge,
                CountrySubdivision::Uganda_district_Kanungu,
                CountrySubdivision::Uganda_district_Kapchorwa,
                CountrySubdivision::Uganda_district_Kapelebyong,
                CountrySubdivision::Uganda_district_Karenga,
                CountrySubdivision::Uganda_district_Kasanda,
                CountrySubdivision::Uganda_district_Kasese,
                CountrySubdivision::Uganda_district_Katakwi,
                CountrySubdivision::Uganda_district_Kayunga,
                CountrySubdivision::Uganda_district_Kazo,
                CountrySubdivision::Uganda_district_Kibaale,
                CountrySubdivision::Uganda_district_Kiboga,
                CountrySubdivision::Uganda_district_Kibuku,
                CountrySubdivision::Uganda_district_Kikuube,
                CountrySubdivision::Uganda_district_Kiruhura,
                CountrySubdivision::Uganda_district_Kiryandongo,
                CountrySubdivision::Uganda_district_Kisoro,
                CountrySubdivision::Uganda_district_Kitagwenda,
                CountrySubdivision::Uganda_district_Kitgum,
                CountrySubdivision::Uganda_district_Koboko,
                CountrySubdivision::Uganda_district_Kole,
                CountrySubdivision::Uganda_district_Kotido,
                CountrySubdivision::Uganda_district_Kumi,
                CountrySubdivision::Uganda_district_Kwania,
                CountrySubdivision::Uganda_district_Kween,
                CountrySubdivision::Uganda_district_Kyankwanzi,
                CountrySubdivision::Uganda_district_Kyegegwa,
                CountrySubdivision::Uganda_district_Kyenjojo,
                CountrySubdivision::Uganda_district_Kyotera,
                CountrySubdivision::Uganda_district_Lamwo,
                CountrySubdivision::Uganda_district_Lira,
                CountrySubdivision::Uganda_district_Luuka,
                CountrySubdivision::Uganda_district_Luwero,
                CountrySubdivision::Uganda_district_Lwengo,
                CountrySubdivision::Uganda_district_Lyantonde,
                CountrySubdivision::Uganda_district_Madi_Okollo,
                CountrySubdivision::Uganda_district_Manafwa,
                CountrySubdivision::Uganda_district_Maracha,
                CountrySubdivision::Uganda_district_Masaka,
                CountrySubdivision::Uganda_district_Masindi,
                CountrySubdivision::Uganda_district_Mayuge,
                CountrySubdivision::Uganda_district_Mbale,
                CountrySubdivision::Uganda_district_Mbarara,
                CountrySubdivision::Uganda_district_Mitooma,
                CountrySubdivision::Uganda_district_Mityana,
                CountrySubdivision::Uganda_district_Moroto,
                CountrySubdivision::Uganda_district_Moyo,
                CountrySubdivision::Uganda_district_Mpigi,
                CountrySubdivision::Uganda_district_Mubende,
                CountrySubdivision::Uganda_district_Mukono,
                CountrySubdivision::Uganda_district_Nabilatuk,
                CountrySubdivision::Uganda_district_Nakapiripirit,
                CountrySubdivision::Uganda_district_Nakaseke,
                CountrySubdivision::Uganda_district_Nakasongola,
                CountrySubdivision::Uganda_district_Namayingo,
                CountrySubdivision::Uganda_district_Namisindwa,
                CountrySubdivision::Uganda_district_Namutumba,
                CountrySubdivision::Uganda_district_Napak,
                CountrySubdivision::Uganda_district_Nebbi,
                CountrySubdivision::Uganda_district_Ngora,
                CountrySubdivision::Uganda_district_Ntoroko,
                CountrySubdivision::Uganda_district_Ntungamo,
                CountrySubdivision::Uganda_district_Nwoya,
                CountrySubdivision::Uganda_district_Obongi,
                CountrySubdivision::Uganda_district_Omoro,
                CountrySubdivision::Uganda_district_Otuke,
                CountrySubdivision::Uganda_district_Oyam,
                CountrySubdivision::Uganda_district_Pader,
                CountrySubdivision::Uganda_district_Pakwach,
                CountrySubdivision::Uganda_district_Pallisa,
                CountrySubdivision::Uganda_district_Rakai,
                CountrySubdivision::Uganda_district_Rubanda,
                CountrySubdivision::Uganda_district_Rubirizi,
                CountrySubdivision::Uganda_district_Rukiga,
                CountrySubdivision::Uganda_district_Rukungiri,
                CountrySubdivision::Uganda_district_Rwampara,
                CountrySubdivision::Uganda_district_Sembabule,
                CountrySubdivision::Uganda_district_Serere,
                CountrySubdivision::Uganda_district_Sheema,
                CountrySubdivision::Uganda_district_Sironko,
                CountrySubdivision::Uganda_district_Soroti,
                CountrySubdivision::Uganda_district_Tororo,
                CountrySubdivision::Uganda_district_Wakiso,
                CountrySubdivision::Uganda_district_Yumbe,
                CountrySubdivision::Uganda_district_Zombo,
                CountrySubdivision::Uganda_geographical_region_Central,
                CountrySubdivision::Uganda_geographical_region_Eastern,
                CountrySubdivision::Uganda_geographical_region_Northern,
                CountrySubdivision::Uganda_geographical_region_Western
            ],
            self::Ukraine => [
                CountrySubdivision::Ukraine_city_Kyiv,
                CountrySubdivision::Ukraine_city_Sevastopol,
                CountrySubdivision::Ukraine_region_Cherkaska_oblast,
                CountrySubdivision::Ukraine_region_Chernihivska_oblast,
                CountrySubdivision::Ukraine_region_Chernivetska_oblast,
                CountrySubdivision::Ukraine_region_Dnipropetrovska_oblast,
                CountrySubdivision::Ukraine_region_Donetska_oblast,
                CountrySubdivision::Ukraine_region_Ivano_Frankivska_oblast,
                CountrySubdivision::Ukraine_region_Kharkivska_oblast,
                CountrySubdivision::Ukraine_region_Khersonska_oblast,
                CountrySubdivision::Ukraine_region_Khmelnytska_oblast,
                CountrySubdivision::Ukraine_region_Kirovohradska_oblast,
                CountrySubdivision::Ukraine_region_Kyivska_oblast,
                CountrySubdivision::Ukraine_region_Luhanska_oblast,
                CountrySubdivision::Ukraine_region_Lvivska_oblast,
                CountrySubdivision::Ukraine_region_Mykolaivska_oblast,
                CountrySubdivision::Ukraine_region_Odeska_oblast,
                CountrySubdivision::Ukraine_region_Poltavska_oblast,
                CountrySubdivision::Ukraine_region_Rivnenska_oblast,
                CountrySubdivision::Ukraine_region_Sumska_oblast,
                CountrySubdivision::Ukraine_region_Ternopilska_oblast,
                CountrySubdivision::Ukraine_region_Vinnytska_oblast,
                CountrySubdivision::Ukraine_region_Volynska_oblast,
                CountrySubdivision::Ukraine_region_Zakarpatska_oblast,
                CountrySubdivision::Ukraine_region_Zaporizka_oblast,
                CountrySubdivision::Ukraine_region_Zhytomyrska_oblast,
                CountrySubdivision::Ukraine_republic_Avtonomna_Respublika_Krym
            ],
            self::United_Arab_Emirates => [
                CountrySubdivision::United_Arab_Emirates_emirate_Abu_Zaby,
                CountrySubdivision::United_Arab_Emirates_emirate_Ajman,
                CountrySubdivision::United_Arab_Emirates_emirate_Al_Fujayrah,
                CountrySubdivision::United_Arab_Emirates_emirate_Ash_Shariqah,
                CountrySubdivision::United_Arab_Emirates_emirate_Dubayy,
                CountrySubdivision::United_Arab_Emirates_emirate_Ra_s_al_Khaymah,
                CountrySubdivision::United_Arab_Emirates_emirate_Umm_al_Qaywayn
            ],
            self::United_Kingdom => [
                CountrySubdivision::United_Kingdom_London_borough_Barking_and_Dagenham,
                CountrySubdivision::United_Kingdom_London_borough_Barnet,
                CountrySubdivision::United_Kingdom_London_borough_Bexley,
                CountrySubdivision::United_Kingdom_London_borough_Brent,
                CountrySubdivision::United_Kingdom_London_borough_Bromley,
                CountrySubdivision::United_Kingdom_London_borough_Camden,
                CountrySubdivision::United_Kingdom_London_borough_Croydon,
                CountrySubdivision::United_Kingdom_London_borough_Ealing,
                CountrySubdivision::United_Kingdom_London_borough_Enfield,
                CountrySubdivision::United_Kingdom_London_borough_Greenwich,
                CountrySubdivision::United_Kingdom_London_borough_Hackney,
                CountrySubdivision::United_Kingdom_London_borough_Hammersmith_and_Fulham,
                CountrySubdivision::United_Kingdom_London_borough_Haringey,
                CountrySubdivision::United_Kingdom_London_borough_Harrow,
                CountrySubdivision::United_Kingdom_London_borough_Havering,
                CountrySubdivision::United_Kingdom_London_borough_Hillingdon,
                CountrySubdivision::United_Kingdom_London_borough_Hounslow,
                CountrySubdivision::United_Kingdom_London_borough_Islington,
                CountrySubdivision::United_Kingdom_London_borough_Kensington_and_Chelsea,
                CountrySubdivision::United_Kingdom_London_borough_Kingston_upon_Thames,
                CountrySubdivision::United_Kingdom_London_borough_Lambeth,
                CountrySubdivision::United_Kingdom_London_borough_Lewisham,
                CountrySubdivision::United_Kingdom_London_borough_Merton,
                CountrySubdivision::United_Kingdom_London_borough_Newham,
                CountrySubdivision::United_Kingdom_London_borough_Redbridge,
                CountrySubdivision::United_Kingdom_London_borough_Richmond_upon_Thames,
                CountrySubdivision::United_Kingdom_London_borough_Southwark,
                CountrySubdivision::United_Kingdom_London_borough_Sutton,
                CountrySubdivision::United_Kingdom_London_borough_Tower_Hamlets,
                CountrySubdivision::United_Kingdom_London_borough_Waltham_Forest,
                CountrySubdivision::United_Kingdom_London_borough_Wandsworth,
                CountrySubdivision::United_Kingdom_London_borough_Westminster,
                CountrySubdivision::United_Kingdom_city_corporation_London_City_of,
                CountrySubdivision::United_Kingdom_council_area_Aberdeen_City,
                CountrySubdivision::United_Kingdom_council_area_Aberdeenshire,
                CountrySubdivision::United_Kingdom_council_area_Angus,
                CountrySubdivision::United_Kingdom_council_area_Argyll_and_Bute,
                CountrySubdivision::United_Kingdom_council_area_Clackmannanshire,
                CountrySubdivision::United_Kingdom_council_area_Dumfries_and_Galloway,
                CountrySubdivision::United_Kingdom_council_area_Dundee_City,
                CountrySubdivision::United_Kingdom_council_area_East_Ayrshire,
                CountrySubdivision::United_Kingdom_council_area_East_Dunbartonshire,
                CountrySubdivision::United_Kingdom_council_area_East_Lothian,
                CountrySubdivision::United_Kingdom_council_area_East_Renfrewshire,
                CountrySubdivision::United_Kingdom_council_area_Edinburgh_City_of,
                CountrySubdivision::United_Kingdom_council_area_Eilean_Siar,
                CountrySubdivision::United_Kingdom_council_area_Falkirk,
                CountrySubdivision::United_Kingdom_council_area_Fife,
                CountrySubdivision::United_Kingdom_council_area_Glasgow_City,
                CountrySubdivision::United_Kingdom_council_area_Highland,
                CountrySubdivision::United_Kingdom_council_area_Inverclyde,
                CountrySubdivision::United_Kingdom_council_area_Midlothian,
                CountrySubdivision::United_Kingdom_council_area_Moray,
                CountrySubdivision::United_Kingdom_council_area_North_Ayrshire,
                CountrySubdivision::United_Kingdom_council_area_North_Lanarkshire,
                CountrySubdivision::United_Kingdom_council_area_Orkney_Islands,
                CountrySubdivision::United_Kingdom_council_area_Perth_and_Kinross,
                CountrySubdivision::United_Kingdom_council_area_Renfrewshire,
                CountrySubdivision::United_Kingdom_council_area_Scottish_Borders,
                CountrySubdivision::United_Kingdom_council_area_Shetland_Islands,
                CountrySubdivision::United_Kingdom_council_area_South_Ayrshire,
                CountrySubdivision::United_Kingdom_council_area_South_Lanarkshire,
                CountrySubdivision::United_Kingdom_council_area_Stirling,
                CountrySubdivision::United_Kingdom_council_area_West_Dunbartonshire,
                CountrySubdivision::United_Kingdom_council_area_West_Lothian,
                CountrySubdivision::United_Kingdom_country_England,
                CountrySubdivision::United_Kingdom_country_Scotland,
                CountrySubdivision::United_Kingdom_country_Wales_Cymru_GB_CYM,
                CountrySubdivision::United_Kingdom_district_Antrim_and_Newtownabbey,
                CountrySubdivision::United_Kingdom_district_Ards_and_North_Down,
                CountrySubdivision::United_Kingdom_district_Armagh_City_Banbridge_and_Craigavon,
                CountrySubdivision::United_Kingdom_district_Belfast_City,
                CountrySubdivision::United_Kingdom_district_Causeway_Coast_and_Glens,
                CountrySubdivision::United_Kingdom_district_Derry_and_Strabane,
                CountrySubdivision::United_Kingdom_district_Fermanagh_and_Omagh,
                CountrySubdivision::United_Kingdom_district_Lisburn_and_Castlereagh,
                CountrySubdivision::United_Kingdom_district_Mid_Ulster,
                CountrySubdivision::United_Kingdom_district_Mid_and_East_Antrim,
                CountrySubdivision::United_Kingdom_district_Newry_Mourne_and_Down,
                CountrySubdivision::United_Kingdom_metropolitan_district_Barnsley,
                CountrySubdivision::United_Kingdom_metropolitan_district_Birmingham,
                CountrySubdivision::United_Kingdom_metropolitan_district_Bolton,
                CountrySubdivision::United_Kingdom_metropolitan_district_Bradford,
                CountrySubdivision::United_Kingdom_metropolitan_district_Bury,
                CountrySubdivision::United_Kingdom_metropolitan_district_Calderdale,
                CountrySubdivision::United_Kingdom_metropolitan_district_Coventry,
                CountrySubdivision::United_Kingdom_metropolitan_district_Doncaster,
                CountrySubdivision::United_Kingdom_metropolitan_district_Dudley,
                CountrySubdivision::United_Kingdom_metropolitan_district_Gateshead,
                CountrySubdivision::United_Kingdom_metropolitan_district_Kirklees,
                CountrySubdivision::United_Kingdom_metropolitan_district_Knowsley,
                CountrySubdivision::United_Kingdom_metropolitan_district_Leeds,
                CountrySubdivision::United_Kingdom_metropolitan_district_Liverpool,
                CountrySubdivision::United_Kingdom_metropolitan_district_Manchester,
                CountrySubdivision::United_Kingdom_metropolitan_district_Newcastle_upon_Tyne,
                CountrySubdivision::United_Kingdom_metropolitan_district_North_Tyneside,
                CountrySubdivision::United_Kingdom_metropolitan_district_Oldham,
                CountrySubdivision::United_Kingdom_metropolitan_district_Rochdale,
                CountrySubdivision::United_Kingdom_metropolitan_district_Rotherham,
                CountrySubdivision::United_Kingdom_metropolitan_district_Salford,
                CountrySubdivision::United_Kingdom_metropolitan_district_Sandwell,
                CountrySubdivision::United_Kingdom_metropolitan_district_Sefton,
                CountrySubdivision::United_Kingdom_metropolitan_district_Sheffield,
                CountrySubdivision::United_Kingdom_metropolitan_district_Solihull,
                CountrySubdivision::United_Kingdom_metropolitan_district_South_Tyneside,
                CountrySubdivision::United_Kingdom_metropolitan_district_St_Helens,
                CountrySubdivision::United_Kingdom_metropolitan_district_Stockport,
                CountrySubdivision::United_Kingdom_metropolitan_district_Sunderland,
                CountrySubdivision::United_Kingdom_metropolitan_district_Tameside,
                CountrySubdivision::United_Kingdom_metropolitan_district_Trafford,
                CountrySubdivision::United_Kingdom_metropolitan_district_Wakefield,
                CountrySubdivision::United_Kingdom_metropolitan_district_Walsall,
                CountrySubdivision::United_Kingdom_metropolitan_district_Wigan,
                CountrySubdivision::United_Kingdom_metropolitan_district_Wirral,
                CountrySubdivision::United_Kingdom_metropolitan_district_Wolverhampton,
                CountrySubdivision::United_Kingdom_province_Northern_Ireland,
                CountrySubdivision::United_Kingdom_two_tier_county_Cambridgeshire,
                CountrySubdivision::United_Kingdom_two_tier_county_Cumbria,
                CountrySubdivision::United_Kingdom_two_tier_county_Derbyshire,
                CountrySubdivision::United_Kingdom_two_tier_county_Devon,
                CountrySubdivision::United_Kingdom_two_tier_county_Dorset,
                CountrySubdivision::United_Kingdom_two_tier_county_East_Sussex,
                CountrySubdivision::United_Kingdom_two_tier_county_Essex,
                CountrySubdivision::United_Kingdom_two_tier_county_Gloucestershire,
                CountrySubdivision::United_Kingdom_two_tier_county_Hampshire,
                CountrySubdivision::United_Kingdom_two_tier_county_Hertfordshire,
                CountrySubdivision::United_Kingdom_two_tier_county_Kent,
                CountrySubdivision::United_Kingdom_two_tier_county_Lancashire,
                CountrySubdivision::United_Kingdom_two_tier_county_Leicestershire,
                CountrySubdivision::United_Kingdom_two_tier_county_Lincolnshire,
                CountrySubdivision::United_Kingdom_two_tier_county_Norfolk,
                CountrySubdivision::United_Kingdom_two_tier_county_North_Yorkshire,
                CountrySubdivision::United_Kingdom_two_tier_county_Nottinghamshire,
                CountrySubdivision::United_Kingdom_two_tier_county_Oxfordshire,
                CountrySubdivision::United_Kingdom_two_tier_county_Somerset,
                CountrySubdivision::United_Kingdom_two_tier_county_Staffordshire,
                CountrySubdivision::United_Kingdom_two_tier_county_Suffolk,
                CountrySubdivision::United_Kingdom_two_tier_county_Surrey,
                CountrySubdivision::United_Kingdom_two_tier_county_Warwickshire,
                CountrySubdivision::United_Kingdom_two_tier_county_West_Sussex,
                CountrySubdivision::United_Kingdom_two_tier_county_Worcestershire,
                CountrySubdivision::United_Kingdom_unitary_authority_Bath_and_North_East_Somerset,
                CountrySubdivision::United_Kingdom_unitary_authority_Bedford,
                CountrySubdivision::United_Kingdom_unitary_authority_Blackburn_with_Darwen,
                CountrySubdivision::United_Kingdom_unitary_authority_Blackpool,
                CountrySubdivision::United_Kingdom_unitary_authority_Blaenau_Gwent,
                CountrySubdivision::United_Kingdom_unitary_authority_Bournemouth_Christchurch_and_Poole,
                CountrySubdivision::United_Kingdom_unitary_authority_Bracknell_Forest,
                CountrySubdivision::United_Kingdom_unitary_authority_Bridgend_Pen_y_bont_ar_Ogwr_GB_POG,
                CountrySubdivision::United_Kingdom_unitary_authority_Brighton_and_Hove,
                CountrySubdivision::United_Kingdom_unitary_authority_Bristol_City_of,
                CountrySubdivision::United_Kingdom_unitary_authority_Buckinghamshire,
                CountrySubdivision::United_Kingdom_unitary_authority_Caerphilly_Caerffili_GB_CAF,
                CountrySubdivision::United_Kingdom_unitary_authority_Cardiff_Caerdydd_GB_CRD,
                CountrySubdivision::United_Kingdom_unitary_authority_Carmarthenshire_Sir_Gaerfyrddin_GB_GFY,
                CountrySubdivision::United_Kingdom_unitary_authority_Central_Bedfordshire,
                CountrySubdivision::United_Kingdom_unitary_authority_Ceredigion_Sir_Ceredigion,
                CountrySubdivision::United_Kingdom_unitary_authority_Cheshire_East,
                CountrySubdivision::United_Kingdom_unitary_authority_Cheshire_West_and_Chester,
                CountrySubdivision::United_Kingdom_unitary_authority_Conwy,
                CountrySubdivision::United_Kingdom_unitary_authority_Cornwall,
                CountrySubdivision::United_Kingdom_unitary_authority_Darlington,
                CountrySubdivision::United_Kingdom_unitary_authority_Denbighshire_Sir_Ddinbych_GB_DDB,
                CountrySubdivision::United_Kingdom_unitary_authority_Derby,
                CountrySubdivision::United_Kingdom_unitary_authority_Durham_County,
                CountrySubdivision::United_Kingdom_unitary_authority_East_Riding_of_Yorkshire,
                CountrySubdivision::United_Kingdom_unitary_authority_Flintshire_Sir_y_Fflint_GB_FFL,
                CountrySubdivision::United_Kingdom_unitary_authority_Gwynedd,
                CountrySubdivision::United_Kingdom_unitary_authority_Halton,
                CountrySubdivision::United_Kingdom_unitary_authority_Hartlepool,
                CountrySubdivision::United_Kingdom_unitary_authority_Herefordshire,
                CountrySubdivision::United_Kingdom_unitary_authority_Isle_of_Anglesey_Sir_Ynys_Mon_GB_YNM,
                CountrySubdivision::United_Kingdom_unitary_authority_Isle_of_Wight,
                CountrySubdivision::United_Kingdom_unitary_authority_Isles_of_Scilly,
                CountrySubdivision::United_Kingdom_unitary_authority_Kingston_upon_Hull,
                CountrySubdivision::United_Kingdom_unitary_authority_Leicester,
                CountrySubdivision::United_Kingdom_unitary_authority_Luton,
                CountrySubdivision::United_Kingdom_unitary_authority_Medway,
                CountrySubdivision::United_Kingdom_unitary_authority_Merthyr_Tydfil_Merthyr_Tudful_GB_MTU,
                CountrySubdivision::United_Kingdom_unitary_authority_Middlesbrough,
                CountrySubdivision::United_Kingdom_unitary_authority_Milton_Keynes,
                CountrySubdivision::United_Kingdom_unitary_authority_Monmouthshire_Sir_Fynwy_GB_FYN,
                CountrySubdivision::United_Kingdom_unitary_authority_Neath_Port_Talbot_Castell_nedd_Port_Talbot_GB_CTL,
                CountrySubdivision::United_Kingdom_unitary_authority_Newport_Casnewydd_GB_CNW,
                CountrySubdivision::United_Kingdom_unitary_authority_North_East_Lincolnshire,
                CountrySubdivision::United_Kingdom_unitary_authority_North_Lincolnshire,
                CountrySubdivision::United_Kingdom_unitary_authority_North_Northamptonshire,
                CountrySubdivision::United_Kingdom_unitary_authority_North_Somerset,
                CountrySubdivision::United_Kingdom_unitary_authority_Northumberland,
                CountrySubdivision::United_Kingdom_unitary_authority_Nottingham,
                CountrySubdivision::United_Kingdom_unitary_authority_Pembrokeshire_Sir_Benfro_GB_BNF,
                CountrySubdivision::United_Kingdom_unitary_authority_Peterborough,
                CountrySubdivision::United_Kingdom_unitary_authority_Plymouth,
                CountrySubdivision::United_Kingdom_unitary_authority_Portsmouth,
                CountrySubdivision::United_Kingdom_unitary_authority_Powys,
                CountrySubdivision::United_Kingdom_unitary_authority_Reading,
                CountrySubdivision::United_Kingdom_unitary_authority_Redcar_and_Cleveland,
                CountrySubdivision::United_Kingdom_unitary_authority_Rhondda_Cynon_Taff_Rhondda_CynonTaf,
                CountrySubdivision::United_Kingdom_unitary_authority_Rutland,
                CountrySubdivision::United_Kingdom_unitary_authority_Shropshire,
                CountrySubdivision::United_Kingdom_unitary_authority_Slough,
                CountrySubdivision::United_Kingdom_unitary_authority_South_Gloucestershire,
                CountrySubdivision::United_Kingdom_unitary_authority_Southampton,
                CountrySubdivision::United_Kingdom_unitary_authority_Southend_on_Sea,
                CountrySubdivision::United_Kingdom_unitary_authority_Stockton_on_Tees,
                CountrySubdivision::United_Kingdom_unitary_authority_Stoke_on_Trent,
                CountrySubdivision::United_Kingdom_unitary_authority_Swansea_Abertawe_GB_ATA,
                CountrySubdivision::United_Kingdom_unitary_authority_Swindon,
                CountrySubdivision::United_Kingdom_unitary_authority_Telford_and_Wrekin,
                CountrySubdivision::United_Kingdom_unitary_authority_Thurrock,
                CountrySubdivision::United_Kingdom_unitary_authority_Torbay,
                CountrySubdivision::United_Kingdom_unitary_authority_Torfaen_Tor_faen,
                CountrySubdivision::United_Kingdom_unitary_authority_Vale_of_Glamorgan_The_Bro_Morgannwg_GB_BMG,
                CountrySubdivision::United_Kingdom_unitary_authority_Warrington,
                CountrySubdivision::United_Kingdom_unitary_authority_West_Berkshire,
                CountrySubdivision::United_Kingdom_unitary_authority_West_Northamptonshire,
                CountrySubdivision::United_Kingdom_unitary_authority_Wiltshire,
                CountrySubdivision::United_Kingdom_unitary_authority_Windsor_and_Maidenhead,
                CountrySubdivision::United_Kingdom_unitary_authority_Wokingham,
                CountrySubdivision::United_Kingdom_unitary_authority_Wrexham_Wrecsam_GB_WRC,
                CountrySubdivision::United_Kingdom_unitary_authority_York
            ],
            self::United_States_Outlying_Islands => [
                CountrySubdivision::U_S_Outlying_Islands_islands_groups_of_islands_Baker_Island,
                CountrySubdivision::U_S_Outlying_Islands_islands_groups_of_islands_Howland_Island,
                CountrySubdivision::U_S_Outlying_Islands_islands_groups_of_islands_Jarvis_Island,
                CountrySubdivision::U_S_Outlying_Islands_islands_groups_of_islands_Johnston_Atoll,
                CountrySubdivision::U_S_Outlying_Islands_islands_groups_of_islands_Kingman_Reef,
                CountrySubdivision::U_S_Outlying_Islands_islands_groups_of_islands_Midway_Islands,
                CountrySubdivision::U_S_Outlying_Islands_islands_groups_of_islands_Navassa_Island,
                CountrySubdivision::U_S_Outlying_Islands_islands_groups_of_islands_Palmyra_Atoll,
                CountrySubdivision::U_S_Outlying_Islands_islands_groups_of_islands_Wake_Island
            ],
            self::United_States_of_America => [
                CountrySubdivision::United_States_district_District_of_Columbia,
                CountrySubdivision::United_States_outlying_area_American_Samoa,
                CountrySubdivision::United_States_outlying_area_Guam,
                CountrySubdivision::United_States_outlying_area_Northern_Mariana_Islands,
                CountrySubdivision::United_States_outlying_area_Puerto_Rico,
                CountrySubdivision::United_States_outlying_area_United_States_Minor_Outlying_Islands,
                CountrySubdivision::United_States_outlying_area_Virgin_Islands_U_S,
                CountrySubdivision::United_States_state_Alabama,
                CountrySubdivision::United_States_state_Alaska,
                CountrySubdivision::United_States_state_Arizona,
                CountrySubdivision::United_States_state_Arkansas,
                CountrySubdivision::United_States_state_California,
                CountrySubdivision::United_States_state_Colorado,
                CountrySubdivision::United_States_state_Connecticut,
                CountrySubdivision::United_States_state_Delaware,
                CountrySubdivision::United_States_state_Florida,
                CountrySubdivision::United_States_state_Georgia,
                CountrySubdivision::United_States_state_Hawaii,
                CountrySubdivision::United_States_state_Idaho,
                CountrySubdivision::United_States_state_Illinois,
                CountrySubdivision::United_States_state_Indiana,
                CountrySubdivision::United_States_state_Iowa,
                CountrySubdivision::United_States_state_Kansas,
                CountrySubdivision::United_States_state_Kentucky,
                CountrySubdivision::United_States_state_Louisiana,
                CountrySubdivision::United_States_state_Maine,
                CountrySubdivision::United_States_state_Maryland,
                CountrySubdivision::United_States_state_Massachusetts,
                CountrySubdivision::United_States_state_Michigan,
                CountrySubdivision::United_States_state_Minnesota,
                CountrySubdivision::United_States_state_Mississippi,
                CountrySubdivision::United_States_state_Missouri,
                CountrySubdivision::United_States_state_Montana,
                CountrySubdivision::United_States_state_Nebraska,
                CountrySubdivision::United_States_state_Nevada,
                CountrySubdivision::United_States_state_New_Hampshire,
                CountrySubdivision::United_States_state_New_Jersey,
                CountrySubdivision::United_States_state_New_Mexico,
                CountrySubdivision::United_States_state_New_York,
                CountrySubdivision::United_States_state_North_Carolina,
                CountrySubdivision::United_States_state_North_Dakota,
                CountrySubdivision::United_States_state_Ohio,
                CountrySubdivision::United_States_state_Oklahoma,
                CountrySubdivision::United_States_state_Oregon,
                CountrySubdivision::United_States_state_Pennsylvania,
                CountrySubdivision::United_States_state_Rhode_Island,
                CountrySubdivision::United_States_state_South_Carolina,
                CountrySubdivision::United_States_state_South_Dakota,
                CountrySubdivision::United_States_state_Tennessee,
                CountrySubdivision::United_States_state_Texas,
                CountrySubdivision::United_States_state_Utah,
                CountrySubdivision::United_States_state_Vermont,
                CountrySubdivision::United_States_state_Virginia,
                CountrySubdivision::United_States_state_Washington,
                CountrySubdivision::United_States_state_West_Virginia,
                CountrySubdivision::United_States_state_Wisconsin,
                CountrySubdivision::United_States_state_Wyoming
            ],
            self::Uruguay => [
                CountrySubdivision::Uruguay_department_Artigas,
                CountrySubdivision::Uruguay_department_Canelones,
                CountrySubdivision::Uruguay_department_Cerro_Largo,
                CountrySubdivision::Uruguay_department_Colonia,
                CountrySubdivision::Uruguay_department_Durazno,
                CountrySubdivision::Uruguay_department_Flores,
                CountrySubdivision::Uruguay_department_Florida,
                CountrySubdivision::Uruguay_department_Lavalleja,
                CountrySubdivision::Uruguay_department_Maldonado,
                CountrySubdivision::Uruguay_department_Montevideo,
                CountrySubdivision::Uruguay_department_Paysandu,
                CountrySubdivision::Uruguay_department_Rio_Negro,
                CountrySubdivision::Uruguay_department_Rivera,
                CountrySubdivision::Uruguay_department_Rocha,
                CountrySubdivision::Uruguay_department_Salto,
                CountrySubdivision::Uruguay_department_San_Jose,
                CountrySubdivision::Uruguay_department_Soriano,
                CountrySubdivision::Uruguay_department_Tacuarembo,
                CountrySubdivision::Uruguay_department_Treinta_y_Tres
            ],
            self::Uzbekistan => [
                CountrySubdivision::Uzbekistan_city_Toshkent,
                CountrySubdivision::Uzbekistan_region_Andijon,
                CountrySubdivision::Uzbekistan_region_Buxoro,
                CountrySubdivision::Uzbekistan_region_Farg_ona,
                CountrySubdivision::Uzbekistan_region_Jizzax,
                CountrySubdivision::Uzbekistan_region_Namangan,
                CountrySubdivision::Uzbekistan_region_Navoiy,
                CountrySubdivision::Uzbekistan_region_Qashqadaryo,
                CountrySubdivision::Uzbekistan_region_Samarqand,
                CountrySubdivision::Uzbekistan_region_Sirdaryo,
                CountrySubdivision::Uzbekistan_region_Surxondaryo,
                CountrySubdivision::Uzbekistan_region_Toshkent,
                CountrySubdivision::Uzbekistan_region_Xorazm,
                CountrySubdivision::Uzbekistan_republic_Qoraqalpog_iston_Respublikasi
            ],
            self::Vanuatu => [
                CountrySubdivision::Vanuatu_province_Malampa,
                CountrySubdivision::Vanuatu_province_Penama,
                CountrySubdivision::Vanuatu_province_Sanma,
                CountrySubdivision::Vanuatu_province_Shefa,
                CountrySubdivision::Vanuatu_province_Tafea,
                CountrySubdivision::Vanuatu_province_Torba
            ],
            self::Venezuela => [
                CountrySubdivision::Venezuela_capital_district_Distrito_Capital,
                CountrySubdivision::Venezuela_federal_dependency_Dependencias_Federales,
                CountrySubdivision::Venezuela_state_Amazonas,
                CountrySubdivision::Venezuela_state_Anzoategui,
                CountrySubdivision::Venezuela_state_Apure,
                CountrySubdivision::Venezuela_state_Aragua,
                CountrySubdivision::Venezuela_state_Barinas,
                CountrySubdivision::Venezuela_state_Bolivar,
                CountrySubdivision::Venezuela_state_Carabobo,
                CountrySubdivision::Venezuela_state_Cojedes,
                CountrySubdivision::Venezuela_state_Delta_Amacuro,
                CountrySubdivision::Venezuela_state_Falcon,
                CountrySubdivision::Venezuela_state_Guarico,
                CountrySubdivision::Venezuela_state_La_Guaira,
                CountrySubdivision::Venezuela_state_Lara,
                CountrySubdivision::Venezuela_state_Merida,
                CountrySubdivision::Venezuela_state_Miranda,
                CountrySubdivision::Venezuela_state_Monagas,
                CountrySubdivision::Venezuela_state_Nueva_Esparta,
                CountrySubdivision::Venezuela_state_Portuguesa,
                CountrySubdivision::Venezuela_state_Sucre,
                CountrySubdivision::Venezuela_state_Tachira,
                CountrySubdivision::Venezuela_state_Trujillo,
                CountrySubdivision::Venezuela_state_Yaracuy,
                CountrySubdivision::Venezuela_state_Zulia
            ],
            self::Viet_Nam => [
                CountrySubdivision::Vietnam_municipality_Can_Tho,
                CountrySubdivision::Vietnam_municipality_Da_Nang,
                CountrySubdivision::Vietnam_municipality_Ha_Noi,
                CountrySubdivision::Vietnam_municipality_Hai_Phong,
                CountrySubdivision::Vietnam_municipality_Ho_Chi_Minh,
                CountrySubdivision::Vietnam_province_An_Giang,
                CountrySubdivision::Vietnam_province_Ba_Ria_Vung_Tau,
                CountrySubdivision::Vietnam_province_Bac_Giang,
                CountrySubdivision::Vietnam_province_Bac_Kan,
                CountrySubdivision::Vietnam_province_Bac_Lieu,
                CountrySubdivision::Vietnam_province_Bac_Ninh,
                CountrySubdivision::Vietnam_province_Ben_Tre,
                CountrySubdivision::Vietnam_province_Binh_Dinh,
                CountrySubdivision::Vietnam_province_Binh_Duong,
                CountrySubdivision::Vietnam_province_Binh_Phuoc,
                CountrySubdivision::Vietnam_province_Binh_Thuan,
                CountrySubdivision::Vietnam_province_Ca_Mau,
                CountrySubdivision::Vietnam_province_Cao_Bang,
                CountrySubdivision::Vietnam_province_Dak_Lak,
                CountrySubdivision::Vietnam_province_Dak_Nong,
                CountrySubdivision::Vietnam_province_Dien_Bien,
                CountrySubdivision::Vietnam_province_Dong_Nai,
                CountrySubdivision::Vietnam_province_Dong_Thap,
                CountrySubdivision::Vietnam_province_Gia_Lai,
                CountrySubdivision::Vietnam_province_Ha_Giang,
                CountrySubdivision::Vietnam_province_Ha_Nam,
                CountrySubdivision::Vietnam_province_Ha_Tinh,
                CountrySubdivision::Vietnam_province_Hai_Duong,
                CountrySubdivision::Vietnam_province_Hau_Giang,
                CountrySubdivision::Vietnam_province_Hoa_Binh,
                CountrySubdivision::Vietnam_province_Hung_Yen,
                CountrySubdivision::Vietnam_province_Khanh_Hoa,
                CountrySubdivision::Vietnam_province_Kien_Giang,
                CountrySubdivision::Vietnam_province_Kon_Tum,
                CountrySubdivision::Vietnam_province_Lai_Chau,
                CountrySubdivision::Vietnam_province_Lam_Dong,
                CountrySubdivision::Vietnam_province_Lang_Son,
                CountrySubdivision::Vietnam_province_Lao_Cai,
                CountrySubdivision::Vietnam_province_Long_An,
                CountrySubdivision::Vietnam_province_Nam_Dinh,
                CountrySubdivision::Vietnam_province_Nghe_An,
                CountrySubdivision::Vietnam_province_Ninh_Binh,
                CountrySubdivision::Vietnam_province_Ninh_Thuan,
                CountrySubdivision::Vietnam_province_Phu_Tho,
                CountrySubdivision::Vietnam_province_Phu_Yen,
                CountrySubdivision::Vietnam_province_Quang_Binh,
                CountrySubdivision::Vietnam_province_Quang_Nam,
                CountrySubdivision::Vietnam_province_Quang_Ngai,
                CountrySubdivision::Vietnam_province_Quang_Ninh,
                CountrySubdivision::Vietnam_province_Quang_Tri,
                CountrySubdivision::Vietnam_province_Soc_Trang,
                CountrySubdivision::Vietnam_province_Son_La,
                CountrySubdivision::Vietnam_province_Tay_Ninh,
                CountrySubdivision::Vietnam_province_Thai_Binh,
                CountrySubdivision::Vietnam_province_Thai_Nguyen,
                CountrySubdivision::Vietnam_province_Thanh_Hoa,
                CountrySubdivision::Vietnam_province_Thua_Thien_Hue,
                CountrySubdivision::Vietnam_province_Tien_Giang,
                CountrySubdivision::Vietnam_province_Tra_Vinh,
                CountrySubdivision::Vietnam_province_Tuyen_Quang,
                CountrySubdivision::Vietnam_province_Vinh_Long,
                CountrySubdivision::Vietnam_province_Vinh_Phuc,
                CountrySubdivision::Vietnam_province_Yen_Bai
            ],
            self::Wallis_and_Futuna => [
                CountrySubdivision::Wallis_Futuna_administrative_precinct_Alo,
                CountrySubdivision::Wallis_Futuna_administrative_precinct_Sigave,
                CountrySubdivision::Wallis_Futuna_administrative_precinct_Uvea
            ],
            self::Yemen => [
                CountrySubdivision::Yemen_governorate_Abyan,
                CountrySubdivision::Yemen_governorate_Ad_Dali,
                CountrySubdivision::Yemen_governorate_Adan,
                CountrySubdivision::Yemen_governorate_Al_Bayda,
                CountrySubdivision::Yemen_governorate_Al_Hudaydah,
                CountrySubdivision::Yemen_governorate_Al_Jawf,
                CountrySubdivision::Yemen_governorate_Al_Mahrah,
                CountrySubdivision::Yemen_governorate_Al_Mahwit,
                CountrySubdivision::Yemen_governorate_Amran,
                CountrySubdivision::Yemen_governorate_Arkhabil_Suqutra,
                CountrySubdivision::Yemen_governorate_Dhamar,
                CountrySubdivision::Yemen_governorate_Hadramawt,
                CountrySubdivision::Yemen_governorate_Hajjah,
                CountrySubdivision::Yemen_governorate_Ibb,
                CountrySubdivision::Yemen_governorate_Lahij,
                CountrySubdivision::Yemen_governorate_Ma_rib,
                CountrySubdivision::Yemen_governorate_Raymah,
                CountrySubdivision::Yemen_governorate_Sa_dah,
                CountrySubdivision::Yemen_governorate_San_a,
                CountrySubdivision::Yemen_governorate_Shabwah,
                CountrySubdivision::Yemen_governorate_Ta_izz,
                CountrySubdivision::Yemen_municipality_Amanat_al_Asimah_city
            ],
            self::Zambia => [
                CountrySubdivision::Zambia_province_Central,
                CountrySubdivision::Zambia_province_Copperbelt,
                CountrySubdivision::Zambia_province_Eastern,
                CountrySubdivision::Zambia_province_Luapula,
                CountrySubdivision::Zambia_province_Lusaka,
                CountrySubdivision::Zambia_province_Muchinga,
                CountrySubdivision::Zambia_province_North_Western,
                CountrySubdivision::Zambia_province_Northern,
                CountrySubdivision::Zambia_province_Southern,
                CountrySubdivision::Zambia_province_Western
            ],
            self::Zimbabwe => [
                CountrySubdivision::Zimbabwe_province_Bulawayo,
                CountrySubdivision::Zimbabwe_province_Harare,
                CountrySubdivision::Zimbabwe_province_Manicaland,
                CountrySubdivision::Zimbabwe_province_Mashonaland_Central,
                CountrySubdivision::Zimbabwe_province_Mashonaland_East,
                CountrySubdivision::Zimbabwe_province_Mashonaland_West,
                CountrySubdivision::Zimbabwe_province_Masvingo,
                CountrySubdivision::Zimbabwe_province_Matabeleland_North,
                CountrySubdivision::Zimbabwe_province_Matabeleland_South,
                CountrySubdivision::Zimbabwe_province_Midlands
            ],
            default => []
        };
    }

    /** @return list<self> */
    public function getSubCountries(): array {
        $subCountries = [];
        foreach ($this->getSubdivisions() as $subdivision) {
            if (($sameAsCountry = $subdivision->getSameAsCountry()) !== null && in_array($sameAsCountry, $subCountries, true) === false) {
                $subCountries[] = $sameAsCountry;
            }
        }

        return $subCountries;
    }

    public function getParentCountry(): ?self {
        foreach (self::cases() as $country) {
            if (in_array($this, $country->getSubCountries(), true)) {
                return $country;
            }
        }

        return null;
    }

    /**
     * If users need code elements to represent country names not included in ISO 3166-1, the series of letters AA, QM to QZ, XA to XZ, and ZZ (...) are available
     *
     * @see https://www.iso.org/glossary-for-iso-3166.html
     */
    public static function isInUserAssignedSpace(string $alpha2): bool {
        if (strlen($alpha2) !== 2) {
            return false;
        }

        if (in_array($alpha2, ['AA', 'ZZ'], true)) {
            return true;
        }

        $firstChar = substr($alpha2, 0, 1);
        $secondChar = substr($alpha2, 1, 1);
        if ($firstChar === 'Q'
            && ord($secondChar) >= ord('M') && ord($secondChar) <= ord('Z')) {
            return true;
        }

        if ($firstChar === 'X'
            && ord($secondChar) >= ord('A') && ord($secondChar) <= ord('Z')) {
            return true;
        }

        return false;
    }
}
